@extends('layouts/layoutMaster')

@section('title', __('locale.Ride Pricing Rules'))

<!-- Vendor Styles -->
@section('vendor-style')
  @vite(['resources/assets/css/datatables.scss', 'resources/assets/vendor/libs/flatpickr/flatpickr.scss'])
  <style>
    .flatpickr-calendar {
      z-index: 99999 !important;
    }

    .flatpickr-time input {
      pointer-events: auto !important;
    }

    .flatpickr-time .numInputWrapper span {
      pointer-events: auto !important;
    }
  </style>
@endsection

<!-- Vendor Scripts -->
@section('vendor-script')
  @vite(['resources/assets/js/datatables.js', 'resources/assets/vendor/libs/flatpickr/flatpickr.js'])
@endsection



@section('content')

  <!-- Form with Tabs -->
  <div class="row">
    <div class="col">
      <div class="card mb-6">
        <div class="card-body">
          <div class="card-header d-flex justify-content-between align-items-center pb-0 pt-1">
            <h5 class="mb-0">{{ __('locale.Ride Pricing Rules') }}</h5>
            <button class="btn btn-primary" data-bs-toggle="modal"
              data-bs-target="#ruleModal">{{ __('locale.Add Rule') }}</button>
          </div>
          <div class="card-body table-responsive">
            <table id="rulesTable" class="table">
              <thead>
                <tr>
                  <th>{{ __('locale.Name') }}</th>
                  <th>{{ __('locale.Country') }}</th>
                  <th>{{ __('locale.Price Type') }}</th>
                  <th>{{ __('locale.Amount') }}</th>
                  <th>{{ __('locale.Conditions') }}</th>
                  <th>{{ __('locale.Status') }}</th>
                  <th>{{ __('locale.Actions') }}</th>
                </tr>
              </thead>
              <tbody>
                @forelse($rules as $rule)
                  <tr>
                    <td>{{ $rule->rule_name }}</td>
                    <td>{{ $rule->country->name ?? '-' }} <span
                        class="badge px-2 bg-label-primary d-none">{{ $rule->priority }}</span></td>

                    <td>
                      <span class="badge bg-label-info text-capitalize">{{ $rule->price_type }}</span>
                    </td>
                    <td>{{ $rule->amount }}</td>

                    <td>
                      @if ($rule->conditions && $rule->conditions->count())
                        <ul class="list-unstyled mb-0">
                          @foreach ($rule->conditions as $condition)
                            <li>
                              <strong>{{ ucfirst(str_replace('_', ' ', $condition->condition_type)) }}:</strong>
                              @php
                                $values = json_decode($condition->condition_value, true) ?? [];
                                $valuesArray = is_string($values) ? json_decode($values, true) : $values;
                              @endphp

                              @if ($condition->condition_type === 'month')
                                {{ collect($valuesArray)->map(fn($m) => \Carbon\Carbon::create()->month((int) $m)->format('F'))->implode(', ') }}
                              @elseif($condition->condition_type === 'day_of_week')
                                {{ implode(', ', $valuesArray) }}
                              @elseif($condition->condition_type === 'time_range')
                                Start: {{ $valuesArray['start'] ?? '-' }}, End: {{ $valuesArray['end'] ?? '-' }}
                              @else
                                {{ implode(', ', $valuesArray) }}
                              @endif
                            </li>
                          @endforeach
                        </ul>
                      @else
                        <em>No conditions</em>
                      @endif

                    </td>
                    <td>
                      <span class="badge {{ $rule->is_active ? 'bg-label-success' : 'bg-label-secondary' }}">
                        {{ $rule->is_active ? 'Active' : 'Inactive' }}
                      </span>
                    </td>
                    <td>
                      <button class="btn btn-icon btn-text-secondary rounded-pill waves-effect editRule p-1"
                        data-id="{{ $rule->id }}" data-rule='@json($rule)'>
                        <i class="icon-base ti tabler-edit me-1"></i>
                      </button>
                      <button class="btn btn-icon btn-text-secondary rounded-pill waves-effect deleteRule p-1"
                        data-id="{{ $rule->id }}">
                        <i class="icon-base ti tabler-x me-1"></i>
                      </button>
                    </td>
                  </tr>
                @empty
                  <tr>
                    <td colspan="10" class="text-center">No pricing rules found.</td>
                  </tr>
                @endforelse
              </tbody>
            </table>


          </div>
        </div>
      </div>
    </div>
  </div>
  @include('_partials._modals.modal-ride-price-rule')
@endsection
<!-- Page Scripts -->
@section('page-script')
  @vite([])
  <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
  <script>
    $('[data-bs-target="#ruleModal"]').on('click', function() {
      let modal = $('#ruleModal');
      let form = modal.find('form')[0];
      if (form) {
        form.reset();
      }

      // Clear select2 / flatpickr
      modal.find('select').val(null).trigger('change');
      modal.find('.flatpickr-input').val('');
    });
    $(document).ready(function() {
      $('#rulesTable').DataTable({
        responsive: true,
        processing: true,
        columnDefs: [{
            orderable: false,
            targets: [6]
          }, // Actions column not sortable
        ],
        pageLength: 10, // Default rows per page
        lengthMenu: [5, 10, 25, 50, 100],
        language: {
          search: "_INPUT_",
          searchPlaceholder: "Search rules...",
        },
        dom: "<'row mb-3'<'col-sm-12 col-md-6'l><'col-sm-12 col-md-6'f>>" +
          "<'table-responsive'tr>" +
          "<'row mt-3'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7'p>>",
      });
    });

    $(function() {
      let $btn = null;

      $(document).on('click', '#ruleForm button[type="submit"]', function() {
        $btn = $(this);
      });
      let form = $('#ruleForm');

      // Create/Edit Rule
      form.on('submit', function(e) {
        e.preventDefault();
        const $form = $(this);
        const id = $('#ruleId').val();
        const url = id ? `${baseURL}/ride_pricing_rules/${id}` : `${baseURL}/ride_pricing_rules`;
        let method = 'POST';

        // Make sure each time_range row's hidden value is synced (we already handle on change, but ensure here)
        $('#conditionsWrapper .condition-row').each(function() {
          const $r = $(this);
          const idx = $r.data('idx');
          const type = $r.find(`select[name="conditions[${idx}][type]"]`).val();

          if (type === 'time_range') {
            const s = $r.find('.condition-value-start').val() || '';
            const e = $r.find('.condition-value-end').val() || '';
            $r.find(`input[name="conditions[${idx}][value]"]`).val(JSON.stringify({
              start: s,
              end: e
            }));
          }
          // For select multiple, values are handled by the browser (conditions[idx][value][])
        });

        // serialize and submit
        let data = $form.serialize();
        if (id) data += '&_method=PATCH';
        $.ajax({
          url: url,
          method: method,
          data: data,
          beforeSend: function() {
            toggleButtonLoading($btn, true, 'Saving...');
          },
          success: function(data) {
            console.log(data);
            toggleButtonLoading($btn, false);
            $('#ruleModal').modal('hide');
            const toastType = data.status == 200 ? 'bg-success' : 'bg-warning';
            window.showToast({
              title: 'Ride Pricing Rules 🗂️',
              message: data.message,
              type: toastType,
              animation: 'animate__fadeInUp',
              time: 'Just now',
              delay: 3000
            });
            if (data.status == 200) {
              setTimeout(() => {
                window.location.reload();
              }, 1500);
            }
          },
          error: function(xhr) {
            console.log(xhr.responseJSON);
            toggleButtonLoading($btn, false);
          }
        });
      });

      // Delete button
      $('.deleteRule').on('click', function() {
        let id = $(this).data('id');
        Swal.fire({
          title: 'Are you sure?',
          text: "This will permanently delete the record!",
          icon: 'warning',
          showCancelButton: true,
          confirmButtonColor: '#d33',
          cancelButtonColor: '#3085d6',
          confirmButtonText: 'Yes, delete it!'
        }).then((result) => {
          if (result.isConfirmed) {
            $.ajax({
              url: `${baseURL}/ride_pricing_rules/${id}`,
              method: 'DELETE',
              success: function(data) {
                const toastType = data.status == 200 ? 'bg-success' : 'bg-warning';
                window.showToast({
                  title: 'Ride Pricing Rules 🗂️',
                  message: data.message,
                  type: toastType,
                  animation: 'animate__fadeInUp',
                  time: 'Just now',
                  delay: 3000
                });
                if (data.status == 200) {
                  setTimeout(() => {
                    window.location.reload();
                  }, 2000);
                }
              }
            });
          }
        });
      });
    });

    $(function() {
      // Render condition row with correct input field
      let condIndex = 0; // increment for each new condition row

      function nextCondIndex() {
        return condIndex++;
      }

      function renderConditionRow(idx, type = '', value = '') {
        let $row = $(`
      <div class="condition-row border p-2 rounded mb-2" data-idx="${idx}">
        <div class="d-flex justify-content-between align-items-center">
          <select class="form-select condition-type" name="conditions[${idx}][type]" required>
            <option value="">-- Select Condition Type --</option>
            <option value="month" ${type === 'month' ? 'selected' : ''}>Month</option>
            <option value="day_of_week" ${type === 'day_of_week' ? 'selected' : ''}>Day of Week</option>
            <option value="date" ${type === 'date' ? 'selected' : ''}>Specific Date</option>
            <option value="time_range" ${type === 'time_range' ? 'selected' : ''}>Time Range</option>
          </select>
          <button type="button" class="btn btn-sm btn-danger removeCondition ms-2">X</button>
        </div>
        <div class="condition-value mt-2"></div>
      </div>
    `);

        // Render value field
        renderConditionValue($row, idx, type, value);

        return $row;
      }

      // Render value field based on type
      function renderConditionValue($row, idx, type, value = '') {

        // Try to parse JSON if stringified
        try {
          if (typeof value === 'string' && value.startsWith('[')) {
            value = JSON.parse(value);
          }
        } catch (e) {
          // ignore if not valid JSON
        }

        let html = '';
        switch (type) {
          case 'month':
            const monthNames = [
              'January', 'February', 'March', 'April', 'May', 'June',
              'July', 'August', 'September', 'October', 'November', 'December'
            ];
            html = `<select class="select2-multiple form-select" multiple name="conditions[${idx}][value][]">
        ${monthNames.map((name, idx) => {
          let val = idx + 1;
          return `<option value="${val}" ${Array.isArray(value) && value.includes(String(val)) ? 'selected' : ''}>${name}</option>`;
        }).join('')}
      </select>`;
            break;

          case 'day_of_week':
            const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
            html = `<select class="select2-multiple form-select" multiple name="conditions[${idx}][value][]">
        ${days.map(d => `<option value="${d}" ${Array.isArray(value) && value.includes(d) ? 'selected' : ''}>${d}</option>`).join('')}
      </select>`;
            break;

          case 'date':
            html =
              `<input type="text" class="form-control flatpickr-date" name="conditions[${idx}][value]" value="${value || ''}" placeholder="YYYY-MM-DD">`;
            break;

          case 'time_range':
            // value should be {start:'HH:MM', end:'HH:MM'}
            const start = value?.start || '';
            const end = value?.end || '';
            html = `
    <div class="d-flex gap-2">
        <input type="text" class="form-control flatpickr-time condition-value-start"
               placeholder="Start Time" value="${start}">
        <input type="text" class="form-control flatpickr-time condition-value-end"
               placeholder="End Time" value="${end}">
        <input type="hidden" name="conditions[${idx}][value]" class="condition-value-hidden" value='${JSON.stringify({start, end})}'>
    </div>`;

            $row.find('.condition-value').html(html);

            // Initialize flatpickr
            $row.find('.flatpickr-time').flatpickr({
              enableTime: true,
              noCalendar: true,
              dateFormat: "H:i",
              time_24hr: true,
              static: true,
              appendTo: document.querySelector("#ruleModal .modal-body")
            });

            // Update hidden JSON when user changes time
            $row.find('.condition-value-start, .condition-value-end').on('change', function() {
              const parent = $(this).closest('.condition-value');
              const startVal = parent.find('.condition-value-start').val();
              const endVal = parent.find('.condition-value-end').val();
              parent.find('.condition-value-hidden').val(JSON.stringify({
                start: startVal,
                end: endVal
              }));
            });
            break;
          default:
            // empty or unknown -> simple text
            const v2 = (typeof value === 'string') ? value : '';
            html =
              `<input type="text" class="form-control" name="conditions[${idx}][value]" value="${escapeHtml(v2)}">`;

        }

        $row.find('.condition-value').html(html);

        // Initialize flatpickr
        if (type === 'date') {
          $row.find('.flatpickr-date').flatpickr({
            dateFormat: "Y-m-d",
            minDate: "today",
            allowInput: true
          });
        }
        if (type === 'time_range') {
          $row.find('.flatpickr-time').flatpickr({
            enableTime: true,
            noCalendar: true,
            dateFormat: "H:i",
            time_24hr: true,
            static: true,
            appendTo: document.querySelector("#ruleModal .modal-body") // ⬅ attach inside modal
          });
        }
        // update hidden json when start/end change
        $row.find('.condition-value-start, .condition-value-end').on('change input', function() {
          const parent = $(this).closest('.condition-value');
          const s = parent.find('.condition-value-start').val() || '';
          const e = parent.find('.condition-value-end').val() || '';
          parent.find('.condition-value-hidden').val(JSON.stringify({
            start: s,
            end: e
          }));
        });
      }

      function escapeHtml(str) {
        if (str === null || str === undefined) return '';
        return String(str)
          .replace(/&/g, '&amp;')
          .replace(/"/g, '&quot;')
          .replace(/'/g, '&#039;')
          .replace(/</g, '&lt;')
          .replace(/>/g, '&gt;');
      }
      // Add new condition
      $('#addCondition').on('click', function() {
        const idx = nextCondIndex();
        const $row = renderConditionRow(idx, '', null);
        $('#conditionsWrapper').append($row);
        $('.select2-multiple').select2({
          placeholder: 'Select values',
          width: '100%',
          dropdownParent: $('#ruleModal')
        });
      });

      // Remove condition
      $(document).on('click', '.removeCondition', function() {
        $(this).closest('.condition-row').remove();
      });

      // Change condition type
      $(document).on('change', '.condition-type', function() {
        const $row = $(this).closest('.condition-row');
        const idx = $row.data('idx');
        const type = $(this).val();
        renderConditionValue($row, idx, type, null);
        $('.select2-multiple').select2({
          placeholder: 'Select values',
          width: '100%',
          dropdownParent: $('#ruleModal')
        });
      });

      // -------------------
      // Editing existing rule
      // -------------------
      $('.editRule').on('click', function() {
        let rule = $(this).data('rule');
        if (typeof rule === 'string') rule = JSON.parse(rule);
        condIndex = 0;
        // Fill main rule fields
        $('#ruleId').val(rule.id);
        $('#country').val(rule.country_id);
        $('[name="rule_name"]').val(rule.rule_name);
        $('[name="price_type"]').val(rule.price_type);
        $('[name="amount"]').val(rule.amount);
        $('[name="priority"]').val(rule.priority);
        $('[name="is_active"]').val(rule.is_active ? 1 : 0);

        // Clear old conditions
        $('#conditionsWrapper').html('');

        // render each condition; ensure condition_value is passed decoded if stored as JSON string
        (rule.conditions || []).forEach(cond => {
          const idx = nextCondIndex();
          let value = cond.condition_value;

          // try decode JSON
          if (typeof value === 'string') {
            try {
              value = JSON.parse(value);
            } catch (e) {}
          }

          const $row = renderConditionRow(idx, cond.condition_type, value);
          $('#conditionsWrapper').append($row);
        });

        $('#ruleModal').modal('show');
        $('.select2-multiple').select2({
          placeholder: 'Select values',
          width: '100%',
          dropdownParent: $('#ruleModal')
        });
      });


    });


    $(document).on('change', 'select[name="price_type"]', function() {
      let value = $(this).val();
      let label = "{{ __('locale.Amount') }}"; // default

      if (value === 'percent') {
        label = "{{ __('locale.Percentage') }}";
      } else if (value === 'multiplier') {
        label = "{{ __('locale.Multiplier') }}";
      } else if (value === 'discount') {
        label = "{{ __('locale.Discount Percent') }}"; // optional
      }

      $('label[for="amount_label"]').text(label);
    });
  </script>

@endsection
