/**
 * app-ecommerce-wallet-list Script
 */

'use strict';

// Datatable (js)

document.addEventListener('DOMContentLoaded', function (e) {
  let borderColor, bodyBg, headingColor;

  borderColor = config.colors.borderColor;
  bodyBg = config.colors.bodyBg;
  headingColor = config.colors.headingColor;

  // Variable declaration for table

  const dt_wallet_table = document.querySelector('.datatables-wallet');
  function getOwnerType() {
    return (
      (window.permissions && window.permissions.ownerType) ||
      $('.datatables-wallet').data('owner-type') ||
      $('#owner_type').val() ||
      new URLSearchParams(location.search).get('owner_type') ||
      null
    );
  }
  // E-commerce Products datatable

  if (dt_wallet_table) {
    const dt_wallet = new DataTable(dt_wallet_table, {
      processing: true,
      serverSide: false,
      ajax: {
        url: baseURL + '/wallet-list-all',
        type: 'GET',
        data: function (d) {
          const ownerType = getOwnerType();
          if (ownerType) {
            d.owner_type = ownerType;
          }
        }
      },
      columns: [
        // columns according to JSON
        { data: 'id' },
        { data: 'id', orderable: false, render: DataTable.render.select() },
        { data: 'name' },
        { data: 'balance' },
        { data: 'pending_dues', visible: false },
        { data: 'id' }
      ],
      createdRow: function (row, data) {
        $(row).attr('id', `trRow-${data.id}`);
      },
      columnDefs: [
        {
          // For Responsive
          className: 'control',
          searchable: false,
          orderable: false,
          responsivePriority: 2,
          targets: 0,
          render: function (data, type, full, meta) {
            return meta.row + meta.settings._iDisplayStart + 1;
          }
        },
        {
          // For Checkboxes
          targets: 1,
          orderable: false,
          searchable: false,
          responsivePriority: 3,
          checkboxes: true,
          visible: false,
          render: function () {
            return '<input type="checkbox" class="dt-checkboxes form-check-input">';
          },
          checkboxes: {
            selectAllRender: '<input type="checkbox" class="form-check-input">'
          }
        },
        {
          // Driver & user details
          targets: 2,
          render: function (data, type, full, meta) {
            const name = full.name ?? '-';
            const email = full.email ?? '-';
            const avatar = full.avatar;
            let output;

            if (avatar) {
              output = `<img src="${avatar}" alt="Avatar" class="rounded-circle">`;
            } else {
              const initials = (name.match(/\b\w/g) || []).slice(0, 2).join('').toUpperCase();
              output = `<span class="avatar-initial rounded-circle bg-label-primary">${initials}</span>`;
            }
            let profileUrl = '';
            if (full.owner_type === 'user') {
              profileUrl = `${baseURL}/users/${full.user_id}`;
            } else if (full.owner_type === 'store') {
              profileUrl = `${baseURL}/stores/${full.user_id}`;
            } else if (full.owner_type === 'driver') {
              profileUrl = `${baseURL}/drivers/${full.user_id}`;
            }
            return `
        <div class="d-flex align-items-center">
          <div class="avatar avatar-sm me-3">${output}</div>
          <div class="d-flex flex-column">
            <h6 class="m-0"><a href="${profileUrl}" class="text-heading">${name}</a></h6>
            <small>${email}</small>
          </div>
        </div>`;
          }
        },

        {
          targets: 3,
          render: function (data, type, full, meta) {
            return `<span>${full.currency}${full.balance}</span>`;
          }
        },
        {
          targets: 4,
          render: function (data, type, full, meta) {
            return `<span>${full.currency}${full.pending_dues}</span>`;
          }
        },

        {
          targets: -1,
          title: 'Actions',
          searchable: false,
          orderable: false,
          render: function (data, type, full, meta) {
            return `
  <div class="d-flex">
    ${
      full.pending_dues > 0
        ? `
      <button
        type="button"
        class="btn btn-sm btn-primary clear-dues-btn"
        data-id="${full.id}"
        data-balance="${full.balance}"
        data-dues="${full.pending_dues}"
        data-bs-toggle="modal"
        data-bs-target="#clearDuesModal">
         <i class="icon-base ti tabler-wallet icon-md"></i> Clear Dues
      </button>
    `
        : ''
    }
    <a href="${baseURL}/wallet-history/${full.id}/${full.user_id}"
      type="button" title="Wallet History"
      class="btn btn-sm btn-info ms-1">
       <i class="icon-base ti tabler-history icon-md"></i>
    </a>
  </div>
`;
          }
        }
      ],
      select: {
        style: 'multi',
        selector: 'td:nth-child(2)'
      },
      order: [0, 'asc'],
      layout: {
        topStart: {
          search: {
            placeholder: 'Search Dues',
            text: '_INPUT_'
          }
        },
        topEnd: {
          rowClass: 'row mx-3 my-0 justify-content-between',
          features: [
            {
              pageLength: {
                menu: [7, 10, 25, 50, 100],
                text: '_MENU_'
              }
            },
            {
              buttons: [
                {
                  extend: 'collection',
                  className: 'btn btn-label-primary dropdown-toggle',
                  text: '<span class="d-flex align-items-center gap-1"><i class="icon-base ti tabler-upload icon-xs"></i> <span class="d-none d-sm-inline-block">Export</span></span>',
                  buttons: [
                    {
                      extend: 'print',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-printer me-1"></i>Print</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;
                            const el = new DOMParser().parseFromString(inner, 'text/html').body.childNodes;
                            let result = '';
                            el.forEach(item => {
                              if (item.classList && item.classList.contains('user-name')) {
                                result += item.lastChild.firstChild.textContent;
                              } else {
                                result += item.textContent || item.innerText || '';
                              }
                            });
                            return result;
                          }
                        }
                      },
                      customize: function (win) {
                        win.document.body.style.color = headingColor;
                        win.document.body.style.borderColor = borderColor;
                        win.document.body.style.backgroundColor = bodyBg;
                        const table = win.document.body.querySelector('table');
                        table.classList.add('compact');
                        table.style.color = 'inherit';
                        table.style.borderColor = 'inherit';
                        table.style.backgroundColor = 'inherit';
                      }
                    },
                    {
                      extend: 'csv',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file me-1"></i>Csv</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;
                            const el = new DOMParser().parseFromString(inner, 'text/html').body.childNodes;
                            let result = '';
                            el.forEach(item => {
                              if (item.classList && item.classList.contains('user-name')) {
                                result += item.lastChild.firstChild.textContent;
                              } else {
                                result += item.textContent || item.innerText || '';
                              }
                            });
                            return result;
                          }
                        }
                      }
                    },
                    {
                      extend: 'excel',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-upload me-1"></i>Excel</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;
                            const el = new DOMParser().parseFromString(inner, 'text/html').body.childNodes;
                            let result = '';
                            el.forEach(item => {
                              if (item.classList && item.classList.contains('user-name')) {
                                result += item.lastChild.firstChild.textContent;
                              } else {
                                result += item.textContent || item.innerText || '';
                              }
                            });
                            return result;
                          }
                        }
                      }
                    },
                    {
                      extend: 'pdf',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file-text me-1"></i>Pdf</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;
                            const el = new DOMParser().parseFromString(inner, 'text/html').body.childNodes;
                            let result = '';
                            el.forEach(item => {
                              if (item.classList && item.classList.contains('user-name')) {
                                result += item.lastChild.firstChild.textContent;
                              } else {
                                result += item.textContent || item.innerText || '';
                              }
                            });
                            return result;
                          }
                        }
                      }
                    },
                    {
                      extend: 'copy',
                      text: `<i class="icon-base ti tabler-copy me-1"></i>Copy`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;
                            const el = new DOMParser().parseFromString(inner, 'text/html').body.childNodes;
                            let result = '';
                            el.forEach(item => {
                              if (item.classList && item.classList.contains('user-name')) {
                                result += item.lastChild.firstChild.textContent;
                              } else {
                                result += item.textContent || item.innerText || '';
                              }
                            });
                            return result;
                          }
                        }
                      }
                    }
                  ]
                }
              ]
            }
          ]
        },
        bottomStart: {
          rowClass: 'row mx-3 justify-content-between',
          features: ['info']
        },
        bottomEnd: 'paging'
      },
      language: {
        paginate: {
          next: '<i class="icon-base ti tabler-chevron-right scaleX-n1-rtl icon-18px"></i>',
          previous: '<i class="icon-base ti tabler-chevron-left scaleX-n1-rtl icon-18px"></i>',
          first: '<i class="icon-base ti tabler-chevrons-left scaleX-n1-rtl icon-18px"></i>',
          last: '<i class="icon-base ti tabler-chevrons-right scaleX-n1-rtl icon-18px"></i>'
        }
      },
      // For responsive popup
      responsive: {
        details: {
          display: DataTable.Responsive.display.modal({
            header: function (row) {
              const data = row.data();
              return 'Details of ' + data['driver_name'];
            }
          }),
          type: 'column',
          renderer: function (api, rowIdx, columns) {
            const data = columns
              .map(function (col) {
                return col.title !== '' // Do not show row in modal popup if title is blank (for check box)
                  ? `<tr data-dt-row="${col.rowIndex}" data-dt-column="${col.columnIndex}">
                      <td>${col.title}:</td>
                      <td>${col.data}</td>
                    </tr>`
                  : '';
              })
              .join('');

            if (data) {
              const div = document.createElement('div');
              div.classList.add('table-responsive');
              const table = document.createElement('table');
              div.appendChild(table);
              table.classList.add('table');
              const tbody = document.createElement('tbody');
              tbody.innerHTML = data;
              table.appendChild(tbody);
              return div;
            }
            return false;
          }
        }
      },
      initComplete: function () {}
    });

    $('#flatpickr-range').on('change', function () {
      dt_wallet.ajax.reload();
    });

    // Fixed header
    if (window.Helpers.isNavbarFixed()) {
      const navHeight = document.getElementById('layout-navbar').offsetHeight;
      new DataTable.FixedHeader(dt_wallet).headerOffset(navHeight);
    } else {
      new DataTable.FixedHeader(dt_wallet);
    }

    //? The 'delete-record' class is necessary for the functionality of the following code.
    document.addEventListener('click', function (e) {
      if (e.target.classList.contains('delete-record')) {
        dt_wallet.row(e.target.closest('tr')).remove().draw();
        const modalEl = document.querySelector('.dtr-bs-modal');
        if (modalEl && modalEl.classList.contains('show')) {
          const modal = bootstrap.Modal.getInstance(modalEl);
          modal?.hide();
        }
      }
    });

    let $btn = null;

    // Capture which submit button was clicked
    $(document).on('click', '#clearDuesForm button[type="submit"]', function () {
      $btn = $(this);
    });

    // Submit form via AJAX
    $('#clearDuesForm').on('submit', function (event) {
      event.preventDefault();

      $.ajax({
        type: 'POST',
        headers: {
          'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        url: $(this).attr('action'), // uses your route('wallet.clear-dues')
        data: new FormData(this),
        dataType: 'JSON',
        contentType: false,
        cache: false,
        processData: false,
        beforeSend: function () {
          toggleButtonLoading($btn, true, 'Processing...');
        },
        success: function (data) {
          if (data.status === 200) {
            // Update datatable row if wallet returned
            const wallet = data.data;
            const row = $(`#trRow-${wallet.id}`);
            if (row.length) {
              dt_wallet.row(row).data(wallet).invalidate().draw(false);
            }

            $('#clearDuesForm')[0].reset();
            const modalEl = document.getElementById('clearDuesModal');
            const bsModal = bootstrap.Modal.getInstance(modalEl);
            if (bsModal) bsModal.hide();
          }

          toggleButtonLoading($btn, false);
          const toastType = data.status == 200 ? 'bg-success' : 'bg-warning';
          window.showToast({
            title: 'Wallet 💰',
            message: data.message,
            type: toastType,
            animation: 'animate__fadeInUp',
            time: 'Just now',
            delay: 3000
          });
        },
        error: function (xhr) {
          toggleButtonLoading($btn, false);
          console.log(xhr.responseJSON);
        }
      });
    });

    // Prefill modal when clicking button
    $(document).on('click', '.clear-dues-btn', function () {
      let id = $(this).data('id');
      let dues = $(this).data('dues');

      $('#wallet_id').val(id);
      $('#amount').val(dues); // prefill pending dues
    });
  }

  // Filter form control to default size
  // ? setTimeout used for wallet-list table initialization
  setTimeout(() => {
    const elementsToModify = [
      { selector: '.dt-buttons .btn', classToRemove: 'btn-secondary', classToAdd: 'btn-label-secondary' },
      { selector: '.dt-search .form-control', classToRemove: 'form-control-sm', classToAdd: 'ms-0' },
      { selector: '.dt-length .form-select', classToRemove: 'form-select-sm' },
      { selector: '.dt-layout-table', classToRemove: 'row mt-2' },
      { selector: '.dt-layout-end', classToAdd: 'gap-md-2 gap-0' },
      { selector: '.dt-layout-full', classToRemove: 'col-md col-12', classToAdd: 'table-responsive' }
    ];

    // Delete record
    elementsToModify.forEach(({ selector, classToRemove, classToAdd }) => {
      document.querySelectorAll(selector).forEach(element => {
        if (classToRemove) {
          classToRemove.split(' ').forEach(className => element.classList.remove(className));
        }
        if (classToAdd) {
          classToAdd.split(' ').forEach(className => element.classList.add(className));
        }
      });
    });
  }, 100);
});
