// product-dropzone.js

import Dropzone from 'dropzone/dist/dropzone'; // or 'dropzone'
import axios from 'axios';

// 1) Global state
let uploadedImages = [];
Dropzone.autoDiscover = false;

// 2) Init after DOM ready
$(function () {
  const csrfToken = document.querySelector('meta[name="csrf-token"]').content;

  // ---------------------------
  // Quill editor
  // ---------------------------
  const commentEditor = document.querySelector('.comment-editor');
  if (commentEditor) {
    const quill = new Quill(commentEditor, {
      modules: { toolbar: '.comment-toolbar' },
      placeholder: 'Product Description',
      theme: 'snow'
    });
    if (window.productDescription) {
      quill.root.innerHTML = window.productDescription;
    }
  }

  // ---------------------------
  // Dropzone init
  // ---------------------------
  const el = document.getElementById('dropzone-basic');
  const url = el?.dataset.url;

  if (!el) {
    console.error('❌ #dropzone-basic not found');
  } else if (!url) {
    console.error('❌ data-url is empty');
  } else {
    const previewTemplate = `
      <div class="dz-preview dz-file-preview">
        <div class="dz-details">
          <div class="dz-thumbnail">
            <img data-dz-thumbnail />
            <div class="dz-success-mark"></div>
            <div class="dz-error-mark"></div>
            <div class="dz-error-message"><span data-dz-errormessage></span></div>
            <div class="progress">
              <div class="progress-bar" data-dz-uploadprogress></div>
            </div>
          </div>
          <div class="dz-filename" data-dz-name></div>
          <div class="dz-size" data-dz-size></div>
        </div>
      </div>`;

    const dz = new Dropzone(el, {
      url,
      method: 'POST',
      paramName: 'file',
      headers: { 'X-CSRF-TOKEN': csrfToken },
      previewTemplate,
      maxFilesize: 5,
      maxFiles: 4,
      parallelUploads: 1,
      acceptedFiles: 'image/*',
      addRemoveLinks: true,

      success(file, response) {
        if (response.status === 200 || response.success) {
          uploadedImages.push(response.data.filename);
          file.previewElement.setAttribute('data-filename', response.data.filename);
        } else {
          file.previewElement.classList.add('dz-error');
          file.previewElement.querySelector('[data-dz-errormessage]').textContent = 'Bad server response';
        }
      },

      error(file, errorMessage) {
        console.error('❌ Upload failed:', errorMessage);
      },

      removedfile(file) {
        const fn = file.previewElement.getAttribute('data-filename');
        if (fn) {
          axios
            .post(`${baseURL}/products/delete-temp`, { filename: fn }, { headers: { 'X-CSRF-TOKEN': csrfToken } })
            .then(() => {
              uploadedImages = uploadedImages.filter(x => x !== fn);
              console.log('🗑️ Deleted:', fn);
            })
            .catch(e => console.error('❌ Delete-temp failed:', e))
            .finally(() => file.previewElement.remove());
        } else {
          file.previewElement.remove();
        }
      }
    });

    // Load existing images in edit mode
    const productId = document.getElementById('product_id')?.value;
    if (productId) {
      $.post(`${baseURL}/products/load-images`, { product_id: productId, _token: csrfToken }, function (images) {
        if (Array.isArray(images)) {
          images.forEach(img => {
            const mockFile = { name: img.file, size: 123456 };
            dz.emit('addedfile', mockFile);
            dz.emit('thumbnail', mockFile, img.file);
            dz.emit('complete', mockFile);
            mockFile.previewElement.setAttribute('data-filename', img.file);
            uploadedImages.push(img.file);
          });
        }
      });
    }
  }

  // ---------------------------
  // Tagify
  // ---------------------------
  const tagifyBasicEl = document.querySelector('#ecommerce-product-tags');
  if (tagifyBasicEl) {
    new Tagify(tagifyBasicEl);
  }

  // ---------------------------
  // Flatpickr
  // ---------------------------
  const productDate = document.querySelector('.product-date');
  if (productDate) {
    productDate.flatpickr({
      monthSelectorType: 'static',
      defaultDate: new Date()
    });
  }

  // ---------------------------
  // Select2
  // ---------------------------
  const select2 = $('.select2');
  if (select2.length) {
    select2.each(function () {
      var $this = $(this);
      $this.wrap('<div class="position-relative"></div>').select2({
        dropdownParent: $this.parent(),
        placeholder: $this.data('placeholder')
      });
    });
  }

  // ---------------------------
  // Form Submit Intercept
  // ---------------------------
  const form = document.getElementById('productForm');
  const submitBtn = document.getElementById('submitProduct');

  if (form && submitBtn) {
    //console.log('✅ Binding submit listener…');
    submitBtn.addEventListener('click', function (e) {
      e.preventDefault();
      //console.log('📦 Intercepted product form submit');

      const productId = document.getElementById('product_id')?.value;
      const isEdit = !!productId;
      const url = isEdit ? `${baseURL}/products/${productId}` : `${baseURL}/products`;

      let formData = new FormData(form);

      // Quill description
      const quillEditor = document.querySelector('#ecommerce-category-description .ql-editor');
      if (quillEditor) {
        formData.append('description', quillEditor.innerHTML);
      }

      uploadedImages.forEach(img => formData.append('images[]', img));

      fetch(url, {
        method: 'POST',
        body: formData,
        headers: { 'X-CSRF-TOKEN': csrfToken }
      })
        .then(async response => {
          const contentType = response.headers.get('content-type');
          if (!response.ok) {
            const errorText = await response.text();
            throw new Error(`HTTP ${response.status}: ${errorText}`);
          }
          if (contentType && contentType.includes('application/json')) {
            return response.json();
          } else {
            throw new Error('Invalid JSON response');
          }
        })
        .then(data => {
          if (data.status == 200) {
            Swal.fire({
              icon: 'success',
              title: 'Success',
              text: data.message || 'Product saved successfully',
              customClass: { confirmButton: 'btn btn-primary' }
            }).then(() => {
              window.location.href = `${baseURL}/products`;
            });
          } else {
            Swal.fire({
              icon: 'error',
              title: 'Error',
              text: data.message || 'Something went wrong',
              customClass: { confirmButton: 'btn btn-danger' }
            });
          }
        })
        .catch(err => console.error('❌ Fetch error:', err));
    });
  } else {
    console.warn('⚠️ productForm or submitProduct not found');
  }

  // ---------------------------
  // Delete existing image
  // ---------------------------
  const uploadedImagesBox = document.getElementById('uploaded-images');
  if (uploadedImagesBox) {
    uploadedImagesBox.addEventListener('click', function (e) {
      const btn = e.target.closest('.delete-image');
      if (!btn) return;

      const imageId = btn.dataset.id;
      const box = btn.closest('.image-box');

      Swal.fire({
        title: 'Are you sure?',
        text: 'This will permanently delete the image.',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, delete it',
        cancelButtonText: 'Cancel',
        customClass: {
          confirmButton: 'btn btn-danger',
          cancelButton: 'btn btn-secondary ms-2'
        },
        buttonsStyling: false
      }).then(result => {
        if (result.isConfirmed) {
          axios
            .delete(`${baseURL}/products/delete-image/${imageId}`, {
              headers: { 'X-CSRF-TOKEN': csrfToken }
            })
            .then(response => {
              if (response.data.status == 200) {
                box.remove();
                Swal.fire({
                  icon: 'success',
                  title: 'Deleted!',
                  text: response.data.message,
                  timer: 1500,
                  showConfirmButton: false
                });
              } else {
                throw new Error(response.data.message || 'Delete failed');
              }
            })
            .catch(err => {
              Swal.fire({
                icon: 'error',
                title: 'Error',
                text: err.response?.data?.message || err.message
              });
            });
        }
      });
    });
  }
});
$(function () {
  var $allOptions = $('#category-org option');

  function filterChildCategories() {
    var parentId = $('#parent-category').val();
    // Save currently selected child category
    var selectedChild = $('#category-org').val();
    // Remove all except placeholder
    $('#category-org').find('option').not(':first').remove();

    if (!parentId) return;

    // Append only matching children
    $allOptions.each(function () {
      var childParentId = $(this).data('parent-id');
      if (childParentId == parentId) {
        $('#category-org').append($(this));
      }
    });
    // Restore previously selected child if still exists
    if (selectedChild) {
      $('#category-org').val(selectedChild);
    }
    // Trigger change for Select2 or other plugins
    $('#category-org').trigger('change');
  }

  $('#parent-category').on('change', filterChildCategories);

  // Trigger on page load
  filterChildCategories();
});
