'use strict';

document.addEventListener('DOMContentLoaded', function () {
  const dtCouponTable = document.querySelector('.datatables-users');
  const userView = baseUrl + 'admin/coupons';

  if (dtCouponTable) {
    const dt_coupon = new DataTable(dtCouponTable, {
      processing: true,
      serverSide: true,
      ajax: {
        url: baseUrl + 'admin/coupons-list',
        data: function (d) {
          d.type = $('#filter_type').val(); // send selected type to server
        },
        dataSrc: function (json) {
          if (typeof json.recordsTotal !== 'number') json.recordsTotal = 0;
          if (typeof json.recordsFiltered !== 'number') json.recordsFiltered = 0;
          json.data = Array.isArray(json.data) ? json.data : [];
          return json.data;
        }
      },
      columns: [
        { data: 'id' },
        { data: 'code' },
        { data: 'type' },
        { data: 'discount' },
        { data: 'status' },
        { data: 'id', visible: false },
        { data: 'start_date' },
        { data: 'end_date', visible: false },
        { data: 'id' }
      ],
      createdRow: function (row, data) {
        $(row).attr('id', `trRow-${data.id}`);
      },
      columnDefs: [
        {
          targets: 0,
          className: 'control',
          orderable: false,
          searchable: false,
          responsivePriority: 5,
          render: () => ''
        },

        {
          targets: 2,
          responsivePriority: 1,
          render: (data, type, full) => {
            let label = '';
            switch (full.type) {
              case 'ride_base':
                label = 'Ride Base';
                break;
              case 'cart_base':
                label = 'Cart Base';
                break;
              case 'product_base':
                label = 'Product Base';
                break;
              case 'customer_base':
                label = 'Customer Base';
                break;
              case 'seller_base':
                label = 'Seller Base';
                break;
              case 'offer_base':
                label = 'Offer Base';
                break;
              case 'category_base':
                label = 'Category Base';
                break;
              case 'register_base':
                label = 'Register';
                const details = full.details_parsed || [];
                if (details.length > 0) {
                  label += `<br><small>Total: ${details[0].maximum_usage ?? ''}<br>Usages: ${full.usage_count}</small>`;
                }
                break;
            }
            return label;
          }
        },
        {
          targets: 3,
          render: (data, type, full) => {
            const symbol = full.discount_type === 'amount' ? window.permissions.currencySymbol : '';
            const suffix =
              full.discount_type === 'amount' ? ' Flat' : full.discount_type === 'points' ? ' Points' : ' %';
            return `${symbol}${full.discount} ${suffix}`;
          }
        },
        {
          targets: 4,
          render: (data, type, full) => {
            const formatDate = dateStr => {
              const date = new Date(dateStr);
              const day = String(date.getDate()).padStart(2, '0');
              const month = String(date.getMonth() + 1).padStart(2, '0');
              const year = date.getFullYear();
              return `${day}-${month}-${year}`;
            };

            const now = new Date();

            const startDate = new Date(full.start_date);
            const endDate = new Date(full.end_date);

            let badgeClass = 'bg-label-primary';
            if (now < startDate) {
              badgeClass = 'bg-label-primary'; // upcoming
            } else if (now >= startDate && now <= endDate) {
              badgeClass = 'bg-label-success'; // active now
            } else if (now > endDate) {
              badgeClass = 'bg-label-danger'; // expired
            }

            return `
      <span class="badge ${badgeClass}">
        ${formatDate(full.start_date)} - ${formatDate(full.end_date)}
      </span>
    `;
          }
        },

        {
          targets: 6,
          render: (data, type, full) => {
            const checked = full.status === 'Enable' ? 'checked' : '';
            if (window.permissions.canEditCoupon) {
              return `
              <label class="switch switch-square">
                <input type="checkbox" class="switch-input changestatus" ${checked}
                  data-id="${full.id}" data-status="${full.status}" data-url="${full['status_url']}">
                <span class="switch-toggle-slider">
                  <span class="switch-on"><i class="icon-base ti tabler-check"></i></span>
                  <span class="switch-off"><i class="icon-base ti tabler-x"></i></span>
                </span>
                <span class="switch-label"></span>
              </label>`;
            } else {
              return `<span class="badge bg-label-primary">${full.status}</span>`;
            }
          }
        },
        {
          targets: -1,
          title: 'Actions',
          orderable: false,
          searchable: false,
          render: (data, type, full) => {
            let actions = `<div class="d-flex align-items-center">`;
            // Seller base popover
            if (full.type === 'seller_base' && full.details_parsed?.length) {
              let sellerHtml = `
      <div class="coupon-details">
        <div class="row header border border-1 p-1">
          <div class="col"><b>User ID</b></div>
          <div class="col"><b>User Name</b></div>
        </div>`;
              full.details_parsed.forEach(discount => {
                if (discount.user_id) {
                  sellerHtml += `
        <div class="row p-1">
          <div class="col">${discount.user_id}</div>
          <div class="col">${discount.user_name ?? 'N/A'}</div>
        </div>`;
                }
              });
              sellerHtml += `</div>`;

              actions += `
      <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect" data-bs-toggle="popover"
         data-bs-trigger="hover" data-bs-html="true"
         data-bs-content='${sellerHtml}' title="Sellers">
        <i class="icon-base ti tabler-users icon-md"></i>
      </a>`;
            }
            // Customer base popover
            if (full.type === 'customer_base' && full.details_parsed?.length) {
              let sellerHtml = `
      <div class="coupon-details">
        <div class="row header border border-1 p-1">
          <div class="col"><b>User ID</b></div>
          <div class="col"><b>User Name</b></div>
           <div class="col"><b>Min Buy</b></div>
           <div class="col"><b>Max Discount</b></div>
        </div>`;
              full.details_parsed.forEach(discount => {
                if (discount.user_id) {
                  sellerHtml += `
        <div class="row p-1">
          <div class="col">${discount.user_id}</div>
          <div class="col">${discount.user_name ?? 'N/A'}</div>
            <div class="col">${discount.min_buy}</div>
          <div class="col">${discount.max_discount ?? 'N/A'}</div>
        </div>`;
                }
              });
              sellerHtml += `</div>`;

              actions += `
      <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect" data-bs-toggle="popover"
         data-bs-trigger="hover" data-bs-html="true"
         data-bs-content='${sellerHtml}' title="Customers">
        <i class="icon-base ti tabler-users icon-md"></i>
      </a>`;
            }
            // Delete button (if allowed)
            if (window.permissions?.canDeleteCoupon) {
              actions += `
      <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect delete-record"
        data-url="${full['delete_url']}" data-id="${full.id}" data-title="${full.name}">
        <i class="icon-base ti tabler-trash icon-md"></i>
      </a>`;
            }

            // Dropdown toggle
            actions += `
    <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect dropdown-toggle hide-arrow"
      data-bs-toggle="dropdown">
      <i class="icon-base ti tabler-dots-vertical icon-md"></i>
    </a>
    <div class="dropdown-menu dropdown-menu-end m-0">`;

            // View
            actions += `
    <a href="${baseUrl}admin/coupons/${full.encrypted_id}" class="dropdown-item d-none">
      <i class="icon-base ti tabler-eye icon-md"></i> View
    </a>`;

            // Edit
            if (window.permissions?.canEditCoupon) {
              actions += `
      <a href="${baseUrl}admin/coupons/${full.encrypted_id}/edit" class="dropdown-item">
        <i class="icon-base ti tabler-edit icon-md"></i> Edit
      </a>`;
            }
            if (window.permissions.canListCoupon) {
              actions += `
              <a href="${baseURL}/coupons/${full.id}" class="dropdown-item">
                <i class="icon-base ti tabler-eye icon-md"></i> View
              </a>`;
            }

            actions += `</div></div>`;
            return actions;
          }
        }
      ],
      drawCallback: function () {
        $('[data-bs-toggle="popover"]').popover({
          trigger: 'hover',
          html: true,
          container: 'body',
          placement: 'auto'
        });
      },
      order: [[5, 'desc']],
      responsive: true,
      language: {
        paginate: {
          next: '<i class="icon-base ti tabler-chevron-right scaleX-n1-rtl icon-18px"></i>',
          previous: '<i class="icon-base ti tabler-chevron-left scaleX-n1-rtl icon-18px"></i>'
        }
      },
      layout: {
        topStart: {
          rowClass: 'row my-md-0 me-3 ms-0 justify-content-between',
          features: [
            {
              pageLength: {
                menu: [10, 25, 50, 100],
                text: '_MENU_'
              }
            }
          ]
        },
        topEnd: {
          features: [
            {
              search: {
                placeholder: 'Search Coupons',
                text: '_INPUT_'
              }
            },
            {
              buttons: [
                {
                  extend: 'collection',
                  className: 'btn btn-label-secondary dropdown-toggle me-4',
                  text: '<span class="d-flex align-items-center gap-1"><i class="icon-base ti tabler-upload icon-xs"></i> <span class="d-inline-block">Export</span></span>',
                  buttons: [
                    {
                      extend: 'print',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-printer me-1"></i>Print</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [1, 2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Check if inner is HTML content
                            if (typeof inner === 'string' && inner.indexOf('<') > -1) {
                              const parser = new DOMParser();
                              const doc = parser.parseFromString(inner, 'text/html');

                              // Get all text content
                              let text = '';

                              // Handle specific elements
                              const userNameElements = doc.querySelectorAll('.role-name');
                              if (userNameElements.length > 0) {
                                userNameElements.forEach(el => {
                                  // Get text from nested structure
                                  const nameText =
                                    el.querySelector('.fw-medium')?.textContent ||
                                    el.querySelector('.d-block')?.textContent ||
                                    el.textContent;
                                  text += nameText.trim() + ' ';
                                });
                              } else {
                                // Get regular text content
                                text = doc.body.textContent || doc.body.innerText;
                              }

                              return text.trim();
                            }

                            return inner;
                          }
                        }
                      },
                      customize: function (win) {
                        win.document.body.style.color = config.colors.headingColor;
                        win.document.body.style.borderColor = config.colors.borderColor;
                        win.document.body.style.backgroundColor = config.colors.bodyBg;
                        const table = win.document.body.querySelector('table');
                        table.classList.add('compact');
                        table.style.color = 'inherit';
                        table.style.borderColor = 'inherit';
                        table.style.backgroundColor = 'inherit';
                      }
                    },
                    {
                      extend: 'csv',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file me-1"></i>Csv</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [1, 2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'excel',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file-export me-1"></i>Excel</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [1, 2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'pdf',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file-text me-1"></i>Pdf</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [1, 2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'copy',
                      text: `<i class="icon-base ti tabler-copy me-1"></i>Copy`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [1, 2, 3, 4],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    }
                  ]
                },
                {
                  text: `<i class="icon-base ti tabler-plus icon-sm me-0 me-sm-2"></i>
                      <span class="d-none d-sm-inline-block">Add New Coupon</span>`,
                  className: 'btn btn-primary d-flex align-items-center',
                  action: function () {
                    window.location.href = `${baseURL}/coupons/create`;
                  }
                }
              ]
            }
          ]
        },
        bottomStart: {
          rowClass: 'row mx-3 justify-content-between',
          features: ['info']
        },
        bottomEnd: 'paging'
      },
      language: {
        paginate: {
          next: '<i class="icon-base ti tabler-chevron-right scaleX-n1-rtl icon-18px"></i>',
          previous: '<i class="icon-base ti tabler-chevron-left scaleX-n1-rtl icon-18px"></i>',
          first: '<i class="icon-base ti tabler-chevrons-left scaleX-n1-rtl icon-18px"></i>',
          last: '<i class="icon-base ti tabler-chevrons-right scaleX-n1-rtl icon-18px"></i>'
        }
      },
      // For responsive popup
      responsive: {
        details: {
          display: DataTable.Responsive.display.modal({
            header: function (row) {
              const data = row.data();
              return 'Details of ' + data['name'];
            }
          }),
          type: 'column',
          renderer: function (api, rowIdx, columns) {
            const data = columns
              .map(function (col) {
                return col.title !== '' // Do not show row in modal popup if title is blank (for check box)
                  ? `<tr data-dt-row="${col.rowIndex}" data-dt-column="${col.columnIndex}">
                      <td>${col.title}:</td>
                      <td>${col.data}</td>
                    </tr>`
                  : '';
              })
              .join('');

            if (data) {
              const div = document.createElement('div');
              div.classList.add('table-responsive');
              const table = document.createElement('table');
              div.appendChild(table);
              table.classList.add('table');
              const tbody = document.createElement('tbody');
              tbody.innerHTML = data;
              table.appendChild(tbody);
              return div;
            }
            return false;
          }
        }
      },
      initComplete: function (settings, json) {
        // Remove btn-secondary from export buttons
        document.querySelectorAll('.dt-buttons .btn').forEach(btn => {
          btn.classList.remove('btn-secondary');
          if (!json.canCreate) {
            $('.add-new').addClass('d-none');
          }
        });
      }
    });

    $('#filter_type').on('change', function () {
      dt_coupon.ajax.reload();
    });
    // Fixed header
    if (window.Helpers.isNavbarFixed()) {
      const navHeight = document.getElementById('layout-navbar').offsetHeight;
      new DataTable.FixedHeader(dt_coupon).headerOffset(navHeight);
    } else {
      new DataTable.FixedHeader(dt_coupon);
    }

    // Delete Record
    document.addEventListener('click', function (e) {
      const deleteBtn = e.target.closest('.delete-record');
      if (!deleteBtn) return;

      const couponId = deleteBtn.dataset.id;

      Swal.fire({
        title: 'Are you sure?',
        text: 'This coupon will be deleted!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, delete it!',
        customClass: {
          confirmButton: 'btn btn-primary me-3',
          cancelButton: 'btn btn-label-secondary'
        },
        buttonsStyling: false
      }).then(function (result) {
        if (result.value) {
          fetch(`${baseURL}/coupons/${couponId}`, {
            method: 'DELETE',
            headers: {
              'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
              'Content-Type': 'application/json'
            }
          })
            .then(response => {
              if (response.ok) {
                dt_coupon.row(`#trRow-${couponId}`).remove().draw(false);
                Swal.fire({
                  icon: 'success',
                  title: 'Deleted!',
                  text: 'Coupon has been deleted.',
                  customClass: {
                    confirmButton: 'btn btn-success'
                  }
                });
              } else {
                throw new Error('Delete failed');
              }
            })
            .catch(error => console.error(error));
        }
      });
    });
  }
});
