'use strict';

document.addEventListener('DOMContentLoaded', function () {
  const dtCategoryTable = document.querySelector('.datatables-users');
  const userView = baseUrl + 'admin/categories';

  if (dtCategoryTable) {
    const roleHelper = document.getElementById('role-helper');
    const addLabel = roleHelper.dataset.add;
    const queryValue = roleHelper.dataset.query;
    const dt_category = new DataTable(dtCategoryTable, {
      processing: true,
      serverSide: false,
      ajax: {
        url: baseUrl + 'admin/categories-list?' + queryValue,
        dataSrc: function (json) {
          if (typeof json.recordsTotal !== 'number') json.recordsTotal = 0;
          if (typeof json.recordsFiltered !== 'number') json.recordsFiltered = 0;
          json.data = Array.isArray(json.data) ? json.data : [];
          return json.data;
        }
      },
      columns: [
        { data: 'id' },
        { data: 'image' },
        { data: 'name' },
        { data: 'parent_name' },
        { data: 'status' },
        { data: 'id', visible: false },
        { data: 'parent_id', visible: false },
        { data: 'prescription' },
        { data: 'id' }
      ],
      createdRow: function (row, data) {
        $(row).attr('id', `trRow-${data.id}`);
      },
      columnDefs: [
        {
          targets: 0,
          className: 'control',
          orderable: false,
          searchable: false,
          responsivePriority: 5,
          render: () => ''
        },
        {
          targets: 1,
          responsivePriority: 1,
          render: (data, type, full) => {
            const imageUrl = full.image ? `${baseUrl}storage/${full.image}` : `${baseUrl}assets/img/placeholder.jpg`;
            return `
      <div class="d-flex align-items-center">
        <div class="avatar avatar-sm me-2">
          <img src="${imageUrl}" alt="${full.name}" class="rounded-circle">
        </div>
        <div class="d-flex flex-column">
          <span class="fw-semibold"></span>
        </div>
      </div>
    `;
          }
        },

        {
          targets: 2,
          responsivePriority: 1,
          render: (data, type, full) => full.name
        },
        {
          targets: 3,
          render: (data, type, full) => full.parent_name || '-'
        },
        {
          targets: 4,
          render: (data, type, full) => {
            const checked = full.status === 1 ? 'checked' : '';
            if (window.permissions.canEditCategory) {
              return `
              <label class="switch switch-square">
                <input type="checkbox" class="switch-input changestatus" ${checked}
                  data-id="${full.id}" data-status="${full.status}" data-url="${full['status_url']}">
                <span class="switch-toggle-slider">
                  <span class="switch-on"><i class="icon-base ti tabler-check"></i></span>
                  <span class="switch-off"><i class="icon-base ti tabler-x"></i></span>
                </span>
                <span class="switch-label"></span>
              </label>`;
            } else {
              return `<span class="badge bg-label-primary">${full.status}</span>`;
            }
          }
        },
        {
          targets: 7,
          render: (data, type, full) => {
            let prescriptionText;
            if (full.is_prescrible) {
              prescriptionText = full.prescriptions == 1 ? 'Yes' : 'No';
            } else {
              prescriptionText = 'N/A';
            }
            let taxText = full.tax ? ` <br>Tax: ${full.tax}%` : '';
            return prescriptionText + taxText;
          }
        },
        {
          targets: -1,
          title: 'Actions',
          orderable: false,
          searchable: false,
          render: (data, type, full) => {
            let actions = `<div class="d-flex align-items-center">`;
            // Delete button (if allowed)
            if (window.permissions.canDeleteCategory && full.parent_id) {
              actions += `
              <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect delete-record"
                data-url="${full['delete_url']}" data-id="${full.id}" data-title="${full.name}">
                <i class="icon-base ti tabler-trash icon-md"></i>
              </a>`;
            }

            // Dropdown toggle
            actions += `
            <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect dropdown-toggle hide-arrow"
              data-bs-toggle="dropdown">
              <i class="icon-base ti tabler-dots-vertical icon-md"></i>
            </a>
            <div class="dropdown-menu dropdown-menu-end m-0">`;

            // Edit button (if allowed)
            if (window.permissions.canEditCategory) {
              actions += `
              <a href="javascript:;" class="dropdown-item edit-category-modal"
                data-id="${full.id}"
                data-name="${full.name}"
                 data-image="${full.image ? `${baseUrl}/storage/${full.image}` : ''}"
                data-prescription="${full.prescriptions}"
                data-tax="${full.tax}"
                data-parent_pres="${full.is_prescrible}"
                data-parent_id="${full.parent_id || ''}">
                <i class="icon-base ti tabler-edit icon-md"></i> Edit
              </a>`;
            }

            actions += `</div></div>`;
            return actions;
          }
        }
      ],
      order: [[5, 'desc']],
      responsive: true,
      language: {
        paginate: {
          next: '<i class="icon-base ti tabler-chevron-right scaleX-n1-rtl icon-18px"></i>',
          previous: '<i class="icon-base ti tabler-chevron-left scaleX-n1-rtl icon-18px"></i>'
        }
      },
      layout: {
        topStart: {
          rowClass: 'row my-md-0 me-3 ms-0 justify-content-between',
          features: [
            {
              pageLength: {
                menu: [10, 25, 50, 100],
                text: '_MENU_'
              }
            }
          ]
        },
        topEnd: {
          features: [
            {
              search: {
                placeholder: 'Search Categories',
                text: '_INPUT_'
              }
            },
            {
              buttons: [
                {
                  extend: 'collection',
                  className: 'btn btn-label-secondary dropdown-toggle me-4',
                  text: '<span class="d-flex align-items-center gap-1"><i class="icon-base ti tabler-upload icon-xs"></i> <span class="d-inline-block">Export</span></span>',
                  buttons: [
                    {
                      extend: 'print',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-printer me-1"></i>Print</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 7],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Check if inner is HTML content
                            if (typeof inner === 'string' && inner.indexOf('<') > -1) {
                              const parser = new DOMParser();
                              const doc = parser.parseFromString(inner, 'text/html');

                              // Get all text content
                              let text = '';

                              // Handle specific elements
                              const userNameElements = doc.querySelectorAll('.role-name');
                              if (userNameElements.length > 0) {
                                userNameElements.forEach(el => {
                                  // Get text from nested structure
                                  const nameText =
                                    el.querySelector('.fw-medium')?.textContent ||
                                    el.querySelector('.d-block')?.textContent ||
                                    el.textContent;
                                  text += nameText.trim() + ' ';
                                });
                              } else {
                                // Get regular text content
                                text = doc.body.textContent || doc.body.innerText;
                              }

                              return text.trim();
                            }

                            return inner;
                          }
                        }
                      },
                      customize: function (win) {
                        win.document.body.style.color = config.colors.headingColor;
                        win.document.body.style.borderColor = config.colors.borderColor;
                        win.document.body.style.backgroundColor = config.colors.bodyBg;
                        const table = win.document.body.querySelector('table');
                        table.classList.add('compact');
                        table.style.color = 'inherit';
                        table.style.borderColor = 'inherit';
                        table.style.backgroundColor = 'inherit';
                      }
                    },
                    {
                      extend: 'csv',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file me-1"></i>Csv</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 7],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'excel',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file-export me-1"></i>Excel</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 7],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'pdf',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file-text me-1"></i>Pdf</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 7],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'copy',
                      text: `<i class="icon-base ti tabler-copy me-1"></i>Copy`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 7],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    }
                  ]
                },
                {
                  text: '<i class="icon-base ti tabler-plus me-0 me-sm-1 icon-16px"></i><span class="d-none d-sm-inline-block">Add New Category</span>',
                  className: 'add-new btn btn-primary rounded-2 waves-effect waves-light',
                  attr: {
                    'data-bs-toggle': 'offcanvas',
                    'data-bs-target': '#offcanvasAdd'
                  },
                  init: function (api, node, config) {
                    if (!queryValue) {
                      $(node).hide();
                    }
                    $(node).removeClass('btn-secondary'); // ✅ force-remove it
                  },
                  action: function () {
                    // ✅ Reset form every time you click Add
                    $('#upload_form')[0].reset();
                    $('#category_id').val('');
                    $('#offcanvasAddLabel').text('Add New Category');
                    $('#saveBtn').text('Submit');
                    $('#preview_image').hide();
                    $('#hidePres').addClass('d-none');
                  }
                }
              ]
            }
          ]
        },
        bottomStart: {
          rowClass: 'row mx-3 justify-content-between',
          features: ['info']
        },
        bottomEnd: 'paging'
      },
      language: {
        paginate: {
          next: '<i class="icon-base ti tabler-chevron-right scaleX-n1-rtl icon-18px"></i>',
          previous: '<i class="icon-base ti tabler-chevron-left scaleX-n1-rtl icon-18px"></i>',
          first: '<i class="icon-base ti tabler-chevrons-left scaleX-n1-rtl icon-18px"></i>',
          last: '<i class="icon-base ti tabler-chevrons-right scaleX-n1-rtl icon-18px"></i>'
        }
      },
      // For responsive popup
      responsive: {
        details: {
          display: DataTable.Responsive.display.modal({
            header: function (row) {
              const data = row.data();
              return 'Details of ' + data['name'];
            }
          }),
          type: 'column',
          renderer: function (api, rowIdx, columns) {
            const data = columns
              .map(function (col) {
                return col.title !== '' // Do not show row in modal popup if title is blank (for check box)
                  ? `<tr data-dt-row="${col.rowIndex}" data-dt-column="${col.columnIndex}">
                      <td>${col.title}:</td>
                      <td>${col.data}</td>
                    </tr>`
                  : '';
              })
              .join('');

            if (data) {
              const div = document.createElement('div');
              div.classList.add('table-responsive');
              const table = document.createElement('table');
              div.appendChild(table);
              table.classList.add('table');
              const tbody = document.createElement('tbody');
              tbody.innerHTML = data;
              table.appendChild(tbody);
              return div;
            }
            return false;
          }
        }
      }
    });

    $('#filter_parent').on('change', function () {
      const value = this.value;
      dt_category.column(6).search(value).draw(); // column index 1 = parent_category
    });

    // Filter by subcategory
    $('#filter_subcategory').on('change', function () {
      const value = this.value;
      dt_category.column(5).search(value).draw(); // column index 2 = subcategory
    });
    // Load subcategories when parent changes
    $('#filter_parent').on('change', function () {
      const parentId = $(this).val();
      $('#filter_subcategory').html('<option value="">All</option>');

      if (parentId) {
        fetch(`${baseURL}/categories/${parentId}/subcategories`)
          .then(res => res.json())
          .then(data => {
            data.forEach(item => {
              $('#filter_subcategory').append(`<option value="${item.id}">${item.name}</option>`);
            });
          });
      }
    });

    // Fixed header
    if (window.Helpers.isNavbarFixed()) {
      const navHeight = document.getElementById('layout-navbar').offsetHeight;
      new DataTable.FixedHeader(dt_category).headerOffset(navHeight);
    } else {
      new DataTable.FixedHeader(dt_category);
    }

    // Delete Record
    document.addEventListener('click', function (e) {
      const deleteBtn = e.target.closest('.delete-record');
      if (!deleteBtn) return;

      const categoryId = deleteBtn.dataset.id;

      Swal.fire({
        title: 'Are you sure?',
        text: 'This category will be deleted!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, delete it!',
        customClass: {
          confirmButton: 'btn btn-primary me-3',
          cancelButton: 'btn btn-label-secondary'
        },
        buttonsStyling: false
      }).then(function (result) {
        if (result.value) {
          fetch(`${baseURL}/categories/${categoryId}`, {
            method: 'DELETE',
            headers: {
              'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
              'Content-Type': 'application/json'
            }
          })
            .then(response => {
              if (response.ok) {
                dt_category.row(`#trRow-${categoryId}`).remove().draw(false);
                Swal.fire({
                  icon: 'success',
                  title: 'Deleted!',
                  text: 'Category has been deleted.',
                  customClass: {
                    confirmButton: 'btn btn-success'
                  }
                });
              } else {
                throw new Error('Delete failed');
              }
            })
            .catch(error => console.error(error));
        }
      });
    });
  }
});
