'use strict';

document.addEventListener('DOMContentLoaded', function () {
  const dtCarTable = document.querySelector('.datatables-users');
  const userView = baseURL + '/cars';
  var statusObj = {
    0: { title: 'Inactive', class: 'bg-label-primary' },
    1: { title: 'Active', class: 'bg-label-success' },
    2: { title: 'Deleted', class: 'bg-label-danger' }
  };
  if (dtCarTable) {
    const dt_model = new DataTable(dtCarTable, {
      processing: true,
      serverSide: true,
      ajax: {
        url: baseUrl + 'admin/car-list-all',
        dataSrc: function (json) {
          if (typeof json.recordsTotal !== 'number') json.recordsTotal = 0;
          if (typeof json.recordsFiltered !== 'number') json.recordsFiltered = 0;
          json.data = Array.isArray(json.data) ? json.data : [];
          return json.data;
        },
        data: function (d) {
          d.vehicle_parent_category_id = $('#filter_parent').val();
          d.vehicle_category_id = $('#filter_subcategory').val();
          d.vehicle_brand_id = $('#filter_brand').val();
          d.vehicle_model_id = $('#filter_model').val();
        }
      },
      columns: [
        { data: 'id' },
        { data: 'registration_number' },
        { data: 'license_plate' },
        { data: 'vehicle_color' },
        { data: 'vehicle_year' },
        { data: 'vehicle_brand', orderable: false, searchable: false },
        { data: 'status' },
        { data: 'id', visible: false },
        { data: 'id' }
      ],
      createdRow: function (row, data) {
        $(row).attr('id', `trRow-${data.id}`);
      },
      columnDefs: [
        {
          targets: 0,
          className: 'control',
          orderable: false,
          searchable: false,
          responsivePriority: 5,
          render: () => ''
        },
        {
          targets: 1,
          responsivePriority: 1,
          render: (data, type, full) => {
            const imageUrl = full.vehicle_image
              ? `${baseUrl}storage/${full.vehicle_image}`
              : `${baseUrl}assets/img/placeholder.jpg`;
            return `
      <div class="d-flex align-items-center">
        <div class="avatar avatar-sm me-2">
          <img src="${imageUrl}" alt="${full.vehicle_model}" class="rounded-circle">
        </div>
        <div class="d-flex flex-column">
          <span class="fw-semibold"></span>
        </div>
      </div>
    `;
          }
        },

        {
          targets: 2,
          responsivePriority: 1,
          render: (data, type, full) => {
            return `<a href="${userView}/${full.id}">${full.vehicle_model}</an> `;
          }
        },

        {
          targets: 3,
          responsivePriority: 2,
          render: (data, type, full) => {
            return `<span class="badge bg-label-primary">${full.parent_category}</span> <br><span class="badge bg-label-success">${full.vehicle_category}</span>`;
          }
        },
        {
          targets: 4,
          responsivePriority: 3,
          render: (data, type, full) => {
            const color = full.color
              ? `<span class="px-2" style="display:inline-block;width:auto;height:22px;background-color:${full.color.code};border:1px solid #ccc;border-radius:3px;margin-left:5px;">${full.color.name}</span>`
              : '';
            return `${full.vehicle_brand} <br>${color}`;
          }
        },
        {
          targets: 5,
          responsivePriority: 3,
          render: (data, type, full) => {
            return `License Plate: ${full.license_plate} <br>Registration: ${full.registration_number}<br>Year: ${full.vehicle_year}`;
          }
        },

        {
          targets: 6,
          visible: false,
          render: (data, type, full) => {
            const checked = full.status === 1 ? 'checked' : '';
            if (window.permissions.canEditCar) {
              return `
              <label class="switch switch-square">
                <input type="checkbox" class="switch-input changestatus" ${checked}
                  data-id="${full.id}" data-status="${full.status}" data-url="${full['status_url']}">
                <span class="switch-toggle-slider">
                  <span class="switch-on"><i class="icon-base ti tabler-check"></i></span>
                  <span class="switch-off"><i class="icon-base ti tabler-x"></i></span>
                </span>
                <span class="switch-label"></span>
              </label>`;
            } else {
              return `<span class="badge ${statusObj[full.status].class}">${statusObj[full.status].title}</span>`;
            }
          }
        },
        {
          targets: -1,
          title: 'Actions',
          orderable: false,
          searchable: false,
          visible: true,
          render: (data, type, full) => {
            let actions = `<div class="d-flex align-items-center">`;
            // Delete button (if allowed)
            if (window.permissions.canDeleteCar) {
              actions += `
              <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect delete-record"
                data-url="${full['delete_url']}" data-id="${full.id}" data-title="${full.name}">
                <i class="icon-base ti tabler-trash icon-md"></i>
              </a>`;
            }

            // Dropdown toggle
            actions += `
            <a href="javascript:;" class="btn btn-icon btn-text-secondary rounded-pill waves-effect dropdown-toggle hide-arrow"
              data-bs-toggle="dropdown">
              <i class="icon-base ti tabler-dots-vertical icon-md"></i>
            </a>
            <div class="dropdown-menu dropdown-menu-end m-0">`;
            actions += `
            <a href="${userView}/${full.id}" class="dropdown-item"><i class="icon-base ti tabler-eye icon-md"></i> View</a>`;
            // Edit button (if allowed)
            if (window.permissions.canEditCar) {
              actions += `
              <a href="${baseURL}/cars/${full.id}/edit" class="dropdown-item">
                <i class="icon-base ti tabler-edit icon-md"></i> Edit
              </a>`;
            }

            actions += `</div></div>`;
            return actions;
          }
        }
      ],
      order: [[7, 'desc']],
      responsive: true,
      language: {
        paginate: {
          next: '<i class="icon-base ti tabler-chevron-right scaleX-n1-rtl icon-18px"></i>',
          previous: '<i class="icon-base ti tabler-chevron-left scaleX-n1-rtl icon-18px"></i>'
        }
      },
      layout: {
        topStart: {
          rowClass: 'row my-md-0 me-3 ms-0 justify-content-between',
          features: [
            {
              pageLength: {
                menu: [10, 25, 50, 100],
                text: '_MENU_'
              }
            }
          ]
        },
        topEnd: {
          features: [
            {
              search: {
                placeholder: 'Search Car',
                text: '_INPUT_'
              }
            },
            {
              buttons: [
                {
                  extend: 'collection',
                  className: 'btn btn-label-secondary dropdown-toggle me-4',
                  text: '<span class="d-flex align-items-center gap-1"><i class="icon-base ti tabler-upload icon-xs"></i> <span class="d-inline-block">Export</span></span>',
                  buttons: [
                    {
                      extend: 'print',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-printer me-1"></i>Print</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 5],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Check if inner is HTML content
                            if (typeof inner === 'string' && inner.indexOf('<') > -1) {
                              const parser = new DOMParser();
                              const doc = parser.parseFromString(inner, 'text/html');

                              // Get all text content
                              let text = '';

                              // Handle specific elements
                              const userNameElements = doc.querySelectorAll('.role-name');
                              if (userNameElements.length > 0) {
                                userNameElements.forEach(el => {
                                  // Get text from nested structure
                                  const nameText =
                                    el.querySelector('.fw-medium')?.textContent ||
                                    el.querySelector('.d-block')?.textContent ||
                                    el.textContent;
                                  text += nameText.trim() + ' ';
                                });
                              } else {
                                // Get regular text content
                                text = doc.body.textContent || doc.body.innerText;
                              }

                              return text.trim();
                            }

                            return inner;
                          }
                        }
                      },
                      customize: function (win) {
                        win.document.body.style.color = config.colors.headingColor;
                        win.document.body.style.borderColor = config.colors.borderColor;
                        win.document.body.style.backgroundColor = config.colors.bodyBg;
                        const table = win.document.body.querySelector('table');
                        table.classList.add('compact');
                        table.style.color = 'inherit';
                        table.style.borderColor = 'inherit';
                        table.style.backgroundColor = 'inherit';
                      }
                    },
                    {
                      extend: 'csv',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file me-1"></i>Csv</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 5],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'excel',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file-export me-1"></i>Excel</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 5],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'pdf',
                      text: `<span class="d-flex align-items-center"><i class="icon-base ti tabler-file-text me-1"></i>Pdf</span>`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 5],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    },
                    {
                      extend: 'copy',
                      text: `<i class="icon-base ti tabler-copy me-1"></i>Copy`,
                      className: 'dropdown-item',
                      exportOptions: {
                        columns: [2, 3, 4, 5],
                        format: {
                          body: function (inner, coldex, rowdex) {
                            if (inner.length <= 0) return inner;

                            // Parse HTML content
                            const parser = new DOMParser();
                            const doc = parser.parseFromString(inner, 'text/html');

                            let text = '';

                            // Handle role-name elements specifically
                            const userNameElements = doc.querySelectorAll('.role-name');
                            if (userNameElements.length > 0) {
                              userNameElements.forEach(el => {
                                // Get text from nested structure - try different selectors
                                const nameText =
                                  el.querySelector('.fw-medium')?.textContent ||
                                  el.querySelector('.d-block')?.textContent ||
                                  el.textContent;
                                text += nameText.trim() + ' ';
                              });
                            } else {
                              // Handle other elements (status, role, etc)
                              text = doc.body.textContent || doc.body.innerText;
                            }

                            return text.trim();
                          }
                        }
                      }
                    }
                  ]
                },
                {
                  text: '<i class="icon-base ti tabler-plus me-0 me-sm-1 icon-16px"></i><span class="d-none d-sm-inline-block">Add New Car</span>',
                  className: `add-new btn btn-primary rounded-2 waves-effect waves-light ${window.permissions?.canCreateCar ? '' : 'd-none'}`,
                  action: function () {
                    window.location.href = `${baseURL}/cars/create`; // Replace with route('drivers.create') if injected
                  },
                  init: function (api, node, config) {
                    $(node).removeClass('btn-secondary'); // ✅ force-remove it
                  }
                }
              ]
            }
          ]
        },
        bottomStart: {
          rowClass: 'row mx-3 justify-content-between',
          features: ['info']
        },
        bottomEnd: 'paging'
      },
      language: {
        paginate: {
          next: '<i class="icon-base ti tabler-chevron-right scaleX-n1-rtl icon-18px"></i>',
          previous: '<i class="icon-base ti tabler-chevron-left scaleX-n1-rtl icon-18px"></i>',
          first: '<i class="icon-base ti tabler-chevrons-left scaleX-n1-rtl icon-18px"></i>',
          last: '<i class="icon-base ti tabler-chevrons-right scaleX-n1-rtl icon-18px"></i>'
        }
      },
      // For responsive popup
      responsive: {
        details: {
          display: DataTable.Responsive.display.modal({
            header: function (row) {
              const data = row.data();
              return 'Details of ' + data['name'];
            }
          }),
          type: 'column',
          renderer: function (api, rowIdx, columns) {
            const data = columns
              .map(function (col) {
                return col.title !== '' // Do not show row in modal popup if title is blank (for check box)
                  ? `<tr data-dt-row="${col.rowIndex}" data-dt-column="${col.columnIndex}">
                      <td>${col.title}:</td>
                      <td>${col.data}</td>
                    </tr>`
                  : '';
              })
              .join('');

            if (data) {
              const div = document.createElement('div');
              div.classList.add('table-responsive');
              const table = document.createElement('table');
              div.appendChild(table);
              table.classList.add('table');
              const tbody = document.createElement('tbody');
              tbody.innerHTML = data;
              table.appendChild(tbody);
              return div;
            }
            return false;
          }
        }
      }
    });

    // Refresh table on filter change
    $('#filter_parent, #filter_subcategory, #filter_brand, #filter_model').on('change', function () {
      dt_model.draw();
      loadCarModels();
    });

    // Fixed header
    if (window.Helpers.isNavbarFixed()) {
      const navHeight = document.getElementById('layout-navbar').offsetHeight;
      new DataTable.FixedHeader(dt_model).headerOffset(navHeight);
    } else {
      new DataTable.FixedHeader(dt_model);
    }
    // Load subcategories when parent changes
    $('#filter_parent').on('change', function () {
      const parentId = $(this).val();
      $('#filter_subcategory').html('<option value="">All</option>');

      if (parentId) {
        fetch(`${baseURL}/vehicle-categories/${parentId}/subcategories`)
          .then(res => res.json())
          .then(data => {
            data.forEach(item => {
              $('#filter_subcategory').append(`<option value="${item.id}">${item.name}</option>`);
            });
          });
      }
    });

    function loadCarModels() {
      const brandId = $('#filter_brand').val();
      const categoryId = $('#filter_subcategory').val();
      const parentCategoryId = $('#filter_parent').val();

      if (!brandId || !categoryId || !parentCategoryId) return;

      $.ajax({
        url: `${baseURL}/vehicle-models-dropdown`, // <-- your route URL
        method: 'GET',
        data: {
          car_brand_id: brandId,
          car_category_id: categoryId,
          car_parent_category_id: parentCategoryId
        },
        success: function (response) {
          $('#filter_model').empty().append('<option value="">Select Model</option>');
          response.data.forEach(model => {
            $('#filter_model').append(`<option value="${model.id}">${model.name}</option>`);
          });
        },
        error: function (xhr) {
          console.error(xhr.responseText);
        }
      });
    }

    /*     // Filter by parent category dropdown for serverSide==false
    $('#filter_parent').on('change', function () {
      const value = this.value;
      table.column(3).search(value).draw(); // column index 1 = parent_category
    });

    // Filter by subcategory
    $('#filter_subcategory').on('change', function () {
      const value = this.value;
      table.column(4).search(value).draw(); // column index 2 = subcategory
    });

    // Filter by brand
    $('#filter_brand').on('change', function () {
      const value = this.value;
      table.column(5).search(value).draw(); // column index 3 = brand
    });
 */

    // Delete Record
    document.addEventListener('click', function (e) {
      const deleteBtn = e.target.closest('.delete-record');
      if (!deleteBtn) return;

      const modelId = deleteBtn.dataset.id;

      Swal.fire({
        title: 'Are you sure?',
        text: 'This model will be deleted!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, delete it!',
        customClass: {
          confirmButton: 'btn btn-primary me-3',
          cancelButton: 'btn btn-label-secondary'
        },
        buttonsStyling: false
      }).then(function (result) {
        if (result.value) {
          fetch(`${baseURL}/car/${modelId}`, {
            method: 'DELETE',
            headers: {
              'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
              'Content-Type': 'application/json'
            }
          })
            .then(async response => {
              const data = await response.json();
              if (response.ok) {
                dt_model.row(`#trRow-${modelId}`).remove().draw(false);
                Swal.fire({
                  icon: 'success',
                  title: 'Deleted!',
                  text: data.message || 'Car has been deleted.',
                  customClass: {
                    confirmButton: 'btn btn-success'
                  }
                });
              } else {
                throw new Error('Delete failed');
              }
            })
            .catch(error => console.error(error));
        }
      });
    });
  }
});
