/**
 * App Kanban
 */

'use strict';

(async function () {
  let boards;
  const kanbanSidebar = document.querySelector('.kanban-update-item-sidebar'),
    kanbanWrapper = document.querySelector('.kanban-wrapper'),
    commentEditor = document.querySelector('.comment-editor'),
    kanbanAddNewBoard = document.querySelector('.kanban-add-new-board'),
    kanbanAddNewInput = [].slice.call(document.querySelectorAll('.kanban-add-board-input')),
    kanbanAddBoardBtn = document.querySelector('.kanban-add-board-btn'),
    datePicker = document.querySelector('#due-date'),
    select2 = $('.select2'), // ! Using jquery vars due to select2 jQuery dependency
    assetsPath = document.querySelector('html').getAttribute('data-assets-path');
  const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
  // Init kanban Offcanvas
  const kanbanOffcanvas = new bootstrap.Offcanvas(kanbanSidebar);

  // Get kanban data
  const kanbanResponse = await fetch(`${baseURL}/kanban/data`, {
    headers: {
      Accept: 'application/json'
    }
  });
  boards = await kanbanResponse.json();
  //console.log(boards);
  // datepicker init
  if (datePicker) {
    datePicker.flatpickr({
      monthSelectorType: 'static',
      static: true,
      altInput: true,
      altFormat: 'j F, Y',
      dateFormat: 'Y-m-d'
    });
  }

  //! TODO: Update Event label and guest code to JS once select removes jQuery dependency
  // select2
  if (select2.length) {
    function renderLabels(option) {
      if (!option.id) {
        return option.text;
      }
      var $badge = "<div class='badge " + $(option.element).data('color') + "'> " + option.text + '</div>';
      return $badge;
    }

    select2.each(function () {
      var $this = $(this);
      $this.wrap("<div class='position-relative'></div>").select2({
        placeholder: 'Select Label',
        dropdownParent: $this.parent(),
        templateResult: renderLabels,
        templateSelection: renderLabels,
        escapeMarkup: function (es) {
          return es;
        }
      });
    });
  }

  // Comment editor
  if (commentEditor) {
    new Quill(commentEditor, {
      modules: {
        toolbar: '.comment-toolbar'
      },
      placeholder: 'Write a Comment...',
      theme: 'snow'
    });
  }

  // Render board dropdown
  const renderBoardDropdown = (boardId = '') => {
    const canDelete = window.permissions?.canDeletekanban;
    const canEdit = window.permissions?.canEditkanban;
    if (!canDelete && !canEdit) return ''; // Nothing to show
    return `
    <div class="dropdown">
      <i class="dropdown-toggle icon-base ti tabler-dots-vertical cursor-pointer"
         id="board-dropdown-${boardId}"
         data-bs-toggle="dropdown"
         aria-haspopup="true"
         aria-expanded="false"></i>
      <div class="dropdown-menu dropdown-menu-end" aria-labelledby="board-dropdown-${boardId}">
        ${
          canDelete
            ? `
          <a class="dropdown-item delete-board" href="javascript:void(0)">
            <i class="icon-base ti tabler-trash icon-xs"></i>
            <span class="align-middle">Delete</span>
          </a>
        `
            : ''
        }
        ${
          canEdit
            ? `
          <a class="dropdown-item" href="javascript:void(0)">
            <i class="icon-base ti tabler-edit icon-xs"></i>
            <span class="align-middle">Rename</span>
          </a>
        `
            : ''
        }
        <a class="dropdown-item d-none" href="javascript:void(0)">
          <i class="icon-base ti tabler-archive icon-xs"></i>
          <span class="align-middle">Archive</span>
        </a>
      </div>
    </div>
  `;
  };

  // Render item dropdown
  const renderDropdown = (taskId = '') => {
    const canDelete = window.permissions?.canDeletekanban;
    const canEdit = window.permissions?.canEditkanban;

    if (!canDelete && !canEdit) return ''; // Hide dropdown if no permission

    return `
    <div class="dropdown kanban-tasks-item-dropdown">
      <i class="dropdown-toggle icon-base ti tabler-dots-vertical"
         id="kanban-tasks-item-dropdown-${taskId}"
         data-bs-toggle="dropdown"
         aria-haspopup="true"
         aria-expanded="false"></i>
      <div class="dropdown-menu dropdown-menu-end"
           aria-labelledby="kanban-tasks-item-dropdown-${taskId}">
        ${
          canEdit
            ? `
          <a class="dropdown-item d-none" href="javascript:void(0)">Copy task link</a>
          <a class="dropdown-item d-none" href="javascript:void(0)">Duplicate task</a>
        `
            : ''
        }
        ${
          canDelete
            ? `
          <a class="dropdown-item delete-task" href="javascript:void(0)">
            <i class="icon-base ti tabler-trash icon-xs"></i>
            <span class="align-middle">Delete</span>
          </a>
        `
            : ''
        }
      </div>
    </div>
  `;
  };

  // Render header
  const renderHeader = (color, text, taskId = '') => `
  <div class="d-flex justify-content-between flex-wrap align-items-center mb-2">
    <div class="item-badges">
      <div class="badge bg-label-${color}">${text}</div>
    </div>
    ${renderDropdown(taskId)}
  </div>
`;

  // Render avatar
  const renderAvatar = (images = '', pullUp = false, size = '', margin = '', members = '') => {
    const transitionClass = pullUp ? ' pull-up' : '';
    const sizeClass = size ? `avatar-${size}` : '';
    const memberList = members ? members.split(',') : [];

    return images
      ? images
          .split(',')
          .map((img, index, arr) => {
            const marginClass = margin && index !== arr.length - 1 ? ` me-${margin}` : '';
            const memberName = memberList[index] || '';
            return `
            <div class="avatar ${sizeClass}${marginClass} w-px-26 h-px-26"
                 data-bs-toggle="tooltip"
                 data-bs-placement="top"
                 title="${memberName}">
                <img src="${assetsPath}img/avatars/${img}"
                     alt="Avatar"
                     class="rounded-circle${transitionClass}">
            </div>
        `;
          })
          .join('')
      : '';
  };

  // Render footer
  const renderFooter = (attachments, comments, assigned, members) => `
<div class="d-flex justify-content-between align-items-center flex-wrap mt-2">
    <div class="d-flex">
        <span class="d-flex align-items-center me-2">
            <i class="icon-base ti tabler-paperclip me-1"></i>
            <span class="attachments">${attachments}</span>
        </span>
        <span class="d-flex align-items-center ms-2">
            <i class="icon-base ti tabler-message-2 me-1"></i>
            <span>${comments}</span>
        </span>
    </div>
    <div class="avatar-group d-flex align-items-center assigned-avatar">
        ${renderAvatar(assigned, true, 'xs', null, members)}
    </div>
</div>
`;

  let currentTaskId = null;
  // Initialize kanban
  const kanban = new jKanban({
    element: '.kanban-wrapper',
    gutter: '12px',
    widthBoard: '250px',
    dragItems: true,
    boards: boards,
    dragBoards: true,
    addItemButton: true,
    buttonContent: '+ Add Item',
    itemAddOptions: {
      enabled: window.permissions?.canCreatekanban ? true : false,
      content: '+ Add New Item',
      class: 'kanban-title-button btn btn-default border-none',
      footer: false
    },
    click: el => {
      const element = el;
      currentTaskId = el.getAttribute('data-eid') || el.dataset.eid;
      const title = element.getAttribute('data-eid')
        ? element.querySelector('.kanban-text').textContent
        : element.textContent;
      const date = element.getAttribute('data-due-date');
      const dateObj = new Date();
      const year = dateObj.getFullYear();
      const dateToUse = date
        ? `${date}, ${year}`
        : `${dateObj.getDate()} ${dateObj.toLocaleString('en', { month: 'long' })}, ${year}`;
      const label = element.getAttribute('data-badge-text');
      const avatars = element.getAttribute('data-assigned');

      // Show kanban offcanvas
      kanbanOffcanvas.show();

      // Populate sidebar fields
      kanbanSidebar.querySelector('#title').value = title;
      kanbanSidebar.querySelector('#due-date').nextSibling.value = dateToUse;

      // Using jQuery for select2
      $('.kanban-update-item-sidebar').find(select2).val(label).trigger('change');

      // Remove and update assigned avatars
      kanbanSidebar.querySelector('.assigned').innerHTML = '';
      kanbanSidebar.querySelector('.assigned').insertAdjacentHTML(
        'afterbegin',
        `${renderAvatar(avatars, false, 'xs', '1', el.getAttribute('data-members'))}
        <div class="avatar avatar-xs ms-1">
            <span class="avatar-initial rounded-circle bg-label-secondary">
                <i class="icon-base ti tabler-plus icon-xs text-heading"></i>
            </span>
        </div>`
      );
    },

    buttonClick: (el, boardId) => {
      const addNewForm = document.createElement('form');
      addNewForm.setAttribute('class', 'new-item-form');
      addNewForm.innerHTML = `
        <div class="mb-4">
            <textarea class="form-control add-new-item" rows="2" placeholder="Add Content" autofocus required></textarea>
        </div>
        <div class="mb-4">
            <button type="submit" class="btn btn-primary btn-sm me-3 waves-effect waves-light">Add</button>
            <button type="button" class="btn btn-label-secondary btn-sm cancel-add-item waves-effect waves-light">Cancel</button>
        </div>
      `;

      kanban.addForm(boardId, addNewForm);
      // Remove form on clicking cancel button
      addNewForm.querySelector('.cancel-add-item').addEventListener('click', () => addNewForm.remove());
      addNewForm.addEventListener('submit', async e => {
        e.preventDefault();
        const currentBoard = Array.from(document.querySelectorAll(`.kanban-board[data-id="${boardId}"] .kanban-item`));
        const title = e.target[0].value.trim();
        if (!title) return;

        try {
          // Send task to backend
          const res = await fetch(`${baseURL}/kanban/task`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-CSRF-TOKEN': csrfToken,
              Accept: 'application/json'
            },
            body: JSON.stringify({
              title: title,
              board_id: boardId
            })
          });

          if (!res.ok) throw new Error(await res.text());
          const task = await res.json(); // { id: ..., title: ..., etc. }

          // Add to kanban visually
          kanban.addElement(boardId, {
            title: `<span class="kanban-text">${e.target[0].value}</span>`,
            id: `${boardId}-${currentBoard.length + 1}`
          });

          // Add dropdown to new tasks
          const kanbanTextElements = Array.from(
            document.querySelectorAll(`.kanban-board[data-id="${boardId}"] .kanban-text`)
          );
          kanbanTextElements.forEach(textElem => {
            textElem.insertAdjacentHTML('beforebegin', renderDropdown());
          });

          // Prevent sidebar from opening on dropdown click
          const newTaskDropdowns = Array.from(document.querySelectorAll('.kanban-item .kanban-tasks-item-dropdown'));
          newTaskDropdowns.forEach(dropdown => {
            dropdown.addEventListener('click', event => event.stopPropagation());
          });

          // Prevent sidebar open on dropdown click
          const dropdown = newTaskElem.querySelector('.kanban-tasks-item-dropdown');
          if (dropdown) {
            dropdown.addEventListener('click', event => event.stopPropagation());
          }

          // Add delete functionality for new tasks
          const deleteTaskButtons = Array.from(
            document.querySelectorAll(`.kanban-board[data-id="${boardId}"] .delete-task`)
          );
          deleteTaskButtons.forEach(btn => {
            btn.addEventListener('click', async () => {
              const taskId = btn.closest('.kanban-item').getAttribute('data-eid');
              await fetch(`${baseURL}/kanban/task/${task.id}`, {
                method: 'DELETE',
                headers: { 'X-CSRF-TOKEN': csrfToken }
              });
              kanban.removeElement(taskId);
            });
          });

          // Remove input form
          addNewForm.remove();
        } catch (err) {
          console.error('Error saving kanban task:', err);
        }
      });
    }
  });

  // Kanban Wrapper scrollbar
  if (kanbanWrapper) {
    new PerfectScrollbar(kanbanWrapper);
  }

  const kanbanContainer = document.querySelector('.kanban-container');
  const kanbanTitleBoard = Array.from(document.querySelectorAll('.kanban-title-board'));
  const kanbanItem = Array.from(document.querySelectorAll('.kanban-item'));

  // Render custom items
  if (kanbanItem.length) {
    kanbanItem.forEach(el => {
      const element = `<span class="kanban-text">${el.textContent}</span>`;
      let img = '';

      if (el.getAttribute('data-image')) {
        img = `
              <img class="img-fluid rounded mb-2"
                   src="${el.getAttribute('data-image')}">
          `;
      }

      el.textContent = '';

      if (el.getAttribute('data-badge') && el.getAttribute('data-badge-text')) {
        el.insertAdjacentHTML(
          'afterbegin',
          `${renderHeader(el.getAttribute('data-badge'), el.getAttribute('data-badge-text'))}${img}${element}`
        );
      }

      if (el.getAttribute('data-comments') || el.getAttribute('data-due-date') || el.getAttribute('data-assigned')) {
        el.insertAdjacentHTML(
          'beforeend',
          renderFooter(
            el.getAttribute('data-attachments') || 0,
            el.getAttribute('data-comments') || 0,
            el.getAttribute('data-assigned') || '',
            el.getAttribute('data-members') || ''
          )
        );
      }
    });
  }

  // Initialize tooltips for rendered items
  const tooltipTriggerList = Array.from(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
  tooltipTriggerList.forEach(tooltipTriggerEl => {
    new bootstrap.Tooltip(tooltipTriggerEl);
  });

  // Prevent sidebar from opening on dropdown button click
  const tasksItemDropdown = Array.from(document.querySelectorAll('.kanban-tasks-item-dropdown'));
  if (tasksItemDropdown.length) {
    tasksItemDropdown.forEach(dropdown => {
      dropdown.addEventListener('click', event => {
        event.stopPropagation();
      });
    });
  }

  // Toggle "add new" input and actions for add-new-btn
  if (kanbanAddBoardBtn) {
    kanbanAddBoardBtn.addEventListener('click', () => {
      kanbanAddNewInput.forEach(el => {
        el.value = ''; // Clear input value
        el.classList.toggle('d-none'); // Toggle visibility
      });
    });
  }

  // Render "add new" inline with boards
  if (kanbanContainer) {
    kanbanContainer.append(kanbanAddNewBoard);
  }

  // Makes kanban title editable for rendered boards
  if (kanbanTitleBoard) {
    kanbanTitleBoard.forEach(elem => {
      elem.addEventListener('mouseenter', () => {
        elem.contentEditable = 'true';
      });

      // Appends delete icon with title
      elem.insertAdjacentHTML('afterend', renderBoardDropdown());
    });
  }

  // Delete Board for rendered boards
  const deleteBoards = Array.from(document.querySelectorAll('.delete-board'));
  deleteBoards.forEach(elem => {
    elem.addEventListener('click', async e => {
      const id = elem.closest('.kanban-board').getAttribute('data-id');
      await fetch(`${baseURL}/kanban/board/${id}`, {
        method: 'DELETE',
        headers: { 'X-CSRF-TOKEN': csrfToken }
      });
      kanban.removeBoard(id);
    });
  });

  // Delete task for rendered boards
  const deleteTasks = Array.from(document.querySelectorAll('.delete-task'));
  deleteTasks.forEach(task => {
    task.addEventListener('click', async e => {
      const id = task.closest('.kanban-item').getAttribute('data-eid');
      await fetch(`${baseURL}/kanban/task/${id}`, {
        method: 'DELETE',
        headers: { 'X-CSRF-TOKEN': csrfToken }
      });
      kanban.removeElement(id);
    });
  });

  // Cancel "Add New Board" input
  const cancelAddNew = document.querySelector('.kanban-add-board-cancel-btn');
  if (cancelAddNew) {
    cancelAddNew.addEventListener('click', () => {
      kanbanAddNewInput.forEach(el => {
        el.classList.toggle('d-none');
      });
    });
  }

  // Add new board
  if (kanbanAddNewBoard) {
    kanbanAddNewBoard.addEventListener('submit', e => {
      e.preventDefault();
      const title = e.target.querySelector('.form-control').value.trim();
      const id = title.replace(/\s+/g, '-').toLowerCase();
      fetch(`${baseURL}/kanban/board`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify({ id, title })
      })
        .then(res => res.json())
        .then(res => {
          if (res.status === 200) {
            kanban.addBoards([{ id, title }]);
            window.showToast({
              title: 'Kanban ✉️',
              message: res.message,
              type: 'bg-success',
              animation: 'animate__fadeInUp',
              time: 'Just now',
              delay: 3000
            });
          }
        })
        .catch(err => {
          console.error('Kanban board add error:', err);
        });

      // Add delete board option to new board and make title editable
      const newBoard = document.querySelector('.kanban-board:last-child');
      if (newBoard) {
        const header = newBoard.querySelector('.kanban-title-board');
        header.insertAdjacentHTML('afterend', renderBoardDropdown());

        // Make title editable
        header.addEventListener('mouseenter', () => {
          header.contentEditable = 'true';
        });

        // Add delete functionality to new board
        const deleteNewBoard = newBoard.querySelector('.delete-board');
        if (deleteNewBoard) {
          deleteNewBoard.addEventListener('click', async e => {
            const id = deleteNewBoard.closest('.kanban-board').getAttribute('data-id');
            await fetch(`${baseURL}/kanban/board/${id}`, {
              method: 'DELETE',
              headers: { 'X-CSRF-TOKEN': csrfToken }
            });
            kanban.removeBoard(id);
          });
        }
      }

      // Hide input fields
      kanbanAddNewInput.forEach(el => {
        el.classList.add('d-none');
      });

      // Re-append the "Add New Board" form
      if (kanbanContainer) {
        kanbanContainer.append(kanbanAddNewBoard);
      }
    });
  }

  // Clear comment editor on Kanban sidebar close
  kanbanSidebar.addEventListener('hidden.bs.offcanvas', () => {
    const editor = kanbanSidebar.querySelector('.ql-editor').firstElementChild;
    if (editor) editor.innerHTML = '';
  });

  // Re-init tooltip when offcanvas opens(Bootstrap bug)
  if (kanbanSidebar) {
    kanbanSidebar.addEventListener('shown.bs.offcanvas', () => {
      const tooltipTriggerList = Array.from(kanbanSidebar.querySelectorAll('[data-bs-toggle="tooltip"]'));
      tooltipTriggerList.forEach(tooltipTriggerEl => {
        new bootstrap.Tooltip(tooltipTriggerEl);
      });
    });
  }

  document.querySelector('.btn-update').addEventListener('click', async () => {
    const taskId = currentTaskId?.replace('task-', '') || null; // assume `currentTaskId` is global
    if (!taskId) {
      alert('Task ID missing.');
      return;
    }

    const formData = new FormData();
    formData.append('title', $('#title').val());
    formData.append('due_date', $('#due-date').val());
    formData.append('label', $('#label').val());
    const labelSelect = document.querySelector('#label');
    const selectedOption = labelSelect.options[labelSelect.selectedIndex];
    formData.append('label_color', selectedOption.getAttribute('data-color'));
    formData.append('comments', document.querySelector('.ql-editor')?.innerHTML || '');

    const attachmentInput = document.querySelector('#attachments');
    const file = attachmentInput.files[0]; // ⬅️ Only one file
    if (file) {
      formData.append('attachment', file); // ⬅️ Not an array
    }

    try {
      const res = await fetch(`${baseURL}/kanban/task/${taskId}`, {
        method: 'POST',
        headers: {
          'X-CSRF-TOKEN': csrfToken,
          Accept: 'application/json'
        },
        body: formData
      });

      if (!res.ok) {
        const err = await res.json();
        throw new Error(err.message || 'Update failed');
      }

      const result = await res.json();
      window.showToast({
        title: 'Kanban 📝',
        message: result.message || 'Task updated',
        type: 'bg-success',
        animation: 'animate__fadeInUp',
        time: 'Just now',
        delay: 3000
      });
      location.reload(); // or update in-place
    } catch (err) {
      console.error('Update failed:', err);
      alert(`Error updating task: ${err.message}`);
    }
  });
})();
