<?php

namespace App\Traits;

use UnitEnum;

trait HasEnumsTrait
{
  /**
   * Boot the trait
   * Cast enum attributes safely on model retrieval
   */
  public static function bootHasEnumsTrait()
  {
    static::retrieved(function ($model) {
      foreach ($model->getEnumAttributes() as $attribute => $enumClass) {
        $value = $model->getAttribute($attribute);

        // Skip null or empty values
        if ($value === null || $value === '') {
          continue;
        }

        // Skip if already cast
        if ($value instanceof $enumClass) {
          continue;
        }

        // Safely cast
        try {
          $model->setAttribute($attribute, $enumClass::from($value));
        } catch (\ValueError $e) {
          // Ignore invalid DB values
        }
      }
    });
  }

  /**
   * Return enum attributes defined in the model
   */
  public function getEnumAttributes(): array
  {
    return property_exists($this, 'enums') ? $this->enums : [];
  }

  /**
   * Override setAttribute to allow assigning enum instances
   */
  public function setAttribute($key, $value)
  {
    $enums = $this->getEnumAttributes();

    if (isset($enums[$key])) {
      $enumClass = $enums[$key];

      if ($value instanceof $enumClass) {
        $value = $value->value;
      }
    }

    return parent::setAttribute($key, $value);
  }

  /**
   * Compare an attribute with enum instance or raw value
   * Usage: $user->enumIs('user_type', UserType::ADMIN) OR $user->enumIs('user_type', 'Admin')
   */
  public function enumIs(string $attribute, $value): bool
  {
    $current = $this->getAttribute($attribute);

    // If current is enum instance, compare to value's value if enum
    if ($current instanceof UnitEnum) {
      if ($value instanceof UnitEnum) {
        return $current === $value; // strict comparison
      }
      return $current->value === $value;
    }

    // current is string/int
    if ($value instanceof UnitEnum) {
      return $current === $value->value;
    }

    return $current === $value;
  }
}
