<?php
namespace App\Services;
use App\Models\Subscription;
use App\Models\AuditLog;
use App\Models\Transaction;
use App\Models\Package;
use App\Models\User;
use Carbon\Carbon;
use UnexpectedValueException;
use Illuminate\Support\Facades\Log;
class SubscriptionService
{

  public function audit(Subscription $subscription, string $action, array $meta = []): void
  {
    AuditLog::create([
      'user_id' => auth()->id() ?: $subscription->user_id,
      'action' => $action,
      'meta' => $meta,
    ]);
  }

  public function transactions(Subscription $sub, $invoice)
  {

    try {
      $check = Transaction::where('txn_id', $invoice->id)->first();
      if ($check) {
        return;
      }

      $package = Package::find($sub->package_id);
      $user = User::find($sub->user_id);

      // Initialize defaults
      $periodStart = now();
      $periodEnd = now()->addMonth();
      $unitAmount = 0;
      $quantity = 1;
      $totalAmount = 0;

      // Determine if it's an Invoice or Subscription object
      if (isset($invoice->lines)) {
        // 👉 INVOICE object
        $item = $invoice->lines->data[0] ?? null;
        if ($item && isset($item->period->start) && isset($item->period->end)) {
          $periodStart = Carbon::createFromTimestamp($item->period->start)->format('Y-m-d');
          $periodEnd = Carbon::createFromTimestamp($item->period->end)->format('Y-m-d');
        }

        $unitAmount = $item->price->unit_amount ?? 0;
        $quantity = $item->quantity ?? 1;
        $totalAmount = ($unitAmount * $quantity) / 100;

      } elseif (isset($invoice->current_period_start) && isset($invoice->current_period_end)) {
        // 👉 SUBSCRIPTION object
        $periodStart = Carbon::createFromTimestamp($invoice->current_period_start)->format('Y-m-d');
        $periodEnd = Carbon::createFromTimestamp($invoice->current_period_end)->format('Y-m-d');

        $item = $invoice->items->data[0] ?? null;
        $unitAmount = $item->price->unit_amount ?? 0;
        $quantity = $item->quantity ?? 1;
        $totalAmount = ($unitAmount * $quantity) / 100;
      }
      $package->transactions()->create([
        'user_id' => $user->id ?? null,
        'subscription_id' => $sub->id,
        'amount' => !empty($invoice->amount_paid) ? $invoice->amount_paid / 100 : $totalAmount,
        'sub_amount' => !empty($invoice->subtotal) ? $invoice->subtotal / 100 : $totalAmount,
        'tax' => isset($invoice->tax) ? ($invoice->tax / 100) : 0,
        'discount' => optional($invoice->discount?->coupon)->amount_off / 100 ?? 0,
        'txn_id' => $invoice->id,
        'txn_details' => json_encode($invoice->toArray()),
        'contact_details' => json_encode([
          'email' => $invoice->customer_email ?? $user->email,
          'phone' => $user->phone ?? '',
        ]),
        'package_details' => json_encode($package),
        'payment_mode' => 'stripe',
        'status' => $invoice->status === 'active' ? 'paid' : ($invoice->status ?? 'paid'),
        'start_date' => $periodStart,
        'end_date' => $periodEnd,
        'seen' => false,
      ]);

      // Update user billing period
      $user->update([
        'start_date' => $periodStart,
        'end_date' => $periodEnd,
      ]);
    } catch (UnexpectedValueException $e) {
      Log::info('[Transaction] Service Error: ' . $e->getMessage());
      Log::info('[Transaction] Service Invoice Data: ' . $invoice);
      return;
    }
  }
}
