<?php
namespace App\Services;
use App\Models\RidePricingRule;
use App\Models\Ride;
use App\Models\Driver;
use Carbon\Carbon;
use App\Models\DeliveryCharge;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\Http;

class RidePriceService
{
  protected $googleApiKey;
  public function __construct()
  {
    $this->googleApiKey = config('services.google.maps_key'); // store in config/services.php
  }
  public function calculateRidePrice($basePrice, $pickupTime, $pickupLat, $pickupLng)
  {
    $date = Carbon::parse($pickupTime);
    $rules = RidePricingRule::with('conditions') // eager load conditions
      ->where('is_active', 1)
      ->orderByDesc('priority')
      ->get();
    $matchedRule = null;
    foreach ($rules as $rule) {
      if ($this->isRuleMatched($rule, $date)) {
        $matchedRule = $rule;
        break; // stop at highest priority match
      }
    }
    $context = [
      'weather' => $this->getWeatherCondition($pickupLat, $pickupLng),
      'lat' => $pickupLat,  //area
      'lng' => $pickupLng,  //area
      'demand_level' => $this->getDemandLevel($pickupLat, $pickupLng, $date),
    ];
    //\Illuminate\Support\Facades\Log::info('Ride context data:', $context);
    if ($matchedRule) {
      return $this->applyRule($basePrice, $matchedRule, $context);
    }
    return $this->applyRule($basePrice, (object) ['price_type' => 'flat', 'amount' => 0], $context);
  }


  private function isRuleMatched($rule, $date)
  {
    return collect($rule->conditions)->every(function ($condition) use ($date) {
      $valuesArray = json_decode($condition->condition_value, true);
      if (!is_array($valuesArray) || empty($valuesArray)) {
        return false; // invalid or empty => fail
      }

      switch ($condition->condition_type) {
        case 'month':
          return in_array($date->month, $valuesArray);

        case 'day_of_week':
          $dow = strtolower($date->format('l'));
          $valuesLower = array_map('strtolower', $valuesArray);
          return in_array($dow, $valuesLower);

        case 'date':
          return in_array($date->toDateString(), $valuesArray);

        case 'time_range':
          // must match at least one range
          return collect($valuesArray)->contains(function ($range) use ($date) {
            [$start, $end] = explode(',', $range);
            $startTime = $date->copy()->setTimeFromTimeString($start);
            $endTime = $date->copy()->setTimeFromTimeString($end);
            return $date->between($startTime, $endTime);
          });

        default:
          return false; // unknown type => fail
      }
    });
  }

  private function applyRule($price, $rule, $context = [])
  {
    if ($rule->price_type == 'multiplier') {
      return $price * $rule->amount;
    } elseif ($rule->price_type == 'flat') {
      return $price + $rule->amount;
    } elseif ($rule->price_type == 'discount') {
      return $price - ($price * ($rule->amount / 100));
    } elseif ($rule->price_type == 'percent') {
      return $price + ($price * ($rule->amount / 100));
    }

    // Surge pricing logic
    if (!empty($context['weather']) && $context['weather'] == 'bad') {
      // Increase price for bad weather
      $badWeather = Helpers::setting('bad_weather', 'site_setting');
      $price *= $badWeather;
    }

    if (!empty($context['lat']) && !empty($context['lng'])) {
      if ($this->busyAreas($context['lat'], $context['lng'])) {
        $busyAreaMultiplier = Helpers::setting('busy_areas', 'site_setting');
        $price *= $busyAreaMultiplier;
      }
    }

    if (!empty($context['demand_level'])) {
      // Example: demand_level can be 'high', 'medium', 'low'
      switch ($context['demand_level']) {
        case 'high':
          $demandLevelHigh = Helpers::setting('demand_level_high', 'site_setting');
          $price *= $demandLevelHigh;
          break;
        case 'medium':
          $demandLevelMedium = Helpers::setting('demand_level_medium', 'site_setting');
          $price *= $demandLevelMedium;
          break;
        case 'low':
        default:
          // no change
          break;
      }
    }
    return round($price, 2); // round to 2 decimals
  }
  private function busyAreas($lat, $lng, $radius = 5, $minutes = 60, $threshold = 30)
  {
    $since = now()->subMinutes($minutes);
    $haversine = "(6371 * acos(cos(radians($lat))
                  * cos(radians(pickup_lat))
                  * cos(radians(pickup_lng) - radians($lng))
                  + sin(radians($lat))
                  * sin(radians(pickup_lat))))";
    $count = Ride::where('created_at', '>=', $since)
      ->whereRaw("$haversine <= ?", [$radius])
      ->count();

    return $count >= $threshold;
  }

  private function getWeatherCondition($lat, $lng)
  {
    $cacheKey = "weather:" . round($lat, 2) . ":" . round($lng, 2);

    return cacheRemember($cacheKey, 60, function () use ($lat, $lng) {
      $apiKey = config('services.openweather.maps_key');
      $response = Http::get("https://api.openweathermap.org/data/2.5/weather", [
        'lat' => $lat,
        'lon' => $lng,
        'appid' => $apiKey,
        'units' => 'metric',
      ]);

      if ($response->ok()) {
        $main = strtolower($response['weather'][0]['main']);
        return in_array($main, ['rain', 'snow', 'storm']) ? 'bad' : 'good';
      }
      return 'good';
    });
  }


  private function getDemandLevel($lat, $lng, $date, $radius = 3)
  {
    $since = $date->copy()->subMinutes(30);

    $haversine = "(6371 * acos(cos(radians($lat))
                  * cos(radians(pickup_lat))
                  * cos(radians(pickup_lng) - radians($lng))
                  + sin(radians($lat))
                  * sin(radians(pickup_lat))))";
    $ridesCount = Ride::whereBetween('created_at', [$since, $date])
      ->whereRaw("$haversine <= ?", [$radius])
      ->count();

    // You can optionally get active driver count in same radius
    $haversineDriver = "(6371 * acos(cos(radians($lat))
                  * cos(radians(latitude))
                  * cos(radians(longitude) - radians($lng))
                  + sin(radians($lat))
                  * sin(radians(latitude))))";
    $driversCount = Driver::where('is_active', 1)
      ->whereRaw("$haversineDriver <= ?", [$radius])
      ->count();

    // Basic ratio
    $ratio = $driversCount > 0 ? $ridesCount / $driversCount : $ridesCount;
    if ($ratio > 3)
      return 'high';      // 3x more demand than supply
    if ($ratio > 1.5)
      return 'medium';
    return 'low';
  }

  /**
   * Calculate distance between two coordinates (in km)
   */

  public function getDistanceAndTime($pickup_lat = 31.3501, $pickup_lng = 75.58795, $drop_lat = 31.3501, $drop_lng = 75.58795)
  {
    try {
      $url = "https://maps.googleapis.com/maps/api/distancematrix/json";
      $response = Http::get($url, [
        'origins' => "$pickup_lat,$pickup_lng",
        'destinations' => "$drop_lat,$drop_lng",
        'key' => $this->googleApiKey,
        'mode' => 'driving',
      ]);
      $data = $response->json();
      if (
        isset($data['rows'][0]['elements'][0]['status']) &&
        $data['rows'][0]['elements'][0]['status'] === 'OK'
      ) {
        $element = $data['rows'][0]['elements'][0];
        $distanceKm = $element['distance']['value'] / 1000; // meters → km
        $timeMin = $element['duration']['value'] / 60; // seconds → minutes
        return [
          'distance' => round($distanceKm, 2) . 'Km',
          'time' => round($timeMin) . 'Minutes',
        ];
      }
    } catch (\Exception $e) {
      // log error
    }
    // Fallback to static method
    $distance = Helpers::haversine((float) $pickup_lat, (float) $pickup_lng, (float) $drop_lat, (float) $drop_lng);
    $time = $this->estimateTime($distance);
    return [
      'distance' => round($distance, 2),  //dont add km and minutes here, iwill gave error
      'time' => $time,
    ];
  }

  public function estimateTime($distance)
  {
    //static method
    if ($distance < 10) {
      return round($distance / 40 * 60);
    } elseif ($distance < 20) {
      return round($distance / 60 * 60);
    } else {
      return round($distance / 80 * 60);
    }
  }

  public function estimatePrice($baseFare, $perKm, $perMinute, $rideDistance, $estimated_time, $pickupat, $pickupLng)
  {
    $pickupTime = date('Y-m-d H:i:s');
    $distance_price = $perKm * $rideDistance;
    $time_price = $perMinute * $estimated_time;
    $basePrice = $distance_price + $time_price;
    $basePrice = $baseFare > $basePrice ? $baseFare : $basePrice;
    return $this->calculateRidePrice($basePrice, $pickupTime, $pickupat, $pickupLng);
  }

  public function estimateDeliveryCharge($countryId, $rideDistance)
  {
    $estimatePrice = DeliveryCharge::where('country_id', $countryId)
      ->where('from_km', '<=', $rideDistance)
      ->where('to_km', '>=', $rideDistance)
      ->value('amount') ?? 0;
    if (empty($deliveryCharge)) {
      // fallback: pick the highest `to_km` slab
      $estimatePrice = DeliveryCharge::where('country_id', $countryId)
        ->orderByDesc('to_km')
        ->value('amount') ?? 0;
    }
    return $estimatePrice;
  }



}
