<?php
namespace App\Services;
use App\Models\Store;
use App\Models\Coupon;
use App\Models\Cart;
use App\Models\Ride;
use App\Models\Order;
use App\Models\CartSummary;
use App\Models\CouponUsage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Traits\JsonResponseTrait;
class CouponService
{
  use JsonResponseTrait;

  public function store(Request $request)
  {
    if (Coupon::where('code', $request->coupon_code)->exists()) {
      return $this->successResponse([], __('locale.Coupon already exists for this code.'));
    }
    $coupon = new Coupon();
    return $this->saveCoupon($coupon, $request);
  }
  public function update(Request $request, $id)
  {
    $coupon = Coupon::findOrFail($id);
    return $this->saveCoupon($coupon, $request);
  }
  protected function saveCoupon(Coupon $coupon, Request $request)
  {
    $range = $this->parseDateRange($request->date);
    $coupon->type = $request->coupon_type;
    $coupon->code = $request->coupon_code;
    $coupon->discount = $request->discount;
    $coupon->discount_type = $request->coupon_type === 'register_base' ? 'points' : $request->discount_type;
    $coupon->start_date = $range['start'];
    $coupon->end_date = $range['end'];
    $coupon->details = json_encode($this->prepareCouponDetails($request));
    if ($coupon->save()) {
      return $this->successResponse([], __('locale.Coupon Uploaded Successfully'));
    }
    return $this->errorResponse(__('locale.Something went wrong'), 400);
  }
  protected function prepareCouponDetails(Request $request): array
  {
    $usageType = $request->usage_type ?? 'unlimited';
    $maxDiscount = $request->max_discount ?? null;
    $minBuy = $request->min_buy ?? null;

    // Start with base details
    $details = [
      [
        'usage_type' => $usageType,
        'maximum_usage' => $usageType === 'limited' ? ($request->maximum_usage ?? 0) : null,
        'min_buy' => $minBuy,
        'max_discount' => $maxDiscount,
      ],
    ];

    switch ($request->coupon_type) {

      case 'product_base':
        foreach ($request->category_ids ?? [] as $i => $categoryId) {
          if (!empty($request->product_ids[$i])) {
            $details[] = [
              'category_id' => $categoryId,
              'subcategory_id' => $request->subcategory_ids[$i] ?? null,
              'product_id' => $request->product_ids[$i] ?? null,
            ];
          }
        }
        break;

      case 'category_base':
        foreach ($request->category_ids ?? [] as $i => $categoryId) {
          $details[] = [
            'category_id' => $categoryId,
            'subcategory_id' => $request->subcategory_ids[$i] ?? null,
          ];
        }
        break;

      case 'cart_base':
      case 'customer_base':
      case 'seller_base':
      case 'ride_base':
      case 'offer_base':
        $customers = is_array($request->customers) ? $request->customers : [];

        if (!empty($customers)) {
          // Add separate entry for each customer
          foreach ($customers as $customerId) {
            $details[] = [
              'user_id' => $customerId,
              'min_buy' => $minBuy,
              'max_discount' => $maxDiscount,
            ];
          }
        } else {
          // Merge min_buy and max_discount into first array
          $details[0] = array_merge($details[0], [
          ]);
        }
        break;

      case 'register_base':
        $details[] = [
          'remarks' => $request->remarks ?? null
        ];
        break;
    }

    // Remove null values from each detail array
    return array_map(fn($d) => array_filter($d, fn($v) => !is_null($v)), $details);
  }

  public function applyCoupon(Request $request)
  {
    $userId = auth()->id();
    $today = now()->toDateString();
    if (!empty($request->coupon_id) && intval($request->coupon_id) > 0) {
      $coupon = $this->applyCouponQueryId(intval($request->coupon_id), $today, $userId);
    } else {
      $coupon = $this->applyCouponQuery($request->coupon, $today, $userId);
    }
    if (empty($coupon)) {
      return $this->errorResponse(__('locale.Coupon code not valid. Please check the code and try again.'), 400);
    }
    // Decode details
    $details = json_decode($coupon->details, true) ?? [];
    $couponDetail = $this->getCouponDetailsForUser($coupon, $details, $request);
    // Validate usage
    try {
      $this->validateUsage($coupon, $couponDetail, $userId);
    } catch (\Exception $e) {
      return $this->errorResponse($e->getMessage(), 400);
    }
    return $this->processCouponApplication($coupon, $couponDetail, $request, $userId);
  }
  protected function getCouponDetailsForUser(Coupon $coupon, array $details, Request $request): array
  {
    $userId = auth()->id();
    return match ($coupon->type) {
      'customer_base' => collect($details)->firstWhere('user_id', $userId) ?? [],
      'seller_base' => collect($details)->firstWhere('user_id', Store::find($request->store_id)?->user_id) ?? [],
      'product_base', 'category_base' => $details,
      default => $details[0] ?? [],
    };
  }
  protected function validateUsage(Coupon $coupon, array $couponDetail, int $userId)
  {
    // accept both possible keys if old code used usageType
    $usageType = $couponDetail[0]['usage_type'] ?? $couponDetail['usageType'] ?? 'unlimited';
    $maximumUsage = $couponDetail[0]['maximum_usage'] ?? null;
    if ($usageType === 'limited') {
      $totalUsed = CouponUsage::where('coupon_id', $coupon->id)->count();
      if (!is_null($maximumUsage) && intval($totalUsed) >= intval($maximumUsage)) {
        throw new \Exception(__('locale.Coupon usage limit reached'));
      }
    }
    if ($usageType === 'per_user') {
      $used = CouponUsage::where('user_id', $userId)
        ->where('coupon_id', $coupon->id)
        ->exists();
      if ($used) {
        throw new \Exception(__('locale.This Coupon code is already applied by you.'));
      }
    }
  }
  protected function processCouponApplication(Coupon $coupon, array $couponDetail, Request $request, int $userId)
  {
    // Preserve offer_base branching exactly as original
    if ($coupon->type === 'offer_base') {
      $isFirst = false;
      if (($request->type ?? null) === 'ride_base') {
        $isFirst = Ride::where('user_id', $userId)->doesntExist();
        if ($isFirst) {
          return $this->processRideCoupon($coupon, $couponDetail, $request);
        } else {
          return $this->errorResponse(__('locale.Coupon only valid on first ride'), 400);
        }
      } else {
        $isFirst = Order::where('user_id', $userId)->doesntExist();
        if ($isFirst) {
          return $this->processCartCoupon($coupon, $couponDetail, $userId);
        } else {
          return $this->errorResponse(__('locale.Coupon only valid on first order'), 400);
        }
      }
    }
    if ($coupon->type === 'ride_base' && (($request->type ?? null) === 'ride_base')) {
      return $this->processRideCoupon($coupon, $couponDetail, $request);
    }
    // Other types:
    return match ($coupon->type) {
      'cart_base' => $this->processCartCoupon($coupon, $couponDetail, $userId),
      'customer_base' => $this->processCartCoupon($coupon, $couponDetail, $userId),
      'seller_base' => $this->processCartCoupon($coupon, $couponDetail, $userId),
      'product_base' => $this->processProductCoupon($coupon, $couponDetail, $userId),
      'category_base' => $this->processCategoryCoupon($coupon, $couponDetail, $userId),
      default => $this->errorResponse(__('locale.Invalid coupon type'), 400),
    };
  }
  protected function processRideCoupon(Coupon $coupon, array $couponDetail, Request $request)
  {
    $vehicles = $request->vehicles ?? [];
    if (is_string($vehicles)) {
      $vehicles = json_decode($vehicles, true) ?: [];
    }
    if (!is_array($vehicles)) {
      $vehicles = [];
    }
    $minBuy = $couponDetail['min_buy'] ?? null;
    $maxDiscount = $couponDetail['max_discount'] ?? null;
    $processed = collect($vehicles)->map(function ($v) use ($coupon, $minBuy, $maxDiscount) {
      $fare = (float) ($v['estimate_fare'] ?? 0);
      // below min buy => do not apply coupon (preserve original behaviour)
      if ($minBuy && $fare < $minBuy) {
        $v['estimate_fare'] = number_format($fare, 2, '.', '');
        $v['estimate_strike_fare'] = '';
        $v['coupon_id'] = null;
        return $v;
      }
      $discount = $this->calculateDiscount($fare, $coupon, $maxDiscount);
      $final = max(0, $fare - $discount);
      $v['estimate_fare'] = number_format($final, 2, '.', '');
      $v['estimate_strike_fare'] = $fare;
      $v['coupon_id'] = $coupon->id;
      return $v;
    });
    return $this->successResponse([
      'vehicles' => $processed,
      'coupon_id' => $coupon->id,
      'coupon_code' => $coupon->code,
    ], __('locale.Coupon code applied successfully'));
  }
  protected function processCartCoupon(Coupon $coupon, array $couponDetail, int $userId)
  {

    $sum = Cart::where('user_id', $userId)
      ->select(DB::raw('SUM(quantity * price) as total'))
      ->value('total');
    $minBuy = $couponDetail['min_buy'] ?? null;
    $maxDiscount = $couponDetail['max_discount'] ?? null;
    if ($minBuy && $sum < $minBuy) {
      return $this->errorResponse(__('locale.This Coupon code not valid for this amount. Minimum amount is :') . $minBuy, 400);
    }

    $discount = $this->calculateDiscount($sum, $coupon, $maxDiscount);
    $amount = max(0, $sum - $discount);
    $summary = CartSummary::where('user_id', $userId)->firstOrFail();
    $summary->update([
      'coupon_id' => $coupon->id,
      'coupon_amount' => $discount,
      'total' => $summary->subtotal + $summary->delivery_charge - $discount,
    ]);

    return $this->successResponse([
      'coupon_cost' => number_format($discount, 2, '.', ''),
      'amount_cost' => number_format($amount, 2, '.', ''),
      'coupon_id' => $coupon->id,
      'coupon_code' => $coupon->code,
    ], __('locale.Coupon code applied successfully'));
  }
  protected function processProductCoupon(Coupon $coupon, array $couponDetails, int $userId)
  {
    $cartItems = Cart::where('user_id', $userId)->get();
    $discount = $sum = 0;
    $minBuy = $couponDetails[0]['min_buy'] ?? null;
    $maxDiscount = $couponDetails[0]['max_discount'] ?? null;
    foreach ($cartItems as $item) {
      $lineTotal = $item->quantity * $item->price;
      $sum += $lineTotal;
      foreach ($couponDetails as $detail) {
        if (($detail['product_id'] ?? null) == $item->product_id) {
          $discount += $this->calculateDiscount($lineTotal, $coupon, $maxDiscount);
        }
      }
    }
    if ($minBuy && $sum < $minBuy) {
      return $this->errorResponse(__('locale.This Coupon code not valid for this amount. Minimum amount is :') . $minBuy, 400);
    }
    $amount = max(0, $sum - $discount);
    $summary = CartSummary::where('user_id', $userId)->firstOrFail();
    $summary->update([
      'coupon_id' => $discount > 0 ? $coupon->id : null,
      'coupon_amount' => $discount,
      'total' => $summary->subtotal + $summary->delivery_charge - $discount,
    ]);
    return $this->successResponse([
      'coupon_cost' => number_format($discount, 2, '.', ''),
      'amount_cost' => number_format($amount, 2, '.', ''),
      'coupon_id' => $coupon->id,
      'coupon_code' => $coupon->code,
    ], $discount > 0 ? __('locale.Coupon code applied successfully') : __('locale.Invalid coupon type'));
  }
  protected function processCategoryCoupon(Coupon $coupon, array $couponDetails, int $userId)
  {
    $cartItems = Cart::with('product.category')->where('user_id', $userId)->get();
    $discount = $sum = 0;
    $minBuy = $couponDetails[0]['min_buy'] ?? null;
    $maxDiscount = $couponDetails[0]['max_discount'] ?? null;
    foreach ($cartItems as $item) {
      $lineTotal = $item->quantity * $item->price;
      $sum += $lineTotal;
      foreach ($couponDetails as $detail) {
        $match = false;
        if (!empty($detail['subcategory_id'])) {
          // Match subcategory directly
          $match = ($detail['subcategory_id'] == $item->product->category_id);
        } elseif (!empty($detail['category_id'])) {
          // Match all subcategories of a category (check parent_id)
          $match = ($item->product->category->parent_id == $detail['category_id']);
        }
        if ($match) {
          $discount += $this->calculateDiscount($lineTotal, $coupon, $maxDiscount);
        }
      }
    }
    if ($minBuy && $sum < $minBuy) {
      return $this->errorResponse(__('locale.This Coupon code not valid for this amount. Minimum amount is :') . $minBuy, 400);
    }
    $amount = max(0, $sum - $discount);

    $summary = CartSummary::where('user_id', $userId)->firstOrFail();
    $summary->update([
      'coupon_id' => $discount > 0 ? $coupon->id : null,
      'coupon_amount' => $discount,
      'total' => $summary->subtotal + $summary->delivery_charge - $discount,
    ]);

    return $this->successResponse([
      'coupon_cost' => number_format($discount, 2, '.', ''),
      'amount_cost' => number_format($amount, 2, '.', ''),
      'coupon_id' => $coupon->id,
      'coupon_code' => $coupon->code,
    ], $discount > 0 ? __('locale.Coupon code applied successfully') : __('locale.Invalid coupon type'));
  }
  protected function calculateDiscount(float $amount, Coupon $coupon, ?float $maxDiscount = null): float
  {
    $discount = $coupon->discount_type === 'percent'
      ? ($amount * $coupon->discount / 100)
      : $coupon->discount;

    if ($maxDiscount && $discount > $maxDiscount) {
      $discount = $maxDiscount;
    }
    return max(0, $discount);
  }
  private function applyCouponQuery($code, $today, $userId = null)
  {
    return Coupon::where('code', $code)
      ->whereDate('start_date', '<=', $today)
      ->whereDate('end_date', '>=', $today)
      ->where(function ($query) use ($userId) {
        $query->where(function ($q) use ($userId) {
          $q->where('type', 'customer_base')
            ->where(function ($json) use ($userId) {
              $json->whereJsonContains('details->user_id', $userId)
                ->orWhere('details', 'LIKE', '%"user_id":"' . $userId . '"%');
            });
        })->orWhere('type', '!=', 'customer_base');
      })
      ->first();
  }
  private function applyCouponQueryId($id, $today, $userId = null)
  {
    return Coupon::where('id', $id)
      ->whereDate('start_date', '<=', $today)
      ->whereDate('end_date', '>=', $today)
      ->where(function ($query) use ($userId) {
        $query->where(function ($q) use ($userId) {
          $q->where('type', 'customer_base')
            ->where(function ($json) use ($userId) {
              $json->whereJsonContains('details->user_id', $userId)
                ->orWhere('details', 'LIKE', '%"user_id":"' . $userId . '"%');
            });
        })->orWhere('type', '!=', 'customer_base');
      })
      ->first();
  }
  public function removeCoupon(Request $request)
  {
    $userId = auth()->id();
    $coupon = Coupon::findOrFail($request->coupon_id);
    $cartItems = Cart::where('user_id', $userId)->get();
    $summary = CartSummary::where('user_id', $userId)->firstOrFail();
    $discount = 0;
    $sum = $cartItems->sum(fn($item) => $item->quantity * $item->price);
    // Ride-mode removal (ride_base or offer_base when request->type == 'ride_base')
    if (in_array($coupon->type, ['ride_base', 'offer_base']) && ($request->type ?? null) === 'ride_base') {
      $vehicles = $request->vehicles ?? [];
      if (is_string($vehicles)) {
        $vehicles = json_decode($vehicles, true) ?: [];
      }
      if (!is_array($vehicles)) {
        $vehicles = [];
      }
      $processedVehicles = collect($vehicles)->map(function ($vehicle) {
        $fare = (float) ($vehicle['estimate_fare'] ?? 0);
        $vehicle['estimate_fare'] = number_format($fare, 2, '.', '');
        $vehicle['estimate_strike_fare'] = $fare;
        $vehicle['coupon_id'] = null;
        return $vehicle;
      });
      return $this->successResponse([
        'vehicles' => $processedVehicles,
        'coupon_id' => null,
        'coupon_code' => null,
      ], __('locale.Coupon code removed successfully'));
    }

    // Cart-like removal for all other cases
    $summary->update([
      'coupon_id' => null,
      'coupon_amount' => null,
      'total' => $summary->subtotal + $summary->delivery_charge - $discount,
    ]);

    return $this->successResponse([
      'coupon_cost' => null,
      'amount_cost' => number_format(max(0, $sum - $discount), 2, '.', ''),
      'coupon_id' => null,
      'coupon_code' => null,
    ], __('locale.Coupon code removed successfully'));
  }
  private function parseDateRange($dateRange): array
  {
    if (empty($dateRange) || !is_string($dateRange)) {
      $now = now()->toDateString();
      return ['start' => $now, 'end' => $now];
    }
    $parts = explode('to', $dateRange);
    $start = isset($parts[0]) ? trim($parts[0]) : now()->toDateString();
    $end = isset($parts[1]) ? trim($parts[1]) : $start;
    return ['start' => $start, 'end' => $end];
  }
}
