<?php

namespace App\Repositories\Eloquent;
use App\Models\WithdrawalRequest;
use App\Models\User;
use App\Models\Store;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Helpers;
use App\Repositories\BaseRepository;
use App\Enums\UserType;
use App\Services\WalletService;
use App\Traits\JsonResponseTrait;
use App\Repositories\Contracts\WithdrawalRequestRepositoryInterface;
class WithdrawalRequestRepository extends BaseRepository implements WithdrawalRequestRepositoryInterface
{
  use JsonResponseTrait;
  public function __construct(WithdrawalRequest $withdrawalRequestModel)
  {
    parent::__construct($withdrawalRequestModel);
  }

  public function allList($request)
  {
    $user = User::find(auth()->id());
    $userType = $user->user_type;
    $storeId = $userType == UserType::STORE && $user->storeId ? $user->storeId : '';
    $canDelete = Auth::user()->user_type == UserType::STORE ? false : Auth::user()->can('wallet-delete');
    $canEdit = Auth::user()->user_type == UserType::STORE ? false : Auth::user()->can('wallet-edit');
    $canCreate = Auth::user()->user_type == UserType::STORE ? true : Auth::user()->can('wallet-create');
    $builder = WithdrawalRequest::query();
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where('txn_id', 'LIKE', "%{$search}%");
    }

    if (!is_null($request->input('status'))) {
      $query->where('status', $request->input('status'));
    }
    if (!is_null($request->input('user_type'))) {
      $query->where('user_type', $request->input('user_type'));
    }
    if (!is_null($request->input('customer'))) {
      $query->where('user_id', (int) $request->input('customer'));
    }
    // 📅 Date filter
    if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
      $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
      $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
      $query->whereBetween('created_at', [$startDate, $endDate]);
    }
    if (!empty($storeId)) {
      $query->where('user_id', (int) $user->id);
    }
    return \DataTables::eloquent($query->with(['user']))
      ->addIndexColumn()
      ->addColumn('customer', fn($withdrawalRequest) => $withdrawalRequest['user']?->name ?? '-')
      ->addColumn('email', fn($withdrawalRequest) => $withdrawalRequest['user']?->email ?? '-')
      ->addColumn('avatar', function ($withdrawalRequest) {
        if ($withdrawalRequest['user']->profile_photo_path) {
          return asset('storage/' . $withdrawalRequest['user']->profile_photo_path);
        }
      })
      ->addColumn('userType', $userType)
      ->addColumn('currency', fn($withdrawalRequest) => $withdrawalRequest->currency ?? $currency)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('delete-url', fn($withdrawalRequest) => url($prefix . 'withdrawal_requests/' . $withdrawalRequest->id))
      ->addColumn('status-url', fn() => url($prefix . 'withdrawal_requests/changestatus'))

      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }


  public function store(Request $request, $Id = null)
  {
    $user = User::find(auth()->id());
    $wallets = new WalletService;
    $userType = strtolower($user->user_type);
    $ownerId = $userType == 'driver' ? $user->driverId : ($userType == 'store' ? $user->storeId : $user->id);
    $wallet = $wallets->getOrCreate($userType, $ownerId);
    if ($wallet->balance < $request->amount) {
      return $this->errorResponse(__('locale.Withdrawal Request Amount should be less than or equal to wallet amount.'), 400);
    }
    $withdrawalRequest = new WithdrawalRequest;
    $withdrawalRequest->user_id = $user->id;
    $withdrawalRequest->user_type = $user->user_type;
    $withdrawalRequest->amount = $request->amount;
    $withdrawalRequest->total_amount = $request->amount;
    $withdrawalRequest->status = 'Pending';
    $withdrawalRequest->currency = $user->currency;
    $withdrawalRequest->bank_details = json_encode([
      'email' => $user->email ?? '',
      'phone' => $user->phone ?? '',
      'payment_method' => $user->bank_details,
    ]);
    $withdrawalRequest->save();
    $wallets->withdrawalRequest($user, $request->amount, 'decrement');
    $wallets->walletTransaction($user, $withdrawalRequest->total_amount, $withdrawalRequest, 'debit');
    cacheForget("dropdown_withdrawal_user", ['dropdown']);
    $wallet = $wallets->getOrCreate($userType, $ownerId);
    return $this->successResponse($wallet, __('locale.Withdrawal Request Created Successfully!'));
  }

  public function changeStatus(Request $request)
  {
    $wallets = new WalletService;
    $withdrawalRequest = WithdrawalRequest::find($request->id);
    $currency = $withdrawalRequest->currency ?? cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $user = User::find($withdrawalRequest->user_id);
    $store = Store::where('user_id', $withdrawalRequest->user_id)->first();
    if ($request->status == 'Approved') {
      $withdrawalRequest->processing_fees = $request->tran_fee;
      $withdrawalRequest->amount = $withdrawalRequest->total_amount - $request->tran_fee;
      $withdrawalRequest->txn_id = $request->txn_id;
      $withdrawalRequest->details = $request->txn_detail;
      $withdrawalRequest->processed_at = now();
      $data = json_decode($withdrawalRequest->bank_details);
      $message = 'Your withdrawal request is approved by admin and ' . $currency . $withdrawalRequest->amount . ' has been transferred to your account via ' . @$data->payment_method ?? $request->txn_detail;
    } else {
      $store->wallet = $store->wallet + $withdrawalRequest->total_amount;
      $store->save();
      $withdrawalRequest->processed_at = now();
      $withdrawalRequest->notes = $request->notes;
      $message = 'Rejected by admin and amount is reverted back to your dashboard wallet';
      $wallets->withdrawalRequest($user, $withdrawalRequest->total_amount, 'increment');
      $wallets->walletTransaction($user, $withdrawalRequest->total_amount, $withdrawalRequest, 'credit');
    }
    $withdrawalRequest->status = $request->status;
    $withdrawalRequest->save();
    Helpers::notifications('Payment Withdrawal Request', 'App\Notifications\WithdrawalRequest', $message, $user->id, $withdrawalRequest->id, 'withdrawal_requests', 'tabler-credit-card-pay');
    return $request->id;
  }

  public function myList($request)
  {
    $user = User::find(auth()->id());
    $userType = $user->user_type;
    $builder = WithdrawalRequest::with(['user'])->where('user_id', $user->id)->paginate(10);
    return $builder;
  }

}
