<?php
namespace App\Repositories\Eloquent;
use App\Models\VehicleCategory;
use App\Models\Country;
use App\Models\RidePricingRule;
use App\Repositories\BaseRepository;
use App\Repositories\Contracts\VehicleCategoryRepositoryInterface;
class VehicleCategoryRepository extends BaseRepository implements VehicleCategoryRepositoryInterface
{
  public function __construct(VehicleCategory $vehicleCategoryModel)
  {
    parent::__construct($vehicleCategoryModel);
  }
  public function allList($type)
  {
    return VehicleCategory::withCount('countryPrices')->when(!empty($type), function ($query) {
      $query->whereNotNull('parent_id');
    }, function ($query) {
      $query->whereNull('parent_id');
    })
      ->orderByDesc('created_at')
      ->get()
      ->map(function ($item) {
        $item->has_prices = $item->countryPrices_count > 0;
        return $item;
      });
  }

  public function allCategories()
  {
    //remove  whereNull('parent_id') for multiple level categories
    $categories = VehicleCategory::whereNull('parent_id')->get();
    $countries = Country::where('status', 1)->get();
    return [
      'categories' => $categories,
      'countries' => $countries
    ];
  }
  function buildIndentedCategoryList($categories, $prefix = '')
  {
    $list = [];
    foreach ($categories as $category) {
      $list[$category->id] = $prefix . $category->name;
      if ($category->children) {
        $list += $this->buildIndentedCategoryList($category->children, $prefix . '— ');
      }
    }
    return $list;
  }

  public function dropdown()
  {
    return VehicleCategory::where('status', 1)->whereNull('parent_id')->orderBy('name')->get();
    // return $this->buildIndentedCategoryList(VehicleCategory::with('children')->whereNull('parent_id')->get());
  }

  public function store(array $data)
  {
    return VehicleCategory::create([
      'parent_id' => $data['parent_id'] ?? null,
      'name' => $data['name'],
      'description' => $data['description'] ?? '',
      'image' => $data['image'] ?? '',
      'map_icon' => $data['map_icon'] ?? '',
      'status' => 1,
      'type' => isset($data['type'])
        ? (is_array($data['type']) ? implode(',', $data['type']) : $data['type'])
        : null,
    ]);
  }

  public function updateCategory($id, array $data)
  {
    $category = VehicleCategory::findOrFail($id);
    if (isset($data['type'])) {
      $data['type'] = is_array($data['type'])
        ? implode(',', $data['type'])
        : $data['type'];
    }
    $category->update($data);
    return $category;
  }

  public function subCategories($id, $type = null)
  {
    $query = VehicleCategory::where('parent_id', $id);
    if (!empty($type)) {
      $query->whereRaw("FIND_IN_SET(?, type)", [$type]);
    }
    return $query->get();
  }

  public function getPricesByCategory($categoryId)
  {
    $category = VehicleCategory::with('countryPrices')->findOrFail($categoryId);
    $data = $category->countryPrices->mapWithKeys(function ($country) {
      return [
        $country->id => [
          'base_fare' => $country->pivot->base_fare,
          'per_km_rate' => $country->pivot->per_km_rate,
          'per_minute_price' => $country->pivot->per_minute_price,
          'per_minute_waiting_price' => $country->pivot->per_minute_waiting_price,
        ]
      ];
    });
    return $data;
  }
  public function getPricesRuleByCategory($categoryId)
  {
    $data = array();
    $category = VehicleCategory::with('countryPrices')->findOrFail($categoryId);
    $countryIds = $category->countryPrices->pluck('id')->toArray();
    $rules = RidePricingRule::with('conditions')
      ->whereIn('country_id', $countryIds)
      ->get();
    $countries = [];

    foreach ($category->countryPrices as $country) {
      $baseFare = $country->pivot->base_fare;
      $perKm = $country->pivot->per_km_rate;
      $perMin = $country->pivot->per_minute_price;

      $countryRules = $rules->where('country_id', $country->id)->map(function ($rule) use ($baseFare, $perKm, $perMin) {
        // Calculate new prices
        $newBaseFare = $baseFare;
        $newPerKm = $perKm;
        $newPerMin = $perMin;

        switch ($rule->price_type) {
          case 'flat':
            $newBaseFare += $rule->amount;
            $newPerKm += $rule->amount;
            $newPerMin += $rule->amount;
            break;

          case 'multiplier':
            $newBaseFare *= $rule->amount;
            $newPerKm *= $rule->amount;
            $newPerMin *= $rule->amount;
            break;

          case 'discount':
            $percent = $rule->amount / 100;
            $newBaseFare -= $newBaseFare * $percent;
            $newPerKm -= $newPerKm * $percent;
            $newPerMin -= $newPerMin * $percent;
            break;

          case 'percent':
            $percent = $rule->amount / 100;
            $newBaseFare += $newBaseFare * $percent;
            $newPerKm += $newPerKm * $percent;
            $newPerMin += $newPerMin * $percent;
            break;
        }

        // Format conditions
        $rule->conditions->transform(function ($cond) {
          $values = json_decode($cond->condition_value, true) ?? [];
          $valuesArray = is_string($values) ? json_decode($values, true) : $values;

          if ($cond->condition_type === 'month') {
            $cond->formatted_value = collect($valuesArray)
              ->map(fn($m) => \Carbon\Carbon::create()->month((int) $m)->format('F'))
              ->implode(', ');
          } elseif ($cond->condition_type === 'day_of_week') {
            $cond->formatted_value = implode(', ', $valuesArray);
          } elseif ($cond->condition_type === 'time_range') {
            $cond->formatted_value = 'Start:' . $valuesArray['start'] ?? '-' . ',' . ' End:' . $valuesArray['end'] ?? '-';
          } else {
            $cond->formatted_value = implode(', ', $values);
          }

          return $cond;
        });

        return [
          'name' => $rule->rule_name,
          'price_type' => $rule->price_type,
          'amount' => $rule->amount,
          'new_prices' => [
            'base_fare' => $newBaseFare,
            'per_km' => $newPerKm,
            'per_minute' => $newPerMin,
          ],
          'conditions' => $rule->conditions,
        ];
      });

      $countries[] = [
        'name' => $country->name,
        'base_fare' => $baseFare,
        'per_km' => $perKm,
        'per_minute' => $perMin,
        'rules' => $countryRules,
      ];
    }
    $data['category'] = $category;
    $data['countries'] = $countries;
    return $data;
  }
  public function syncCountryPrices(int $categoryId, array $prices): void
  {
    $category = VehicleCategory::findOrFail($categoryId);
    $syncData = [];
    foreach ($prices as $price) {
      $syncData[$price['country_id']] = [
        'base_fare' => $price['base_fare'] ?? 0,
        'per_km_rate' => $price['per_km_rate'] ?? 0,
        'per_minute_price' => $price['per_minute_price'] ?? 0,
        'per_minute_waiting_price' => $price['per_minute_waiting_price'] ?? 0,
      ];
    }
    $category->countryPrices()->sync($syncData);
  }

}
