<?php

namespace App\Repositories\Eloquent;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Helpers;
use Carbon\Carbon;
use App\Repositories\BaseRepository;
use App\Enums\UserType;
use App\Repositories\Contracts\TransactionRepositoryInterface;
class TransactionRepository extends BaseRepository implements TransactionRepositoryInterface
{
  public function __construct(Transaction $transactionModel)
  {
    parent::__construct($transactionModel);
  }

  public function allList($request)
  {
    $user = User::find(auth()->id());
    $userType = $user->user_type;
    $storeId = $userType == UserType::STORE && $user->storeId ? $user->storeId : '';
    $canDelete = Auth::user()->user_type == UserType::STORE ? false : Auth::user()->can('transaction-delete');
    $canEdit = Auth::user()->user_type == UserType::STORE ? false : Auth::user()->can('transaction-edit');
    $canCreate = Auth::user()->user_type == UserType::STORE ? true : Auth::user()->can('transaction-create');
    $builder = Transaction::query();
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where('txn_id', 'LIKE', "%{$search}%");
    }

    if (!is_null($request->input('status'))) {
      $query->where('status', $request->input('status'));
    }
    if (!is_null($request->input('transactionable_type'))) {
      $query->where('transactionable_type', $request->input('transactionable_type'));
    }
    if (!is_null($request->input('customer'))) {
      $query->where('user_id', (int) $request->input('customer'));
    }
    if (!empty($storeId)) {
      $query->where('user_id', (int) $user->id);
    }
    // 📅 Date filter
    if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
      $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
      $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
      $query->whereBetween('created_at', [$startDate, $endDate]);
    }
    return \DataTables::eloquent($query->with(['user']))
      ->addIndexColumn()
      ->addColumn('customer', fn($transaction) => $transaction['user']?->name ?? '-')
      ->addColumn('email', fn($transaction) => $transaction['user']?->email ?? '-')
      ->addColumn('userType', fn($transaction) => $transaction['user']?->user_type ?? '-')
      ->addColumn('avatar', function ($transaction) {
        if ($transaction['user']?->profile_photo_path) {
          return asset('storage/' . $transaction['user']->profile_photo_path);
        }
      })
      ->addColumn('transactionable_type', function ($transaction) {
        return class_basename($transaction->transactionable_type);
      })
      ->addColumn('user_type', $userType)
      ->addColumn('currency', fn($transaction) => $transaction->currency ?? $currency)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('delete-url', fn($transaction) => url($prefix . 'transactions/' . $transaction->id))
      ->addColumn('status-url', fn() => url($prefix . 'transactions/changestatus'))
      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }

  public function store(array $data = [])
  {
    $transaction = Transaction::create($data);

    return $transaction;
  }

}
