<?php
namespace App\Repositories\Eloquent;
use Illuminate\Http\Request;
use App\Models\RideRequest;
use App\Models\Driver;
use App\Models\RideTemp;
use App\Helpers\Helpers;
use App\Events\NewRideRequestEvent;
use App\Jobs\RideRequestTimeoutJob;
use Illuminate\Support\Facades\DB;
use App\Repositories\Eloquent\RideRepository;
use App\Repositories\Contracts\RideRequestRepositoryInterface;
class RideRequestRepository implements RideRequestRepositoryInterface
{
  protected ?RideRepository $rideRepository = null;

  protected function rideRepository(): RideRepository
  {
    return $this->rideRepository ??= app(RideRepository::class);
  }


  public function findTemp($id)
  {
    return RideTemp::with('user')->find($id);
  }

  public function requestRide(Request $request)
  {
    $user = auth()->user();
    $countryId = $user->current_country_id ?? 94;
    $currency = $user->currency ?? cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $details = [
      'estimate_strike_fare' => $request->estimate_strike_fare,
      'coupon_id' => $request->coupon_id,
      'payment_type' => $request->payment_type,
      'payment_id' => $request->payment_id,
      'country_id' => $countryId,
    ];
    $temp = RideTemp::create([
      'user_id' => auth()->id(),
      'vehicle_category_id' => $request->vehicle_id,
      'pickup_location' => $request->pickup_location,
      'drop_location' => $request->drop_location,
      'pickup_lat' => $request->pickup_lat,
      'pickup_lng' => $request->pickup_lng,
      'drop_lat' => $request->drop_lat,
      'drop_lng' => $request->drop_lng,
      'trip_distance' => $request->trip_distance,
      'trip_time' => $request->trip_time,
      'sender_name' => $request->sender_name ?? '',
      'receiver_name' => $request->receiver_name ?? '',
      'receiver_phone' => $request->receiver_phone ?? '',
      'parcel_category_id' => $request->parcel_category_id ?? null,
      'package_details' => $request->package_details ?? '',
      'order_id' => $request->order_id ?? 0,
      'type' => $request->type ?? 'Ride',
      'fare' => $request->estimate_fare,
      'currency' => $currency,
      'details' => !empty($details) ? json_encode($details) : '',
    ]);
    $driverLimit = Helpers::setting('driver_request_simultaneously', 'site_setting');
    $nearestDrivers = $this->rideRepository()->findNearestDrivers($request->pickup_lat, $request->pickup_lng, $driverLimit, $request->vehicle_id, $countryId, []);
    $this->createForDrivers($temp, $nearestDrivers);
    return $temp;
  }
  public function cancelTempRide(int $tempId): bool
  {
    return DB::transaction(function () use ($tempId) {
      $tempRide = RideTemp::findOrFail($tempId);
      $tempRide->update(['status' => 'Cancelled']);

      RideRequest::where('ride_temp_id', $tempRide->id)
        ->where('status', 'Pending')
        ->update(['status' => 'Timeout']);

      return true;
    });
  }
  public function createForDrivers(RideTemp $temp, array $driverIds)
  {
    $timeout = Helpers::setting('driver_response_timeout_seconds', 'site_setting');
    foreach ($driverIds as $driverId) {
      $expiry = now()->addSeconds((int) $timeout);
      $rideRequest = RideRequest::create([
        'user_id' => $temp->user_id,
        'driver_id' => $driverId,
        'ride_temp_id' => $temp->id,
        'status' => 'Pending',
        'expires_at' => $expiry,
      ]);
      // Dispatch timeout job for each driver
      RideRequestTimeoutJob::dispatch($rideRequest->id)->delay(now()->addSeconds((int) $timeout));
      broadcast(new NewRideRequestEvent($temp, $driverId, (int) $timeout))->toOthers();
      $this->sendFCM($driverId, $temp); // push notification
    }
  }

  protected function sendFCM($driverId, RideTemp $temp)
  {
    $driver = Driver::with('user')->find($driverId);
    if ($driver && $driver['user'] && $driver['user']->fcm_token) {
      Helpers::fcm_notifications('New Ride Request', 'Pickup at: ' . $temp->pickup_location, $driver['user']->fcm_token, ['type' => 'temp_rides', 'type_id' => $temp?->id], $image = null);
    }
  }

  public function rideRequest()
  {
    $driver = Driver::where('user_id', auth()->id())->first();
    $rideRequest = RideRequest::where('driver_id', $driver?->id)->where('status', 'Pending')->first();
    if (empty($rideRequest)) {
      return $rideRequest;
    } else {
      $rideTemp = RideTemp::with('user')->find($rideRequest->ride_temp_id);
      // Calculate remaining seconds
      $timeout = 0;
      if ($rideRequest->expires_at) {
        $timeout = (int) max(0, now()->diffInSeconds($rideRequest->expires_at, false));
      }
      return [
        'ride_temp_id' => $rideTemp->id,
        'fare' => $rideTemp->currency . $rideTemp->fare,
        'trip_distance' => ($rideTemp->trip_distance ?? '') . ' Km',
        'trip_time' => ($rideTemp->trip_time ?? '') . ' Minutes',
        'timer' => $timeout,  // ⏱ Remaining time left
        'pickup' => $rideTemp->pickup_location,
        'drop' => $rideTemp->drop_location,
        'pickup_lat' => $rideTemp->pickup_lat,
        'pickup_lng' => $rideTemp->pickup_lng,
        'drop_lat' => $rideTemp->drop_lat,
        'drop_lng' => $rideTemp->drop_lng,
        // 👇 Add user details here
        'user' => [
          'name' => $rideTemp->user?->name,
          'image' => $rideTemp->user?->profile_photo_url,
          'rating' => $rideTemp->user?->rating,  // assuming 'rating' column exists in users table
        ],
      ];
    }

  }

}
