<?php

namespace App\Repositories\Eloquent;
use App\Models\Review;
use App\Models\User;
use App\Models\Ride;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Helpers\Helpers;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Repositories\BaseRepository;
use App\Repositories\Contracts\ReviewRepositoryInterface;
class ReviewRepository extends BaseRepository implements ReviewRepositoryInterface
{
  public function __construct(Review $reviewModel)
  {
    parent::__construct($reviewModel);
  }
  public function allReview($userId = null, $type = null)
  {
    if ($type == 'driver' || $type == 'store' || $type == 'user') {
      $user = !empty($userId) ? User::find($userId) : User::find(auth()->id());
      $userType = $user->user_type;
      if ($userType == 'Store') {
        $Id = $user->storeId ?? '';
      } elseif ($userType == 'Driver') {
        $Id = $user->driverId ?? '';
      } else {
        $Id = $user->id ?? '';
      }
    } else {
      $userType = $type;
      $Id = $userId;
    }
    $reviewableClass = strtolower($userType);
    // 1. Main review stats
    $reviewStats = Review::where('reviewable_id', $Id)
      ->where('reviewable_type', $reviewableClass)
      ->selectRaw('
            ROUND(AVG(rating), 2) as avg_rating,
            COUNT(*) as total_reviews,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as new_reviews
        ')
      ->first();

    // 2. Star counts (grouped)
    $starCounts = Review::where('reviewable_id', $Id)
      ->where('reviewable_type', $reviewableClass)
      ->select('rating', DB::raw('COUNT(*) as count'))
      ->groupBy('rating')
      ->pluck('count', 'rating')
      ->toArray();

    // 3. Progress percentage
    $totalReviews = $reviewStats->total_reviews ?: 1; // avoid division by zero
    $progress = [];
    for ($i = 5; $i >= 1; $i--) {
      $count = $starCounts[$i] ?? 0;
      $progress[$i] = [
        'count' => $count,
        'percent' => round(($count / $totalReviews) * 100, 2)
      ];
    }

    $weeklyReviewCounts = Review::where('reviewable_id', $Id)
      ->where('reviewable_type', $reviewableClass)
      ->selectRaw('DAYNAME(created_at) as day, COUNT(*) as count')
      ->where('created_at', '>=', now()->subDays(6))
      ->groupBy('day')
      ->pluck('count', 'day')
      ->toArray();

    // Ensure Monday → Sunday order
    $daysOrder = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
    $weeklyReviewCountsOrdered = [];
    foreach ($daysOrder as $day) {
      $weeklyReviewCountsOrdered[] = $weeklyReviewCounts[$day] ?? 0;
    }


    // Example response for blade
    $reviewData = [
      'avg_rating' => $reviewStats->avg_rating ?? 0,
      'total_reviews' => $reviewStats->total_reviews ?? 0,
      'new_reviews' => $reviewStats->new_reviews ?? 0,
      'progress' => $progress,
      'weeklyReviewCounts' => $weeklyReviewCountsOrdered,
    ];
    return $reviewData;

  }

  public function allList($request)
  {

    $user = User::find(auth()->id());
    $userType = $user->user_type;
    $storeId = $userType == 'Store' && $user->storeId ? $user->storeId : '';
    $canDelete = Auth::user()->user_type == 'Store' ? false : Auth::user()->can('review-delete');
    $canEdit = Auth::user()->user_type == 'Store' ? false : Auth::user()->can('review-edit');
    $canCreate = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('review-create');
    $builder = Review::query();
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('id', 'LIKE', "%{$search}%")
          ->orWhere('comment', 'LIKE', "%{$search}%")
          ->orWhere('status', 'LIKE', "%{$search}%")
          ->orWhere('reviewable_type', 'LIKE', "%{$search}%");
      });
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', (int) $request->input('status'));
    }
    if (!is_null($request->input('customer'))) {
      $query->where('user_id', (int) $request->input('customer'));
    }
    /*     if (!empty($storeId)) {
          $query->where('reviewable_id', (int) $storeId);
        } */
    if (!is_null($request->input('reviewable_id'))) {
      $query->where('reviewable_id', (int) $request->input('reviewable_id'));
    }
    if (!is_null($request->input('reviewable_type'))) {
      $reviewableClass = $request->input('reviewable_type');
      $query->where('reviewable_type', $reviewableClass);
    }
    // 📅 Date filter
    if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
      $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
      $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
      $query->whereBetween('created_at', [$startDate, $endDate]);
    }
    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('customer', fn($review) => $review['user']?->name ?? '-')
      ->addColumn('email', fn($review) => $review['user']?->email ?? '-')
      ->addColumn('avatar', function ($review) {
        if ($review['user']->profile_photo_path) {
          return asset('storage/' . $review['user']->profile_photo_path);
        }
      })
      ->addColumn('reviewable_type', function ($review) {
        return class_basename(ucfirst($review->reviewable_type));
      })
      ->addColumn('user_type', $userType)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('delete-url', fn($review) => url($prefix . 'reviews/' . $review->id))
      ->addColumn('status-url', fn() => url($prefix . 'reviews/changestatus'))
      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }
  public function store(Request $request, $id, $type)
  {
    $userId = auth()->id();
    switch ($type) {
      case 'Ride':
        return $this->handleRideReview($id, $userId, $request);
      case 'Store':
        return $this->handleStoreReview($id, $userId, $request);
      case 'Order':
        return $this->handleOrderReview($id, $userId, $request);
      case 'User':
        return $this->handleUserReview($id, $userId, $request);
      default:
        abort(400, 'Invalid type.');
    }
  }

  private function saveReview($reviewableId, $reviewableType, $userId, $rating, $comment = null, $extra = [])
  {
    // Build the search criteria
    $criteria = [
      'reviewable_id' => $reviewableId,
      'reviewable_type' => $reviewableType,
      'user_id' => $userId,
    ];

    // Include comment_id from $extra if exists
    if (isset($extra['comment_id'])) {
      $criteria['comment_id'] = $extra['comment_id'];
      // remove from $extra so it's not duplicated in update values
      unset($extra['comment_id']);
    }
    // Merge rating, comment, and remaining extra fields
    $values = array_merge([
      'rating' => $rating,
      'comment' => $comment ?? '',
    ], $extra);
    return Review::updateOrCreate($criteria, $values);
  }

  private function updateModelRating($reviewModel, $newRating)
  {
    $reviewModel->increment('rating_count');
    $reviewModel->rating = (($reviewModel->rating * ($reviewModel->rating_count - 1)) + $newRating) / $reviewModel->rating_count;
    $reviewModel->save();
  }

  private function handleRideReview($rideId, $userId, $request)
  {
    $ride = Ride::findOrFail($rideId);
    // Ride Review
    $rideReview = $this->saveReview($ride->id, 'ride', $userId, $request->rating, $request->comment, [
      'driver_id' => $ride->driver_id,
      'comment_id' => $rideId,
    ]);
    $ride->rating = $request->rating;
    $ride->save();
    // Driver Review
    $driverReview = $this->saveReview($ride->driver_id, 'driver', $userId, $request->rating, $request->comment, [
      'comment_id' => $rideId,
    ]);
    $this->updateModelRating($ride->driver, $request->rating);
    return [
      'message' => __('locale.Driver review saved successfully.'),
      'review' => $rideReview,
    ];
  }

  private function handleStoreReview($storeId, $userId, $request)
  {
    $review = $this->saveReview($storeId, 'store', $userId, $request->rating, $request->comment, [
      'comment_id' => $storeId,
    ]);

    return [
      'message' => __('locale.Store review saved successfully.'),
      'review' => $review,
    ];
  }

  private function handleOrderReview($orderId, $userId, $request)
  {
    $order = Order::with('items.product')->findOrFail($orderId);

    // 1️⃣ Review for Order
    $orderReview = $this->saveReview($order->id, 'order', $userId, $request->rating, $request->comment, [
      'comment_id' => $orderId,
    ]);
    $order->rating = $request->rating;
    $order->save();

    // 2️⃣ Review for Store
    $storeReview = $this->saveReview($order->store_id, 'store', $userId, $request->rating, $request->comment, [
      'comment_id' => $orderId,
    ]);
    $this->updateModelRating($order->store, $request->rating);

    // 3️⃣ Product reviews (expects inputs like product_{id} => rating)
    $productReviews = [];
    foreach ($order->items as $item) {
      $field = "product_{$item->product_id}";
      if ($request->has($field)) {
        $rating = (int) $request->input($field) ?? $request->rating;
        $productReviews[] = $this->saveReview($item->product_id, 'product', $userId, $rating, [
          'comment_id' => $orderId,
        ]);
        // update product rating
        $this->updateModelRating($item->product, $rating);
      }
    }

    // 4 Driver reviews
    $ride = Ride::where('order_id', $order->id)->first();
    if ($ride && $request->filled('driver_rating')) {
      $request->rating = $request->driver_rating;
      $request->comment = $request->driver_comment;
      $this->handleRideReview($ride->id, $userId, $request);
    }
    $ride->rating = $request->driver_rating;
    $ride->save();

    return [
      'message' => __('locale.Order review saved successfully.'),
      'order_review' => $orderReview,
      'store_review' => $storeReview,
      'product_reviews' => $productReviews,
    ];
  }

  private function handleUserReview($rideId, $userId, $request)
  {
    $ride = Ride::findOrFail($rideId);
    // User Review
    $userReview = $this->saveReview($ride->user_id, 'user', $userId, $request->rating, $request->comment, [
      'comment_id' => $rideId,
    ]);
    $this->updateModelRating($ride->user, $request->rating);
    return [
      'message' => __('locale.User review saved successfully.'),
      'review' => $userReview,
    ];
  }

}
