<?php
namespace App\Repositories\Eloquent;
use App\Models\Rental;
use App\Models\Installment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Repositories\Contracts\CarRepositoryInterface;
use App\Repositories\Contracts\DriverRepositoryInterface;
use App\Repositories\Contracts\RentalRepositoryInterface;
use App\Helpers\Helpers;
use App\Repositories\BaseRepository;
class RentalRepository extends BaseRepository implements RentalRepositoryInterface
{
  protected $carRepository;
  protected $driverRepository;
  public function __construct(
    Rental $rentalModel,
    CarRepositoryInterface $carRepository,
    DriverRepositoryInterface $driverRepository
  ) {
    parent::__construct($rentalModel);
    $this->carRepository = $carRepository;
    $this->driverRepository = $driverRepository;
  }

  public function allList(Request $request)
  {

    $canDelete = Auth::user()->can('rental-delete');
    $canEdit = Auth::user()->can('rental-edit');
    $canCreate = Auth::user()->can('rental-create');
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $builder = Rental::query();
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('id', 'LIKE', "%{$search}%")
          ->orWhere('total_amount', 'LIKE', "%{$search}%")
          ->orWhere('order_type', 'LIKE', "%{$search}%")
          ->orWhere('status', 'LIKE', "%{$search}%");
      });
    }

    if (!is_null($request->input('order_type'))) {
      $query->where('order_type', $request->input('order_type'));
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', $request->input('status'));
    }
    if ($request->filled('car_id')) {
      $query->where('car_id', $request->car_id);
    }
    if ($request->filled('driver_id')) {
      $query->where('driver_id', $request->driver_id);
    }
    return \DataTables::eloquent($query->with(['driver.user', 'car', 'installments']))
      ->addIndexColumn()
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('currency', fn($rental) => $rental['driver']['user']?->currency ?? $currency)
      ->addColumn('driver_name', fn($rental) => $rental['driver']['user']?->name ?? '-')
      ->addColumn('driver_email', fn($rental) => $rental['driver']['user']?->email ?? '-')
      ->addColumn('driver_avatar', function ($rental) {
        if ($rental['driver']['user']->profile_photo_path) {
          return asset('storage/' . $rental['driver']['user']->profile_photo_path);
        }
      })
      ->addColumn('car_model', fn($rental) => $rental['car']?->vehicle_model ?? '-')
      ->addColumn('car_brand', fn($rental) => $rental['car']?->vehicle_brand ?? '-')
      ->addColumn('car_plate', fn($rental) => $rental['car']?->license_plate ?? '-')
      ->addColumn('car_image', function ($rental) {
        if ($rental['car']->vehicle_image) {
          return asset('storage/' . $rental['car']->vehicle_image);
        }
      })
      ->addColumn('installments_paid', function ($rental) {
        // Check if any installment has been paid
        return $rental->installments->where('paid_amount', '>', 0)->isNotEmpty() ? true : false;
      })
      ->addColumn('installments_list', function ($rental) {
        // Return installments as array for modal
        return $rental->installments->map(function ($ins) {
          return [
            'id' => $ins->id,
            'due_date' => $ins->payment_date,
            'amount' => $ins->amount,
            'paid_amount' => $ins->paid_amount,
            'status' => $ins->paid_amount > 0 ? 'Paid' : 'Partially',
          ];
        });
      })
      ->addColumn('delete-url', fn($rental) => url('admin/rentals/' . $rental->id))
      ->addColumn('status-url', fn() => url('admin/rentals/changestatus'))
      ->with([
        'recordsTotal' => $builder->count(), // ✅ total regardless of filters
        'create' => $canCreate // ✅ make it root-level
      ])
      ->make(true);
  }

  public function allRentals()
  {
    $cars = $this->carRepository->dropdown();
    $drivers = $this->driverRepository->dropdown();
    return [
      'cars' => $cars,
      'drivers' => $drivers
    ];
  }

  public function dropdown()
  {
    return Rental::where('status', 1)->orderBy('name')->get();
  }

  public function store(Request $request, $Id = null): array
  {
    $data = [];

    $rental = $Id ? Rental::find($Id) : new Rental;
    $rental->car_id = $request->car_id;
    $rental->driver_id = $request->driver_id;
    $rental->order_type = $request->order_type ?? $rental->order_type;
    $rental->total_amount = $request->total_amount ?? 0;
    $rental->down_payment = $request->down_payment ?? 0;
    $rental->pending_amount = ($rental->total_amount - $rental->down_payment - ($request->paid_installments * $rental->installment_amount));
    $rental->no_of_installments = $request->no_of_installments ?? 0;
    $rental->installment_amount = $request->installment_amount ?? 0;
    $rental->security_amount = $request->security_amount ?? 0;
    if (!$Id) {
      $rental->start_date = $request->start_date ?? date('Y-m-d');
      $rental->due_date = $request->due_date;
    }

    $rental->status = $request->status ?? 'Active';
    $rental->save();
    $data['id'] = $rental->id;
    return $data;
  }

  public function nextInstallment($rentalId)
  {
    $rental = Rental::findOrFail($rentalId);
    // Find the next pending installment
    $paidSum = Installment::where('rental_id', $rental->id)->sum('paid_amount');
    $pendingAmount = $rental->total_amount - ($rental->down_payment + $paidSum);
    $installmentsPaid = $rental->no_of_installments - ceil($pendingAmount / $rental->installment_amount);
    $amount = $pendingAmount - (($rental->no_of_installments - ($installmentsPaid + 1)) * $rental->installment_amount);
    $result = [
      'rental_id' => $rentalId,
      'installment_no' => $installmentsPaid + 1,
      'amount' => $amount,
      'paid_amount' => $amount,
      'due_date' => $rental->due_date,
    ];
    return $result;
  }

}
