<?php

namespace App\Repositories\Eloquent;
use App\Models\Product;
use App\Models\Store;
use App\Models\User;
use App\Models\Favorite;
use Illuminate\Http\Request;
use App\Models\ProductImage;
use App\Models\Category;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use App\Helpers\Helpers;
use App\Repositories\BaseRepository;
use App\Repositories\Contracts\ProductRepositoryInterface;
class ProductRepository extends BaseRepository implements ProductRepositoryInterface
{
  public function __construct(Product $productModel)
  {
    parent::__construct($productModel);
  }

  public function allList($request)
  {
    $user = User::find(auth()->id());
    $storeId = $user->user_type == 'Store' && $user->storeId ? $user->storeId : '';
    $canDelete = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('product-delete');
    $canEdit = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('product-edit');
    $canCreate = Auth::user()->user_type == 'Store' ? true : Auth::user()->can('product-create');
    $builder = Product::with(['category:id,name,parent_id', 'category.parent:id,name']);
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = $user->currency ?? cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('id', 'LIKE', "%{$search}%")
          ->orWhere('title', 'LIKE', "%{$search}%")
          ->orWhere('description', 'LIKE', "%{$search}%")
          ->orWhere('tags', 'LIKE', "%{$search}%")
          ->orWhere('sku', 'LIKE', "%{$search}%");
      });
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', (int) $request->input('status'));
    }
    // 📦 In stock filter
    if (!is_null($request->input('in_stock'))) {
      $query->where('in_stock', (int) $request->input('in_stock'));
    }

    if (!empty($request->input('category'))) {
      $category = $request->input('category');
      $query->whereHas('category', function ($q) use ($category) {
        $q->where('name', $category);
      });
    }

    // 🧭 Parent category filter
    if (!empty($request->input('parent_category'))) {
      $parentCategory = $request->input('parent_category');
      $query->whereHas('category.parent', function ($q) use ($parentCategory) {
        $q->where('name', $parentCategory);
      });
    }
    if (!empty($storeId)) {
      $query->where('store_id', (int) $storeId);
    }

    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('image', function ($product) {
        $image = $product->images->first();
        if ($image) {
          return asset('storage/' . $image->file);
        }
      })
      ->addColumn('category_name', fn($product) => $product['category']->name)
      ->addColumn('parent_category', fn($product) => $product['category']->parent_name)
      ->addColumn('diet', fn($product) => $product['category']->is_diet ? $product->is_non_veg ?? 0 : '')
      ->addColumn('currency', $currency)
      ->addColumn('edit', $canEdit)
      ->addColumn('delete', $canDelete)
      ->addColumn('create', $canCreate)
      ->addColumn('delete-url', fn($product) => url($prefix . 'products/' . $product->id))
      ->addColumn('status-url', fn() => url($prefix . 'products/changestatus'))
      ->addColumn('stock-url', fn() => url($prefix . 'products/changestock'))
      /*   ->with([
          'recordsTotal' => $builder->count() // ✅ total regardless of filters
        ]) */
      ->make(true);
  }

  public function getProductsByStore(Request $request)
  {
    $store_id = $request->store;
    return Product::where('store_id', $store_id)->select('id', 'title')->get();
    ;
  }

  public function getProductsByCategory(Request $request)
  {
    $category_id = $request->category_id;
    return Product::where('category_id', $category_id)->select('id', 'title')->get();
    ;
  }

  public function store(Request $request, $Id = null)
  {

    $product = $Id ? Product::find($Id) : new Product;
    $product->title = $request->title;
    if (empty($Id)) {
      $slug = Str::slug($request->title);
      $originalSlug = $slug;
      $count = 1;
      while (Product::where('sku', $slug)->exists()) {
        $slug = $originalSlug . '-' . $count++;
      }
      $user = User::find(auth()->id());
      $currency = $user->currency ?? cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
      $product->currency = $currency;
      $product->sku = $slug;
      $storeId = Store::where('user_id', $user->id)->first()?->id;
      $product->store_id = $storeId;
    }
    $category = Category::find($request->category);
    $product->description = $request->description;
    $product->price = $request->price;
    $product->discounted_price = $request->discounted_price;
    $product->category_id = $request->category;
    $product->status = $request->status ?? 0;
    $product->tags = collect(json_decode($request->product_tags))->pluck('value')->implode(',');
    $product->is_non_veg = $request->has('veg_type') && $request->veg_type == 'nonveg' ? 1 : 0;
    $product->in_stock = $request->has('in_stock');
    $product->prescriptions = $category->prescriptions ?? 0;
    $product->prescriptions = $category->prescriptions ?? 0;
    $product->save();
    // Handle image storage
    if (!empty($request->images)) {
      // Loop through each temp‐path (e.g. "products/temp/abc.jpg")
      foreach ($request->images as $tempPath) {
        if (Storage::disk('public')->exists($tempPath)) {
          $filename = basename($tempPath);
          $newPath = 'products/' . $filename;
          Storage::disk('public')->move($tempPath, $newPath);
          ProductImage::create([
            'product_id' => $product->id,
            'file' => $newPath,
          ]);
        }
      }
    }
    cacheForget("dropdown_store_categories_{$product->store_id}", ['dropdown']);
    cacheForget("dropdown_store_parent_categories_{$product->store_id}", ['dropdown']);
    return true;
  }

  public function changeStock($id, $status)
  {
    $product = Product::find($id);
    if ($product) {
      $product->in_stock = $status == 1 ? 0 : 1;
      $product->save();
    }
    return true;
  }

  //Api's
  public function productList(Request $request, $storeId)
  {
    $query = Product::query()
      ->withCount('orderItems')
      ->with([
        'images' => function ($img) {
          $img->orderBy('id')->limit(1); // only get first image
        }
      ])
      ->where('store_id', $storeId);

    // Price filter
    if (!empty($request->filled('price'))) {
      $priceType = $request->price;
      if ($priceType === 'low_to_high') {
        $query->orderBy('price', 'asc');
      } elseif ($priceType === 'high_to_low') {
        $query->orderBy('price', 'desc');
      }
    }

    // Category filter
    if ($request->filled('category_id')) {
      $category = (int) $request->category_id;
      $query->where('category_id', $category);
    }

    // Rating filter
    if ($request->filled('rating')) {
      $rating = (int) $request->rating;
      // only allow valid thresholds
      if (in_array($rating, [2, 3, 4, 5])) {
        $query->where('rating', '>=', $rating);
      }
    }
    // Veg Type filter
    if ($request->filled('veg_type')) {
      $vegType = $request->veg_type;
      if ($vegType === 'veg') {
        $query->where('is_non_veg', 0);
      } elseif ($vegType === 'non_veg') {
        $query->where('is_non_veg', 1);
      }
    }
    // Search filter (store fields + user name)
    if ($request->filled('search')) {
      $search = $request->search;
      $query->where(function ($q) use ($search) {
        $q->where('title', 'like', "%{$search}%")
          ->orWhere('sku', 'like', "%{$search}%")
          ->orWhere('description', 'like', "%{$search}%")
          ->orWhere('tags', 'like', "%{$search}%");
      });
    }

    // Default ordering: best sellers first
    if (empty($request->filled('price'))) {
      $query->orderByDesc('order_items_count')
        ->orderByDesc('rating');
    }

    return $query->paginate(10);
  }

  public function productDetail($id)
  {
    $user = auth()->user();
    $query = Product::query()
      ->withCount('orderItems')
      ->with(['images'])
      ->where('id', $id);
    // Favorite stores
    $favoriteProductIds = $user
      ? Favorite::where('user_id', $user->id)
        ->where('favoritable_type', Product::class)
        ->pluck('favoritable_id')
        ->toArray()
      : [];
    $product = $query->firstOrFail();
    $product->is_favorite = in_array($product->id, $favoriteProductIds);
    return $product;
  }

}
