<?php

namespace App\Repositories\Eloquent;
use App\Enums\CouponType;
use App\Models\Coupon;
use Illuminate\Http\Request;
use App\Services\CouponService;
use App\Repositories\BaseRepository;
use App\Repositories\Contracts\CouponRepositoryInterface;
class CouponRepository extends BaseRepository implements CouponRepositoryInterface
{
  public function __construct(Coupon $couponModel)
  {
    parent::__construct($couponModel);
  }

  public function allCoupons(Request $request)
  {
    $query = Coupon::query();

    if ($request->filled('type')) {
      $query->where('type', $request->type);
    }

    // Preload usage counts
    $query->withCount('usage');

    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('details_parsed', function ($row) {
        $details = json_decode($row->details, true);

        if (!is_array($details)) {
          return [];
        }

        // Collect user_ids
        $userIds = collect($details)
          ->pluck('user_id')
          ->filter()
          ->unique();

        // Fetch users in one query
        $users = \App\Models\User::whereIn('id', $userIds)
          ->pluck('name', 'id');

        // Attach user_name
        foreach ($details as &$detail) {
          if (isset($detail['user_id'])) {
            $detail['user_name'] = $users[$detail['user_id']] ?? 'N/A';
          }
        }

        return $details;
      })
      ->addColumn('usage_count', fn($row) => $row->usage_count) // ✅ preloaded
      ->addColumn('encrypted_id', fn($row) => encrypt($row->id))
      ->addColumn('status_url', fn() => url('admin/coupons/changestatus'))
      ->make(true);
  }


  public function store(Request $request)
  {
    return (new CouponService)->store($request);
  }

  public function show($id)
  {
    $coupon = Coupon::with('usage')->find(decrypt($id));
    return view('admin.coupons.show', compact('coupon'));
  }

  public function updateCoupon(Request $request, $id)
  {
    return (new CouponService)->update($request, $id);
  }


  public function getCouponForm(Request $request)
  {
    $type = $request->coupon_type;
    $view = match ($type) {
      CouponType::PRODUCT => 'admin.coupons.product_base_coupon',
      CouponType::CART => 'admin.coupons.cart_base_coupon',
      CouponType::CUSTOMER => 'admin.coupons.customer_base_coupon',
      CouponType::SELLER => 'admin.coupons.seller_base_coupon',
      CouponType::OFFER => 'admin.coupons.offers_base_coupon',
      CouponType::REGISTER => 'admin.coupons.register_base_coupon',
      CouponType::RIDE => 'admin.coupons.ride_base_coupon',
      default => abort(404)
    };
    return view($view);
  }


  public function changeStatus($id, $status)
  {
    $coupon = Coupon::findOrFail($id);
    $coupon->status = $status == 'Enable' ? 'Disable' : 'Enable';
    $coupon->save();
    return $coupon;
  }

  public function applyCoupon(Request $request)
  {
  }

}
