<?php

namespace App\Repositories\Eloquent;

use App\Models\Contact;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use App\Mail\SupportEmail;
use Illuminate\Support\Facades\Notification;
use App\Notifications\UserNotification;
use App\Repositories\BaseRepository;
use App\Repositories\Contracts\ContactRepositoryInterface;

class ContactRepository extends BaseRepository implements ContactRepositoryInterface
{
  public function __construct(Contact $contactModel)
  {
    parent::__construct($contactModel);
  }
  public function getTicketListWithStats()
  {
    $tickets = Contact::with('user')
      ->where('status', '!=', 'Deleted')
      ->latest()
      ->get();

    // Pre-calculate stats (single pass, no collection filter)
    $total = $tickets->count();
    $active = $tickets->where('status', 'Open')->count();
    $inactive = $tickets->where('status', 'Close')->count();

    // Cache permissions once
    $canDelete = auth()->user()->can('ticket-delete');
    $canEdit = auth()->user()->can('ticket-edit');
    $canCreate = auth()->user()->can('ticket-create');

    // Transform data using map() for cleaner code
    $data = $tickets->map(function ($ticket) use ($canDelete, $canEdit, $canCreate) {
      $user = $ticket->user;
      $isGuest = !$user || $ticket->user_id == 0;

      return [
        'id' => $ticket->id,
        'full_name' => $isGuest ? $ticket->username : $user->name,
        'username' => $isGuest ? $ticket->username : $user->name,
        'email' => $isGuest ? $ticket->email : $user->email,
        'avatar' => $isGuest ? '' : $user->profile_photo_path,
        'category' => $ticket->category,
        'subject' => $ticket->subject,
        'body' => $ticket->body,
        'ticket_no' => $ticket->ticket_no,
        'admin_note' => $ticket->admin_note ?? '--',
        'status' => $ticket->status,
        'user_id' => $isGuest ? 0 : $user->id,
        'user_id_encode' => $isGuest ? 0 : base64_encode($user->id),
        'edit' => $canEdit,
        'delete' => $canDelete,
        'create' => $canCreate,
        'delete-url' => url('admin/tickets/' . $ticket->id),
      ];
    });

    return [
      'data' => $data,
      'stats' => [
        'total' => $total,
        'active' => $active,
        'inactive' => $inactive
      ]
    ];
  }


  public function updateTicketMessage($id, $message)
  {
    $ticket = Contact::find($id);
    $ticket->body = $message;
    $ticket->status = 'Close';
    $ticket->save();
    return $ticket;
  }

  public function changeTicketStatus($request)
  {
    $ticket = Contact::find($request->id);
    $ticket->status = $request->status;
    $ticket->admin_note = $request->admin_note;
    $ticket->save();

    $name = $ticket->user_id == 0 ? $ticket->username : $ticket->user->name ?? '';
    $email = $ticket->user_id == 0 ? $ticket->email : $ticket->user->email ?? '';

    // Send mail
    if (!empty($email)) {
      $array = [
        'view' => 'emails.tickets',
        'subject' => 'Support Ticket - ' . $ticket->ticket_no,
        'from' => config('app.MAIL_USERNAME'),
        'name' => $name,
        'reward' => $ticket->user,
        'content' => 'This email is to notify that your support ticket number: ' . $ticket->ticket_no . ' has been Closed by ' . config('app.name') . ' admin.'
      ];

      try {
        Mail::to($email)->queue(new SupportEmail($array));
      } catch (\Exception $e) {
        // Log or ignore
      }
    }

    // Send notification
    if (!empty($ticket->user)) {
      $notifiable = User::find($ticket->user->id);

      $details = [
        'body' => 'Your support ticket number: ' . $ticket->ticket_no . ' has been Closed by ' . config('app.name') . ' admin.',
        'head' => 'Support Tickets',
        'request_id' => $ticket->id,
        'request_main_id' => $ticket->id,
        'user_id' => $ticket->user_id,
        'type' => 'App\Notifications\Contact',
        'notifiable_type' => 'App\Contact',
        'class' => 'icon-base ti tabler-help icon-lg warning',
        'color' => 'success media-heading',
        'url' => 'tickets',
        'sender_id' => Auth::id(),
        'tags' => 'Support Tickets',
      ];

      Notification::send($notifiable, new UserNotification($details));
    }

    return $ticket;
  }
}
