<?php
namespace App\Models;
use Illuminate\Database\Eloquent\SoftDeletes;
class VehicleCategory extends BaseModel
{
  use SoftDeletes;

  protected $fillable = ['parent_id', 'name', 'description', 'image', 'map_icon', 'status', 'type'];
  protected $appends = ['parent_name', 'delete_url', 'status_url'];
  protected static function boot()
  {
    parent::boot();
    static::deleting(function ($category) {
      // Prevent delete if this category has vehicles
      if ($category->vehicles()->exists()) {
        throw new \Exception(__('locale.Cannot delete category with associated vehicles.'));
      }
      // Prevent delete if any direct child has vehicles
      foreach ($category->children as $child) {
        if ($child->vehicles()->exists()) {
          throw new \Exception(__('locale.Cannot delete subcategory ":name" with vehicles.', ['name' => $child->name]));
        }
      }
      // Delete children categories safely
      $category->children->each->delete();
      // Delete related models (only if no vehicles)
      foreach ($category->models as $model) {
        if ($model->vehicles()->exists()) {
          throw new \Exception(__('locale.Cannot delete model ":name" under this category because vehicles exist.', ['name' => $model->name]));
        }
        $model->delete();
      }
    });
  }


  public function countryPrices()
  {
    return $this->belongsToMany(Country::class, 'country_vehicle_category')->withPivot('base_fare', 'per_km_rate', 'per_minute_price', 'per_minute_waiting_price');
  }

  public function parent()
  {
    return $this->belongsTo(VehicleCategory::class, 'parent_id');
  }
  public function children()
  {
    return $this->hasMany(VehicleCategory::class, 'parent_id');
  }
  // Recursive children
  public function childrenRecursive()
  {
    return $this->children()->with('childrenRecursive');
  }

  public function getParentNameAttribute()
  {
    return $this->parent ? $this->parent->name : '';
  }
  public function getDeleteUrlAttribute()
  {
    $prefix = request()->segment(1);
    return url("{$prefix}/vehicle-categories/{$this->id}");
  }
  public function getStatusUrlAttribute()
  {
    $prefix = request()->segment(1);
    return url("{$prefix}/vehicle-categories/changestatus");
  }
  public function models()
  {
    return $this->hasMany(VehicleModel::class, 'vehicle_category_id');
  }

  public function vehicles()
  {
    return $this->hasMany(Vehicle::class, 'vehicle_category_id');
  }

}
