<?php

namespace App\Models;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Laravel\Jetstream\HasProfilePhoto;
use Laravel\Sanctum\HasApiTokens;
use Musonza\Chat\Traits\Messageable;
use Spatie\Permission\Traits\HasRoles;
use App\Traits\HasEnumsTrait;
use Illuminate\Database\Eloquent\SoftDeletes;
use Laravel\Cashier\Billable;
use App\Enums\UserType;
class User extends Authenticatable implements MustVerifyEmail
{
  use HasApiTokens, SoftDeletes; // ✅ Add SoftDeletes

  /** @use HasFactory<\Database\Factories\UserFactory> */
  use HasFactory;
  use HasProfilePhoto;
  use Notifiable;
  use TwoFactorAuthenticatable;
  use Messageable;
  use HasRoles;
  use Billable;
  use HasEnumsTrait;

  protected function defaultProfilePhotoUrl(): string
  {
    $name = trim(collect(explode(' ', $this->name))->map(function ($segment) {
      return mb_substr($segment, 0, 1);
    })->join(' '));

    // Force format=webp
    return 'https://ui-avatars.com/api/?name='
      . urlencode($name)
      . '&color=7F9CF5&background=EBF4FF&format=jpg';
  }

  protected array $enums = [
    'user_type' => UserType::class,
  ];
  protected $fillable = ['name', 'email', 'email_verified_at', 'dial_code_iso', 'dial_code', 'phone', 'password', 'user_type', 'two_factor_secret', 'two_factor_recovery_codes', 'two_factor_confirmed_at', 'remember_token', 'current_team_id', 'current_country_id', 'timezone', 'profile_photo_path', 'fcm_token', 'referral_code', 'referred_by', 'referral_bonus_credited', 'stripe_id', 'pm_type', 'pm_last_four', 'trial_ends_at', 'default_payment_method', 'start_date', 'end_date', 'status', 'is_verified', 'last_seen', 'wallet', 'currency', 'bank_details'];
  /**
   * The attributes that should be hidden for serialization.
   *
   * @var array<int, string>
   */
  protected $hidden = [
    'password',
    'remember_token',
    'two_factor_recovery_codes',
    'two_factor_secret',
  ];

  /**
   * The accessors to append to the model's array form.
   *
   * @var array<int, string>
   */
  protected $appends = [
    'profile_photo_url',
    'storeId',
    'driverId'
  ];

  /**
   * Get the attributes that should be cast.
   *
   * @return array<string, string>
   */
  protected function casts(): array
  {
    return [
      'email_verified_at' => 'datetime',
      'password' => 'hashed',
    ];
  }

  public function driver()
  {
    return $this->hasOne(Driver::class, 'user_id');
  }

  public function setPhoneAttribute($value)
  {
    $this->attributes['phone'] = $value === null ? null : preg_replace('/\s+/', '', $value);
  }

  public function store()
  {
    return $this->hasOne(Store::class, 'user_id');
  }

  public function getStoreIdAttribute()
  {
    return $this->store()->first()?->id;
  }

  public function getDriverIdAttribute()
  {
    return $this->driver()->first()?->id;
  }

  public function subscriptionRecord()
  {
    return $this->hasOne(Subscription::class, 'user_id');
  }

  public function activeSubscriptions()
  {
    return $this->hasMany(Subscription::class)
      ->whereIn('status', ['paid', 'complete']);
  }

  public function auditLogs()
  {
    return $this->hasMany(AuditLog::class);
  }
  public function wallet()
  {
    return $this->morphOne(Wallet::class, 'owner');
  }

  public function addresses()
  {
    return $this->hasMany(Address::class);
  }

  public function updatePushSubscription()
  {
    return $this->morphMany(WebpushSubscription::class, 'subscribable');
  }
  public function favoriteStores()
  {
    return $this->morphedByMany(Store::class, 'favoritable', 'favorites');
  }
  public function favoriteProducts()
  {
    return $this->morphedByMany(Product::class, 'favoritable', 'favorites');
  }

}
