<?php
namespace App\Models;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class Driver extends BaseModel
{
  protected $fillable = ['user_id', 'type', 'dob', 'gender', 'address', 'country', 'country_id', 'state_id', 'city_id', 'state', 'city', 'postcode', 'landmark', 'license_number', 'license_expiry', 'license_image', 'rides_count', 'rating', 'rating_count', 'wallet', 'latitude', 'longitude', 'is_active', 'has_ride', 'is_blocked', 'blocked_until'];
  // Cast fields to proper type
  protected $casts = [
    'rides_count' => 'integer',
    'rating_count' => 'integer',
  ];

  // Set default attribute values
  protected $attributes = [
    'rides_count' => 0,
    'rating_count' => 0
  ];

  protected $appends = ['driver_name'];

  public function scopeActive($query, $countryId = null)
  {
    $query->where('is_active', 1)
      ->where('has_ride', 0)
      ->where('is_blocked', 0)
      ->whereDate('license_expiry', '>=', Carbon::now())
      ->whereHas('user', function ($q) {
        $q->where('status', 1)
          ->where('is_verified', 1);
      });
    if ($countryId) {
      $query->where('country_id', $countryId);
    }
    return $query;
  }
  // app/Models/Driver.php
  public function scopeWithDistance($query, $lat, $lng, $alias = 'distance', $radius = null)
  {
    $query->addSelect(DB::raw("
        ROUND(
            6371 * ACOS(
                LEAST(1,
                    COS(RADIANS(?)) * COS(RADIANS(drivers.latitude)) *
                    COS(RADIANS(drivers.longitude) - RADIANS(?)) +
                    SIN(RADIANS(?)) * SIN(RADIANS(drivers.latitude))
                )
            ), 2
        ) as $alias
    "))
      ->addBinding([$lat, $lng, $lat], 'select');
    // Apply radius filter if provided
    if ($radius !== null) {
      $query->having($alias, '<=', $radius);
    }
    return $query;
  }



  // Driver.php
  public function user()
  {
    return $this->belongsTo(User::class, 'user_id')->withTrashed();
  }
  public function vehicle()
  {
    return $this->hasOne(Vehicle::class, 'driver_id');
  }
  public function getDriverNameAttribute()
  {
    return $this->user ? $this->user->name : null;
  }
  public function orders()
  {
    return $this->hasMany(Rental::class);
  }
  public function wallet()
  {
    return $this->morphOne(Wallet::class, 'owner');
  }
}
