<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;

class BaseModel extends Model
{
  protected $dateFormat = 'Y-m-d H:i:s'; // keep DB format

  /**
   * Always serialize default *_at fields
   */
  protected function serializeDate(\DateTimeInterface $date): string
  {
    return $this->convertToUserTimezone($date)->format('Y-m-d H:i:s');
  }

  /**
   * Convert a value to user timezone
   */
  protected function convertToUserTimezone($value): Carbon
  {
    $timezone = Auth::user()->timezone ?? config('app.timezone', 'UTC');
    return Carbon::parse($value)->timezone($timezone);
  }

  /**
   * Magic accessor handler: return *_date, *_time, *_full dynamically
   */
  public function __get($key)
  {
    // check if it matches created_at_date, updated_at_full, etc.
    if (preg_match('/^(.*)_(date|time|full)$/', $key, $matches)) {
      $field = $matches[1];
      $formatType = $matches[2];

      if (!empty($this->attributes[$field] ?? null)) {
        $dt = $this->convertToUserTimezone($this->attributes[$field]);

        return match ($formatType) {
          'date' => $dt->format('d M, Y'),
          'time' => $dt->format('H:i A'),
          'full' => $dt->format('d M, Y H:i A'),
        };
      }
    }

    return parent::__get($key);
  }
}
