<?php

namespace App\Jobs;

use App\Models\RideRequest;
use App\Models\RideTemp;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Events\RideExpiredEvent;
use App\Repositories\Eloquent\RideRequestRepository;
use App\Repositories\Eloquent\RideRepository;
use App\Helpers\Helpers;

class RideRequestTimeoutJob implements ShouldQueue
{
  use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

  protected $rideRequestId;
  protected $rideRequestRepository;
  protected $rideRepository;

  public function __construct(int $rideRequestId)
  {
    $this->rideRequestId = $rideRequestId;
  }

  /**
   * Execute the job.
   */
  public function handle(RideRequestRepository $rideRequestRepository, RideRepository $rideRepository): void
  {
    $rideRequest = RideRequest::find($this->rideRequestId);
    if (!$rideRequest || $rideRequest->status !== 'Pending') {
      return; // already responded
    }

    // Mark as timeout
    $rideRequest->update(['status' => 'Timeout']);

    $rideTempId = $rideRequest->ride_temp_id;
    $temp = RideTemp::find($rideTempId);

    $driverLimit = Helpers::setting('driver_request_simultaneously', 'site_setting');
    if ($temp->type == 'Delivery') {
      $requestTime = Helpers::setting('request_time_delivery_minutes', 'site_setting');
    } else {
      $requestTime = Helpers::setting('request_time_minutes', 'site_setting');
    }

    $pendingCount = RideRequest::where('ride_temp_id', $temp->id)
      ->where('status', 'Pending')
      ->count();

    if ($pendingCount === 0) {
      // Stop if ride already expired
      if ($temp->status === 'Confirmed' || $temp->status === 'Expired') {
        return;
      }
      // Check if specific minutes passed
      if ($temp->created_at->addMinutes((int) $requestTime)->isPast()) {
        $temp->update(['status' => 'Expired']);
        broadcast(new RideExpiredEvent($temp))->toOthers();
        return;
      }
      // All current requests handled → send next batch
      $alreadyTried = RideRequest::where('ride_temp_id', $temp->id)
        ->pluck('driver_id')->toArray();
      $details = json_decode($temp->details);
      $countryId = $details->country_id ?? 94;
      $nextDrivers = $rideRepository->findNearestDrivers($temp->pickup_lat, $temp->pickup_lng, $driverLimit, $temp->vehicle_category_id, $countryId, $alreadyTried);

      if (count($nextDrivers)) {
        $rideRequestRepository->createForDrivers($temp, $nextDrivers);
      } else {
        $temp->update(['status' => 'Expired']);
        broadcast(new RideExpiredEvent($temp))->toOthers();
      }
    }
  }
}
