<?php
namespace App\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;
use App\Helpers\Helpers;

class OrderResource extends JsonResource
{
  public function toArray($request)
  {
    $defaultCurrency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    $data = $this;
    $currency = $data->currency ?? $defaultCurrency;
    $address = json_decode($data->address);
    // ✅ Decode txn_details JSON (Stripe response)
    $txnDetails = !empty($data->transaction?->txn_details)
      ? json_decode($data->transaction->txn_details, true)
      : [];

    // ✅ Driver details (only if ride is assigned AND order is Dispatched/Delivered)
    $driver = (object) [];
    if ($data->ride && in_array($data->status, ['Dispatched', 'Delivered'])) {
      $driver = (object) [
        'id' => $data->ride->driver?->id,
        'name' => $data->ride->driver?->user?->name ?? '',
        'phone' => $data->ride->driver?->user?->dial_code . $data->ride->driver?->user?->phone ?? '',
        'image' => $data->ride->driver?->user?->profile_photo_path
          ? asset('storage/' . $data->ride->driver?->user?->profile_photo_path)
          : asset('assets/img/placeholder.jpg'),
        'otp' => $data->ride->ride_otp ?? '',
        'rating' => $data->ride->driver?->rating ?? '',
        'rating_count' => $data->ride->driver?->rating_count ?? '',
      ];
    }
    // ✅ Transaction object (only if payment_type = online)
    $transaction = (object) [];
    if ($data->payment_type === 'online' && $data->transaction) {
      $transaction = (object) [
        'txn_id' => $data->transaction->txn_id,
        'payment_mode' => $data->transaction->payment_mode,
        'card' => isset($txnDetails['payment_method_details']['card'])
          ? [
            'brand' => $txnDetails['payment_method_details']['card']['brand'] ?? '',
            'last4' => $txnDetails['payment_method_details']['card']['last4'] ?? '',
            'exp' => ($txnDetails['payment_method_details']['card']['exp_month'] ?? '') . '/' . ($txnDetails['payment_method_details']['card']['exp_year'] ?? ''),
          ]
          : [],
      ];
    }
    // ✅ Map
    $map = (object) [];
    if (in_array($data->status, ['Out for Delivery'])) {
      $map = (object) [
        'driver_latitude' => $data->ride->driver?->latitude,
        'driver_longitude' => $data->ride->driver?->longitude ?? '',
        'latitude' => $data->latitude ?? '',
        'longitude' => $data->longitude ?? '',
      ];
    }
    return [
      'id' => (int) $data->id,
      'name' => $data->store?->user?->name ?? '',
      'address' => $data->store?->address,
      'rating' => $data->store?->rating,
      'orderId' => $data->order_id,
      'tax' => $data->tax > 0 ? $currency . $data->tax : '',
      'subtotal' => $currency . $data->subtotal,
      'discount' => $data->discount > 0 ? $currency . $data->discount : '',
      'delivery_charges' => $data->shipping > 0 ? $currency . $data->shipping : '',
      'grand_total' => $currency . $data->grand_total,
      'date_time' => $data->created_at
        ? \Carbon\Carbon::parse($data->created_at)->format('d M, h:i a')
        : null,
      'status' => $data->status,
      'payment_status' => $data->payment_status,
      'payment_type' => $data->payment_type,
      'is_rateable' => empty($data->review->rating) && $data->status == 'Delivered' ? true : false,
      'you_rated' => $data->review->rating ?? '',
      'delivery_address' => $address->address . ' ' . $address->state . ', ' . $address->city . ', ' . $address->postcode,
      'delivered_at' => $data->delivered_at ? \Carbon\Carbon::parse($data->created_at)->format('d M, h:i a')
        : null,
      'image' => $data->store?->user?->profile_photo_path
        ? asset('storage/' . $data->store?->user?->profile_photo_path)
        : asset('assets/img/placeholder.jpg'),

      // ✅ Order items
      'items' => $data->items->map(function ($item) use ($currency) {
        return [
          'id' => $item->id,
          'product_id' => $item->product_id,
          'product_name' => $item->product?->title ?? '',
          'category' => $item->product?->category?->name ?? '',
          'quantity' => $item->quantity,
          'is_non_veg' => $item->is_non_veg ?? 0,
          'price' => $currency . $item->price,
          'subtotal' => $currency . ($item->price * $item->quantity),
          'image' => $item->product?->images[0]
            ? asset('storage/' . $item->product?->images[0]->file)
            : asset('assets/img/placeholder.jpg'),
        ];
      }),
      'driver' => $driver,
      'transaction' => $transaction,
      'map' => $map,
      // ✅ Track order timeline
      'track_order' => $data->statusLogs
        ->sortBy('created_at')
        ->map(function ($log, $index) use ($data) {
          $isLast = $index === $data->statusLogs->count() - 1;
          return [
            'status' => str_replace('_', ' ', $log->status),
            'message' => Helpers::getStatusMessage($log->status),
            'time' => $log->created_at ? \Carbon\Carbon::parse($data->created_at)->format('l h:i A')
              : null,
            'is_last' => $isLast,
          ];
        })
        ->values(),
    ];
  }

  public function with($request)
  {
    return [
      'success' => true,
      'status' => 200
    ];
  }


}
