<?php
namespace App\Http\Requests;
use Illuminate\Foundation\Http\FormRequest;

use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;
use App\Traits\ApiResponseTrait;
use App\Rules\PhoneNumber;
class DriverRequest extends FormRequest
{
  use ApiResponseTrait;
  public function authorize(): bool
  {
    return true; // or add your auth logic
  }

  public function rules(): array
  {
    $userId = $this->input('user_id');
    $screen = $this->input('screen_type'); // e.g., 'details', 'vehicle', 'documents', null
    $rules = [];

    // Basic fields (name, email, phone, password, type)
    if (empty($screen)) {
      $rules = array_merge($rules, [
        'name' => 'required|string|min:3',
        'email' => [
          'required',
          'unique:users,email,' . ($userId ?? 'NULL') . ',id',
        ],
        'phone' => [
          'required',
          'min:4',
          'max:17',
          new PhoneNumber($this->input('dial_code')),
          'unique:users,phone,' . ($userId ?? 'NULL') . ',id',
        ],
        'password' => $userId ? 'nullable|min:6|confirmed' : 'required|confirmed|min:6',
        'type' => 'required|in:Ride,Delivery',
      ]);
    }

    // User details
    if (empty($screen) || $screen === 'details') {
      $rules = array_merge($rules, [
        'dob' => ['required', 'date', 'before_or_equal:' . now()->subYears(18)->toDateString()],
        'gender' => 'required|in:Male,Female,Other,N/A',
        'address' => 'required|string',
        'city' => 'required|string',
        'state' => 'required|string',
        'country' => 'required|string',
        'postcode' => 'required|string',
        'landmark' => 'nullable|string',
        'bank_details' => 'nullable',
      ]);
    }

    // Vehicle details
    if (empty($screen) || $screen === 'vehicle') {
      $rules = array_merge($rules, [
        'vehicle_parent_category_id' => 'required|exists:vehicle_categories,id',
        'vehicle_category_id' => 'required|exists:vehicle_categories,id',
        'vehicle_model_id' => 'required|exists:vehicle_models,id',
        'vehicle_image' => ($userId ? 'nullable' : 'required') . '|image',
        'insurance_image' => ($userId ? 'nullable' : 'required') . '|image',
        'vehicle_registration_image' => ($userId ? 'nullable' : 'required') . '|image',
        'road_worthiness_image' => 'nullable|image',
      ]);
    }

    // Documents
    if (empty($screen) || $screen === 'documents') {
      $rules = array_merge($rules, [
        'license_number' => 'required|string',
        'license_expiry' => ['required', 'date', 'after_or_equal:' . now()->toDateString()],
        'license_image' => ($userId ? 'nullable' : 'required') . '|image',
      ]);
    }

    return $rules;
  }


  public function messages(): array
  {
    return [

    ];
  }

  public function failedValidation(Validator $validator)
  {
    $errors = $validator->errors()->all();
    $message = implode("\n", $errors);
    throw new HttpResponseException(
      $this->unprocessableResponse($errors, $message),
    );
  }


}
