<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\Exceptions\HttpResponseException;
use App\Traits\ApiResponseTrait;

class CouponRequest extends FormRequest
{
  use ApiResponseTrait;

  public function authorize()
  {
    return true;
  }

  public function rules(): array
  {
    $couponId = $this->route('coupon') ?? $this->route('id'); // supports both named and implicit route bindings

    $rules = [
      'coupon_type' => 'required|in:ride_base,cart_base,product_base,customer_base,seller_base,offer_base,register_base,category_base',
      'coupon_code' => 'required|string|max:255|unique:coupons,code,' . $couponId,
      'discount' => 'required|numeric|min:0',
      'discount_type' => 'required_unless:coupon_type,register_base|in:percent,amount,points',
      'max_discount' => 'required_if:discount_type,percent',
      'date' => 'required|string',
      'usage_type' => 'required',
      'maximum_usage' => 'required_if:usage_type,limited',
    ];

    switch ($this->coupon_type) {
      case 'ride_base':
        break;
      case 'product_base':
        $rules['category_ids'] = 'required|array';
        $rules['product_ids'] = 'required|array';
        $rules['subcategory_ids'] = 'nullable|array';
        break;

      case 'category_base':
        $rules['category_ids'] = 'required|array';
        $rules['subcategory_ids'] = 'nullable|array';
        break;

      case 'cart_base':
        $rules['min_buy'] = 'required|numeric|min:0';
        break;

      case 'customer_base':
      case 'seller_base':
        $rules['min_buy'] = 'required|numeric|min:0';
        $rules['customers'] = 'nullable|array';
        break;

      case 'register':
        $rules['remarks'] = 'nullable|string|max:1000';
        break;

      case 'offer_base':
        break;
    }

    return $rules;
  }

  public function failedValidation(Validator $validator)
  {
    $errors = $validator->errors()->all();
    $message = implode("\n", $errors);
    throw new HttpResponseException(
      $this->unprocessableResponse($errors, $message),
    );
  }
}
