<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use App\Traits\JsonResponseTrait;
use App\Models\WalletTransaction;
use App\Models\User;
use App\Models\Driver;
use App\Models\Store;
use App\Models\Wallet;
use Illuminate\Http\Request;
use App\Helpers\Helpers;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Relations\Relation;
class WalletTransactionController extends Controller
{
  use JsonResponseTrait;
  function __construct()
  {
    if (auth()->check() && auth()->user()->order_type === 'admin') {
      $this->middleware('permission:transaction-list|transaction-create|transaction-edit|transaction-delete', ['only' => ['index', 'show', 'transactionListAll', 'walletHistory']]);
      $this->middleware('permission:transaction-create', ['only' => ['create']]);
      $this->middleware('permission:transaction-edit', ['only' => ['edit', 'update', 'changeStatus']]);
      $this->middleware('permission:transaction-delete', ['only' => ['destroy']]);
    }
  }
  public function index()
  {
    return view('store.wallet_transactions.index');
  }

  public function transactionListAll(Request $request)
  {
    // ✅ Define morph map only for this function
    Relation::enforceMorphMap([
      'user' => User::class,
      'driver' => Driver::class,
      'store' => Store::class,
    ]);
    $user = User::find(auth()->id());
    $userType = $user->user_type;
    $storeId = $userType == 'Store' && $user->storeId ? $user->storeId : '';
    $builder = WalletTransaction::with('wallet', 'order')
      ->when($storeId, function ($query, $storeId) {
        // ✅ If storeId is provided → filter for store owner
        $query->whereHas('wallet', function ($q) use ($storeId) {
          $q->where('owner_type', 'store')
            ->where('owner_id', $storeId);
        })->where('order_type', '!=', 'ride');
      }, function ($query) {
        // ✅ If storeId is empty → filter for user owner
        $query->whereHas('wallet', function ($q) {
          $q->where('owner_type', '!=', 'user');
        });
      });

    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->where('order_id', 'LIKE', "%{$search}%")
          ->orWhere('payment_mode', 'LIKE', "%{$search}%")
          ->orWhere('status', 'LIKE', "%{$search}%")
          ->orWhere('type', 'LIKE', "%{$search}%");
      });
    }
    if (!is_null($request->input('status'))) {
      $query->where('status', $request->input('status'));
    }
    if (!is_null($request->input('wallet_id'))) {
      $query->where('wallet_id', $request->input('wallet_id'));
    }
    if (!is_null($request->input('payment_mode'))) {
      $query->where('payment_mode', $request->input('payment_mode'));
    }
    if (!is_null($request->input('type'))) {
      $query->where('type', $request->input('type'));
    }
    if (!is_null($request->input('order_type'))) {
      $query->where('order_type', $request->input('order_type'));
    }
    if (!is_null($request->input('user_type'))) {
      $userType = $request->input('user_type');

      $query->whereHas('wallet', function ($q) use ($userType) {

        // Case 1: owner is a User → filter directly
        $q->whereHasMorph(
          'owner',                // relation on Wallet, not WalletTransaction
          [User::class],
          function ($q2) use ($userType) {
            $q2->where('user_type', $userType);
          }
        );

        // Case 2: owner is Driver or Store → filter on related user
        $q->orWhereHasMorph(
          'owner',
          [Driver::class, Store::class],
          function ($q2) use ($userType) {
            $q2->whereHas('user', function ($q3) use ($userType) {
              $q3->where('user_type', $userType);
            });
          }
        );

      });
    }
    // 📅 Date filter
    if (!empty($request->input('start_date')) && !empty($request->input('end_date'))) {
      $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
      $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
      $query->whereBetween('created_at', [$startDate, $endDate]);
    }
    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('currency', function ($txn) use ($currency) {
        return $txn->wallet?->currency ?? $currency;
      })
      ->addColumn('order_id', function ($txn) {
        return $txn->order_id ?? '';
      })
      ->addColumn('rideId', function ($txn) {
        return $txn->order?->ride_id ?? '';
      })
      ->addColumn('orderId', function ($txn) {
        return $txn->order?->order_id ?? '';
      })
      ->addColumn('user_id', function ($txn) {
        return $txn->wallet?->owner?->user?->id ?? '-';
      })
      ->addColumn('name', function ($txn) {
        return $txn->wallet?->owner?->user?->name ?? '-';
      })
      ->addColumn('email', function ($txn) {
        return $txn->wallet?->owner?->user?->email ?? '-';
      })
      ->addColumn('user_type', function ($txn) {
        return $txn->wallet?->owner?->user?->user_type ?? '-';
      })
      ->with([
        'recordsTotal' => $builder->count() // ✅ total regardless of filters
      ])
      ->make(true);
  }

  public function walletHistory($id, $userId)
  {
    $user = User::find($userId);
    return view('store.wallet_transactions.index', compact('id', 'user'));
  }
  public function walletHistoryUser($userId)
  {
    $user = User::find($userId);
    if ($user->user_type == 'Driver') {
      $wallet = Wallet::where('owner_type', 'driver')->where('owner_id', $user->driverId)->first();
    } elseif ($user->user_type == 'Store') {
      $wallet = Wallet::where('owner_type', 'store')->where('owner_id', $user->storeId)->first();
    } else {
      $wallet = Wallet::where('owner_type', 'user')->where('owner_id', $user->id)->first();
    }
    $id = $wallet?->id ?? 0;

    return view('store.wallet_transactions.index', compact('id', 'user'));
  }
  public function walletSettlements()
  {
    return view('admin.wallets.settlements');
  }

}
