<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use App\Traits\JsonResponseTrait;
use App\Http\Requests\ClearDueRequest;
use App\Http\Requests\ClearBalanceRequest;
use App\Models\Wallet;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use Illuminate\Http\Request;
use App\Helpers\Helpers;
use App\Services\WalletService;
use App\Models\WalletTransaction;
class WalletController extends Controller
{
  use JsonResponseTrait;
  function __construct()
  {
    if (auth()->check() && auth()->user()->order_type === 'admin') {
      $this->middleware('permission:wallet-list|wallet-create|wallet-edit|wallet-delete', ['only' => ['index', 'show', 'walletStores', 'walletDrivers', 'walletListAll', 'walletHistory', 'walletHistoryListAll']]);
      $this->middleware('permission:wallet-create', ['only' => ['create', 'clearDues']]);
      $this->middleware('permission:wallet-edit', ['only' => ['edit', 'update', 'changeStatus']]);
      $this->middleware('permission:wallet-delete', ['only' => ['destroy']]);
    }
  }
  public function walletStores()
  {
    $totals = Wallet::where('owner_type', 'store')
      ->selectRaw('
        wallets.currency,
        SUM(wallets.balance) as total_balance,
        SUM(wallets.pending_dues) as total_pending_dues,
        COALESCE((
            SELECT SUM(wr.total_amount)
            FROM withdrawal_requests as wr
            WHERE wr.currency COLLATE utf8mb4_general_ci = wallets.currency COLLATE utf8mb4_general_ci
              AND wr.status = "Pending"
              AND wr.user_type = "store"
        ), 0) as total_withdrawals
    ')
      ->groupBy('currency')
      ->get();

    return view('admin.wallets.stores', compact('totals'));
  }

  public function walletDrivers()
  {
    $totals = Wallet::where('owner_type', 'driver')
      ->selectRaw('
        wallets.currency,
        SUM(wallets.balance) as total_balance,
        SUM(wallets.pending_dues) as total_pending_dues,
        COALESCE((
            SELECT SUM(wr.total_amount)
            FROM withdrawal_requests as wr
            WHERE wr.currency COLLATE utf8mb4_general_ci = wallets.currency COLLATE utf8mb4_general_ci
              AND wr.status = "Pending"
              AND wr.user_type = "driver"
        ), 0) as total_withdrawals
    ')
      ->groupBy('currency')
      ->get();

    return view('admin.wallets.drivers', compact('totals'));
  }


  public function walletListAll(Request $request)
  {
    $user = User::find(auth()->id());

    $userType = $user->user_type;
    if ($userType == 'Store') {
      $Id = $user->storeId ?? '';
    } elseif ($userType == 'Driver') {
      $Id = $user->driverId ?? '';
    } else {
      $Id = '';
    }
    $builder = Wallet::with('owner.user');
    $prefix = Helpers::unreadCounts();
    $prefix = $prefix['prefix'];
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));
    // Clone for filters
    $query = clone $builder;
    if (!empty($request->input('search.value'))) {
      $search = $request->input('search.value');
      $query->where(function ($q) use ($search) {
        $q->whereHasMorph('owner', [User::class], function ($q2) use ($search) {
          $q2->where('name', 'LIKE', "%{$search}%")
            ->orWhere('email', 'LIKE', "%{$search}%");
        });
      });
    }
    if (!is_null($request->input('owner_type'))) {
      $query->where('owner_type', $request->input('owner_type'));
    }
    if (!empty($Id)) {
      $query->where('owner_id', (int) $Id);
    }

    return \DataTables::eloquent($query)
      ->addIndexColumn()
      ->addColumn('name', function ($wallet) {
        return $wallet->owner?->user?->name ?? '-';
      })
      ->addColumn('user_id', function ($wallet) {
        return $wallet->owner?->user?->id ?? '-';
      })
      ->addColumn('email', function ($wallet) {
        return $wallet->owner?->user?->email ?? '-';
      })
      // ✅ Add avatar column (show image HTML)
      ->addColumn('avatar', function ($wallet) {
        if ($path = $wallet->owner?->user?->profile_photo_path) {
          return asset('storage/' . $path);
        }
        return null;
      })
      /*   ->with([
          'recordsTotal' => $builder->count() // ✅ total regardless of filters
        ]) */
      ->make(true);
  }

  public function clearDues(ClearDueRequest $request, WalletService $walletService)
  {
    $wallet = Wallet::findOrFail($request->wallet_id);
    // ✅ Rule 1: Amount must not exceed wallet dues
    if ($request->amount > $wallet->pending_dues) {
      return $this->errorResponse(__('locale.Amount cannot exceed wallet dues.'), 400);
    }
    if ($request->amount > $wallet->balance) {
      return $this->errorResponse(__('locale.Amount cannot exceed wallet balance.'), 400);
    }

    $walletService->manualPayDues(
      $wallet,
      $request->amount,
      "Txn ID: {$request->txn_id}, Fee: {$request->tran_fee}, Details: {$request->txn_detail}"
    );
    $wallet = Wallet::with('owner.user')->findOrFail($request->wallet_id);
    $user = $wallet->owner?->user;

    // Add computed attributes
    $wallet->name = $user?->name ?? '-';
    $wallet->user_id = $user?->id ?? '-';
    $wallet->email = $user?->email ?? '-';
    $wallet->avatar = $user?->profile_photo_path ? asset('storage/' . $user->profile_photo_path) : null;
    return $this->successResponse($wallet, __('locale.Dues cleared successfully!'));
  }

  public function clearBalance(ClearBalanceRequest $request, WalletService $walletService)
  {
    $wallet = Wallet::findOrFail($request->wallet_id);
    // ✅ Rule 1: Amount must not exceed wallet dues
    if ($request->amount > $wallet->pending_dues) {
      return $this->errorResponse(__('locale.Amount cannot exceed wallet dues.'), 400);
    }

    $walletService->manualReceiveDues(
      $wallet,
      $request->amount,
      "Txn ID: {$request->txn_id}, Fee: {$request->tran_fee}, Details: {$request->txn_detail}"
    );
    $wallet = Wallet::with('owner.user')->findOrFail($request->wallet_id);
    $user = $wallet->owner?->user;

    // Add computed attributes
    $wallet->name = $user?->name ?? '-';
    $wallet->user_id = $user?->id ?? '-';
    $wallet->email = $user?->email ?? '-';
    $wallet->avatar = $user?->profile_photo_path ? asset('storage/' . $user->profile_photo_path) : null;
    return $this->successResponse($wallet, __('locale.Amount Received successfully!'));
  }


}
