<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use App\Http\Requests\VehicleModelRequest;
use App\Models\VehicleModel;
use App\Repositories\Contracts\VehicleModelRepositoryInterface;
use App\Traits\JsonResponseTrait;
use App\Traits\ImageUploadTrait;
use Illuminate\Http\Request;
class VehicleModelController extends Controller
{
  use JsonResponseTrait, ImageUploadTrait;

  protected $vehicleModelRepository;

  public function __construct(VehicleModelRepositoryInterface $vehicleModelRepository)
  {
    $this->middleware('permission:vehicle-list|vehicle-create|vehicle-edit|vehicle-delete', ['only' => ['index', 'show', 'modelListAll']]);
    $this->middleware('permission:vehicle-create', ['only' => ['create', 'store']]);
    $this->middleware('permission:vehicle-edit', ['only' => ['edit', 'update', 'changeStatus']]);
    $this->middleware('permission:vehicle-delete', ['only' => ['destroy']]);

    $this->vehicleModelRepository = $vehicleModelRepository;
  }

  public function index()
  {
    $data = $this->vehicleModelRepository->allModels();
    $parentCategories = $data['categories'];
    $brands = $data['brands'];
    return view('admin.vehicle_models.index', compact('parentCategories', 'brands'));
  }

  public function modelListAll(Request $request)
  {
    $filters = $request->only([
      'vehicle_parent_id',
      'vehicle_category_id',
      'vehicle_brand_id',
    ]);
    return $this->vehicleModelRepository->all($filters, true);
  }

  public function store(VehicleModelRequest $request)
  {
    $data = $request->validated();
    if ($request->hasFile('image')) {
      $data['image'] = $this->uploadImage($request->file('image'), 'models');
    }
    $model = $this->vehicleModelRepository->store($data);
    return $this->successResponse($model, __('locale.Model Uploaded Successfully'));
  }

  public function update(VehicleModelRequest $request, $id)
  {
    $model = $this->vehicleModelRepository->update($id, $request->validated());
    if ($request->hasFile('image')) {
      $path = $this->uploadImage($request->file('image'), 'models', $model->image);
      $model->image = $path;
      $model->save();
    }
    return $this->successResponse($model, __('locale.Model Edited Successfully'));
  }

  public function destroy($id)
  {
    try {
      $this->vehicleModelRepository->delete($id);
      return $this->successResponse($id, __('locale.Model Deleted Successfully'));
    } catch (\Exception $e) {
      return $this->errorResponse(__('locale.Deletion Failed') . ': ' . $e->getMessage(), 500);
    }
  }

  public function changeStatus()
  {
    $model = $this->vehicleModelRepository->toggleStatus(request('id'), request('status'));
    return $this->successResponse($model, __('locale.Model Status Changes Successfully'));
  }

  public function modelDropdown(Request $request)
  {
    $brandId = $request->vehicle_brand_id;
    $categoryId = $request->vehicle_category_id;
    $parentCategoryId = $request->vehicle_parent_category_id;

    if (!$brandId || !$categoryId || !$parentCategoryId) {
      return response()->json(['data' => []]);
    }

    $models = VehicleModel::query()
      ->where('vehicle_brand_id', $brandId)
      ->where('vehicle_category_id', $categoryId);

    if (!empty($parentCategoryId)) {
      $models->where('vehicle_parent_id', $parentCategoryId);
    }

    $models = $models->select('id', 'name')->orderBy('name')->get();

    return response()->json(['data' => $models]);
  }

}
