<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use App\Http\Requests\VehicleCategoryRequest;
use App\Http\Requests\PriceRequest;
use App\Repositories\Contracts\VehicleCategoryRepositoryInterface;
use App\Traits\JsonResponseTrait;
use App\Traits\ImageUploadTrait;
use Illuminate\Http\Request;

class VehicleCategoryController extends Controller
{
  use JsonResponseTrait, ImageUploadTrait;

  protected $vehicleCategoryRepository;

  public function __construct(VehicleCategoryRepositoryInterface $vehicleCategoryRepository)
  {
    $this->middleware('permission:vehicle-list|vehicle-create|vehicle-edit|vehicle-delete', ['only' => ['index', 'show', 'categoryListAll', 'categoryType', 'categories', 'subcategories', 'getCountryPrices', 'getCountryPricesRule']]);
    $this->middleware('permission:vehicle-create', ['only' => ['create', 'store', 'storePrice']]);
    $this->middleware('permission:vehicle-edit', ['only' => ['edit', 'update', 'changeStatus']]);
    $this->middleware('permission:vehicle-delete', ['only' => ['destroy']]);

    $this->vehicleCategoryRepository = $vehicleCategoryRepository;
  }

  public function index()
  {
    $data = $this->vehicleCategoryRepository->allCategories();
    $allCategories = $data['categories'];
    $countries = $data['countries'];
    return view('admin.vehicle_categories.index', compact('allCategories', 'countries'));
  }

  public function categoryType()
  {
    $data = $this->vehicleCategoryRepository->allCategories();
    $allCategories = $data['categories'];
    $countries = $data['countries'];
    return view('admin.vehicle_categories.types', compact('allCategories', 'countries'));
  }

  public function categoryListAll(Request $request)
  {
    $query = $request->query('type') ?? null;
    $categories = $this->vehicleCategoryRepository->allList($query);
    return \DataTables::of($categories)->addIndexColumn()->make(true);
  }

  public function store(VehicleCategoryRequest $request)
  {
    $data = $request->validated();
    if ($request->hasFile('image')) {
      $data['image'] = $this->uploadImage($request->file('image'), 'categories');
    }
    if ($request->hasFile('map_icon')) {
      $data['map_icon'] = $this->uploadImage($request->file('map_icon'), 'categories');
    }
    $category = $this->vehicleCategoryRepository->store($data);
    return $this->successResponse($category, __('locale.Category Uploaded Successfully'));
  }

  public function update(VehicleCategoryRequest $request, $id)
  {
    $category = $this->vehicleCategoryRepository->updateCategory($id, $request->validated());
    if ($request->hasFile('image')) {
      $path = $this->uploadImage($request->file('image'), 'categories', $category->image);
      $category->image = $path;
      $category->save();
    }
    if ($request->hasFile('map_icon')) {
      $pathIcon = $this->uploadImage($request->file('map_icon'), 'categories', $category->image);
      $category->map_icon = $pathIcon;
      $category->save();
    }
    return $this->successResponse($category->load('parent'), __('locale.Category Edited Successfully'));
  }

  public function destroy($id)
  {
    try {
      $this->vehicleCategoryRepository->delete($id);
      return $this->successResponse($id, __('locale.Category Deleted Successfully'));
    } catch (\Exception $e) {
      return $this->errorResponse(__('locale.Deletion Failed') . ': ' . $e->getMessage(), 500);
    }
  }

  public function changeStatus()
  {
    $category = $this->vehicleCategoryRepository->toggleStatus(request('id'), request('status'));
    return $this->successResponse($category, __('locale.Category Status Changes Successfully'));
  }

  public function categories(Request $request)
  {
    return $this->vehicleCategoryRepository->subCategories(null, $request->input('type'));
  }
  public function subcategories($id)
  {
    return $this->vehicleCategoryRepository->subCategories($id, null);
  }
  public function getCountryPrices($categoryId)
  {
    $data = $this->vehicleCategoryRepository->getPricesByCategory($categoryId);
    return $this->successResponse($data, __('locale.Data fetched successfully.'));
  }

  public function getCountryPricesRule($categoryId)
  {
    $data = $this->vehicleCategoryRepository->getPricesRuleByCategory($categoryId);
    $category = $data['category'];
    $countries = $data['countries'];
    $html = view('_partials.category_pricing', compact('category', 'countries'))->render();
    return $this->successResponse($html, __('locale.Data fetched successfully.'));
  }
  public function storePrice(PriceRequest $request)
  {
    $this->vehicleCategoryRepository->syncCountryPrices(
      $request->vehicle_category_id,
      $request->input('prices', [])
    );
    return $this->successResponse([], __('locale.Prices saved successfully!'));
  }



}
