<?php
namespace App\Http\Controllers\admin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Requests\PasswordRequest;
use App\Models\User;
use App\Traits\JsonResponseTrait;
use App\Repositories\Contracts\UserRepositoryInterface;
use App\Repositories\Contracts\ReviewRepositoryInterface;
use App\Http\Requests\UserRequest;
use Stripe\PaymentMethod;
use Stripe\Stripe;
use Stripe\Exception\InvalidRequestException;

class UserController extends Controller
{
  use JsonResponseTrait;
  protected $userRepository;
  protected $reviewRepository;
  function __construct(UserRepositoryInterface $userRepository, ReviewRepositoryInterface $reviewRepository)
  {
    $this->middleware('permission:user-list|user-create|user-edit|user-delete', ['only' => ['index', 'show', 'userList']]);
    $this->middleware('permission:user-create', ['only' => ['create', 'store']]);
    $this->middleware('permission:user-edit', ['only' => ['edit', 'update', 'changeStatus', 'setPassword']]);
    $this->middleware('permission:user-delete', ['only' => ['destroy']]);
    $this->userRepository = $userRepository;
    $this->reviewRepository = $reviewRepository;
    Stripe::setApiKey(config('cashier.secret'));
  }
  public function show(Request $request, $id)
  {
    $user = $this->userRepository->find($id);
    if (!$user) {
      abort(404); // no such user at all
    }
    if ($user->trashed()) {
      return redirect()->route('users.index')
        ->with('error', __('locale.This user account has been deleted.'));
    }
    $rideCount = \App\Models\Ride::where('user_id', $id)->count();
    $orderCount = \App\Models\Order::where('user_id', $id)->count();
    $paymentMethods = collect(); // default empty
    try {
      $paymentMethods = PaymentMethod::all([
        'customer' => $user->stripe_id,
        'type' => 'card',
      ]);
    } catch (InvalidRequestException $e) {

    }
    $reviewData = $this->reviewRepository->allReview($id, 'user');
    $walletId = \App\Models\Wallet::where('owner_type', 'user')->where('owner_id', $id)->first()?->id;
    $breadcrumbs = [
      'Home' => route('admin.dashboard'),
      'User' => route('users.index'),
      $user->name => null, // last one is active
    ];
    return view('/admin/users/show', compact('user', 'id', 'paymentMethods', 'rideCount', 'orderCount', 'breadcrumbs', 'walletId', 'reviewData'));
  }


  public function index()
  {
    $userStats = User::selectRaw("
        COUNT(*) as totalUser,
        COUNT(email_verified_at) as verified,
               SUM(CASE WHEN email_verified_at IS NULL THEN 1 ELSE 0 END) as emailNotVerified,
        SUM(CASE WHEN is_verified = 0 THEN 1 ELSE 0 END) as notVerified,
        SUM(CASE WHEN status = 0 THEN 1 ELSE 0 END) as userDisabled
    ")
      ->where('user_type', 'User')
      ->first();

    return view('admin.users.index', [
      'totalUser' => $userStats->totalUser,
      'verified' => $userStats->verified,
      'notVerified' => $userStats->notVerified,
      'userDisabled' => $userStats->userDisabled,
    ]);
  }

  public function userListAll(Request $request)
  {
    return $this->userRepository->allList($request);
  }

  public function edit($id)
  {
    $user = User::findOrFail($id);
    return response()->json($user);
  }

  public function store(UserRequest $request)
  {
    $result = $this->userRepository->store($request);
    $id = $result['id'];
    $user = $this->userRepository->find($id);
    if ($result['form'] == 'Updated') {
      return $this->successResponse($user, __('locale.User Updated Successfully!'));
    } else {
      return $this->successResponse($user, __('locale.User Created Successfully!'));
    }

  }

  public function destroy($id)
  {
    $this->userRepository->softDelete($id);
    return $this->successResponse($id, __('locale.User Deleted Successfully!'));
  }

  public function changeStatus(Request $request)
  {
    $id = $request->post('id');
    $status = $request->post('status');
    $this->userRepository->changeStatus($id, $status);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Status Changed Successfully'));
  }

  public function setPassword(PasswordRequest $request)
  {
    $data = $request->validated();
    $data['id'] = $request->id;
    $user = $this->userRepository->setPassword($data);
    return $this->successResponse(null, __('locale.Password reset successfully'));
  }


}
