<?php
namespace App\Http\Controllers\admin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Store;
use App\Traits\JsonResponseTrait;
use App\Repositories\Contracts\StoreRepositoryInterface;
use App\Repositories\Contracts\CategoryRepositoryInterface;
use App\Repositories\Contracts\ReviewRepositoryInterface;
use App\Http\Requests\StoreRequest;
class StoreController extends Controller
{
  use JsonResponseTrait;
  protected $storeRepository;
  protected $categoryRepository;
  protected $reviewRepository;
  function __construct(StoreRepositoryInterface $storeRepository, CategoryRepositoryInterface $categoryRepository, ReviewRepositoryInterface $reviewRepository)
  {
    if (auth()->check() && auth()->user()->user_type === 'admin') {
      $this->middleware('permission:store-list|store-create|store-edit|store-delete', ['only' => ['index', 'show', 'storeListAll', 'showDetails', 'storeMaps', 'impersonate', 'stopImpersonate', 'availability', 'settings']]);
      $this->middleware('permission:store-create', ['only' => ['create', 'store']]);
      $this->middleware('permission:store-edit', ['only' => ['edit', 'update', 'changeStatus', 'changePopular', 'storeStatus', 'availabilityUpdate', 'storePaymentStatus']]);
      $this->middleware('permission:store-delete', ['only' => ['destroy']]);
    }
    $this->storeRepository = $storeRepository;
    $this->categoryRepository = $categoryRepository;
    $this->reviewRepository = $reviewRepository;
  }
  public function index()
  {
    $userStats = User::selectRaw("
        COUNT(*) as totalUser,
        COUNT(email_verified_at) as verified,
        SUM(CASE WHEN email_verified_at IS NULL THEN 1 ELSE 0 END) as emailNotVerified,
        SUM(CASE WHEN is_verified = 0 THEN 1 ELSE 0 END) as notVerified,
        SUM(CASE WHEN status = 0 THEN 1 ELSE 0 END) as userDisabled
    ")
      ->where('user_type', 'Store')
      ->first();

    return view('admin.stores.index', [
      'totalUser' => $userStats->totalUser,
      'verified' => $userStats->verified,
      'notVerified' => $userStats->notVerified,
      'userDisabled' => $userStats->userDisabled,
    ]);
  }

  public function storeListAll(Request $request)
  {
    return $this->storeRepository->allList($request);
  }

  public function storeMaps()
  {
    $stores = $this->storeRepository->allMaps();
    return view('admin.stores.maps', compact('stores'));
  }

  public function showDetails($id)
  {
    $user = $this->storeRepository->find($id);
    return view('_partials\_modals\modal-store-approval', compact('user'));

  }

  public function show(Request $request, $id)
  {
    $user = $this->storeRepository->find($id);
    $reviewData = $this->reviewRepository->allReview($id, 'store');
    $breadcrumbs = [
      'Home' => route('admin.dashboard'),
      'Store' => route('stores.index'),
      $user->name => null, // last one is active
    ];
    return view('/admin/stores/show', compact('user', 'id', 'reviewData', 'breadcrumbs'));
  }

  public function getStores(Request $request)
  {
    $store = $this->storeRepository->getStores($request);
    return $this->successResponse($store, __(''));
  }

  public function create()
  {
    $categories = $this->categoryRepository->dropdown();
    return view('/admin/stores/create', compact('categories'));
  }

  public function edit(int $id)
  {
    $categories = $this->categoryRepository->dropdown();
    $store = Store::with('user', 'categories')->findOrFail($id);
    return view('admin.stores.create', compact('store', 'categories'));
  }

  public function store(StoreRequest $request)
  {
    $store = $this->storeRepository->store($request, null);
    return $this->successResponse($store, __('locale.Store Created Successfully!'));
  }

  public function update(StoreRequest $request, int $id)
  {
    $store = $this->storeRepository->store($request, $id);
    return $this->successResponse($store, __('locale.Store Updated Successfully!'));
  }

  public function changeStatus(Request $request, $id)
  {
    $status = $request->post('status');
    $this->storeRepository->changeStatus($id, $status);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Store Status Changed Successfully'));
  }


  public function changePopular(Request $request)
  {
    $status = $request->post('status');
    $id = $request->post('id');
    $this->storeRepository->changePopular($id, $status);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Status Changed Successfully'));
  }

  public function impersonate($storeId)
  {
    $this->storeRepository->impersonate($storeId);
    // Redirect to store dashboard
    return redirect()->route('store.dashboard');
  }

  public function stopImpersonate()
  {
    $this->storeRepository->stopImpersonate();
    return redirect()->route('admin.dashboard');
  }

  public function profile()
  {
    $storeId = User::find(auth()->id())?->storeId;
    $categories = $this->categoryRepository->dropdown();
    $store = Store::with('user', 'categories')->findOrFail($storeId);
    return view('admin.stores.create', compact('store', 'categories'));
  }

  public function storeProfile(StoreRequest $request)
  {
    $storeId = User::find(auth()->id())?->storeId;
    $store = $this->storeRepository->store($request, $storeId);
    return $this->successResponse($store, __('locale.Store Updated Successfully!'));
  }

  public function storeStatus(Request $request)
  {
    $id = $request->post('id');
    $status = $request->post('status');
    $this->storeRepository->storeStatus($id, $status);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Store Status Changed Successfully'));
  }

  public function availability()
  {
    $storeId = User::find(auth()->id())?->storeId;
    $store = Store::with('businessHours')->findOrFail($storeId);
    return view('store.availability', compact('store'));
  }

  public function availabilityUpdate(Request $request)
  {
    $this->storeRepository->availabilityUpdate($request);
    return $this->successResponse([], __('locale.Store Availability updated successfully'));

  }

  public function settings()
  {
    return view('/store/settings');
  }

  public function storePaymentStatus(Request $request, $type)
  {
    $id = $request->post('id');
    $status = $request->post('status');
    $this->storeRepository->storePaymentStatus($id, $status, $type);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Store Payment Status Changed Successfully'));
  }

}
