<?php
namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\RidePricingRule;
use App\Models\Country;
use App\Traits\JsonResponseTrait;
use App\Http\Requests\RidePricingRuleRequest;
use Illuminate\Support\Facades\DB;
class RidePricingRuleController extends Controller
{
  use JsonResponseTrait;
  function __construct()
  {
    $this->middleware('permission:site_Settings-list|site_Settings-create|site_Settings-edit|site_Settings-delete', ['only' => ['index']]);
    $this->middleware('permission:site_Settings-create', ['only' => ['create', 'store']]);
    $this->middleware('permission:site_Settings-edit', ['only' => ['edit', 'update']]);
    $this->middleware('permission:site_Settings-delete', ['only' => ['destroy']]);
  }
  public function index()
  {
    $countries = Country::where('status', 1)->get();
    $rules = RidePricingRule::with(['country', 'conditions'])->orderByDesc('priority')->get();
    return view('admin.ride_pricing_rules.index', compact('rules', 'countries'));
  }

  public function store(RidePricingRuleRequest $request)
  {
    DB::transaction(function () use ($request) {
      $rule = RidePricingRule::create($request->only([
        'country_id',
        'rule_name',
        'price_type',
        'amount',
        'is_active',
        'priority'
      ]));

      foreach ($request->input('conditions', []) as $condition) {
        $type = $condition['type'] ?? null;

        $value = $condition['value'] ?? null;
        if (!is_string($value)) {
          $value = json_encode($value);
        }

        $rule->conditions()->create([
          'condition_type' => $type,
          'condition_value' => $value,
        ]);
      }
    });

    return $this->successResponse([], __('locale.Rule created successfully'));
  }

  public function update(RidePricingRuleRequest $request, $id)
  {
    return DB::transaction(function () use ($request, $id) {
      $rule = RidePricingRule::findOrFail($id);

      // update main fields
      $rule->update($request->only([
        'country_id',
        'rule_name',
        'price_type',
        'amount',
        'is_active',
        'priority',
      ]));

      // clear old conditions
      $rule->conditions()->delete();

      foreach ($request->input('conditions', []) as $condition) {
        $type = $condition['type'] ?? null;

        $value = $condition['value'] ?? null;
        if (!is_string($value)) {
          $value = json_encode($value);
        }

        $rule->conditions()->create([
          'condition_type' => $type,
          'condition_value' => $value,
        ]);
      }

      return $this->successResponse($request->all(), __('locale.Rule updated successfully'));
    });
  }




  public function destroy($id)
  {
    $rule = RidePricingRule::findOrFail($id);
    DB::transaction(function () use ($rule) {
      $rule->conditions()->delete();
      $rule->delete();
    });
    return $this->successResponse([], __('locale.Rule deleted successfully'));
  }
}
