<?php
namespace App\Http\Controllers\admin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Store;
use App\Models\Product;
use App\Models\Category;
use App\Models\ProductImage;
use App\Models\User;
use App\Traits\JsonResponseTrait;
use App\Repositories\Contracts\ProductRepositoryInterface;
use App\Repositories\Contracts\CategoryRepositoryInterface;
use App\Http\Requests\ProductRequest;
use App\Repositories\Contracts\ReviewRepositoryInterface;
use Illuminate\Support\Facades\Storage;
class ProductController extends Controller
{
  use JsonResponseTrait;
  protected $productRepository, $reviewRepository;
  protected $categoryRepository;
  function __construct(ProductRepositoryInterface $productRepository, CategoryRepositoryInterface $categoryRepository, ReviewRepositoryInterface $reviewRepository)
  {
    if (auth()->check() && auth()->user()->user_type === 'admin') {
      $this->middleware('permission:product-list|product-create|product-edit|product-delete', ['only' => ['index', 'show', 'productListAll']]);
      $this->middleware('permission:product-create', ['only' => ['create', 'product', '']]);
      $this->middleware('permission:product-edit', ['only' => ['edit', 'update', 'changeStatus', 'changeStock', 'deleteImage']]);
      $this->middleware('permission:product-delete', ['only' => ['destroy', '']]);
    }
    $this->productRepository = $productRepository;
    $this->categoryRepository = $categoryRepository;
    $this->reviewRepository = $reviewRepository;
  }
  public function index()
  {
    $user = User::find(auth()->id());
    $storeId = $user->user_type == 'Store' && $user->storeId ? $user->storeId : '';
    $productStats = Product::selectRaw("
        COUNT(*) as totalProduct,
        SUM(CASE WHEN status= 1 THEN 1 ELSE 0 END) as enabled,
        SUM(CASE WHEN status = 0 THEN 1 ELSE 0 END) as disabled
    ")->when(!empty($storeId), fn($q) => $q->where('store_id', $storeId))
      ->first();

    // Categories (with optional store filter)
    $categories = cacheRemember("dropdown_store_categories_{$storeId}", 600, function () use ($storeId) {
      return Category::join('products', 'categories.id', '=', 'products.category_id')
        ->when(!empty($storeId), fn($q) => $q->where('products.store_id', $storeId))
        ->select('categories.id', 'categories.name')
        ->distinct()
        ->orderBy('categories.name')
        ->pluck('categories.name', 'categories.id');
    }, ['dropdown']);

    // Parent categories (distinct)
    $parentCategories = cacheRemember("dropdown_store_parent_categories_{$storeId}", 600, function () use ($storeId) {
      return Category::join('products', 'categories.id', '=', 'products.category_id')
        ->join('categories as parents', 'categories.parent_id', '=', 'parents.id')
        ->when(!empty($storeId), fn($q) => $q->where('products.store_id', $storeId))
        ->select('parents.id', 'parents.name as parent_category')
        ->distinct()
        ->orderBy('parents.name')
        ->pluck('parent_category', 'parents.id');
    }, ['dropdown']);



    return view('store.products.index', [
      'totalProduct' => $productStats->totalProduct,
      'enabled' => $productStats->enabled,
      'disabled' => $productStats->disabled,
      'parentCategories' => $parentCategories,
      'categories' => $categories,
    ]);
  }

  public function productListAll(Request $request)
  {
    return $this->productRepository->allList($request);
  }

  public function show(Request $request, $id)
  {
    $product = $this->productRepository->find($id);
    $user = User::find(auth()->id());
    $storeId = $user->user_type == 'Store' && $user->storeId ? $user->storeId : '';
    if ($user->user_type == 'Store' && $product->store_id != $storeId) {
      abort(403);
    }
    $reviewData = $this->reviewRepository->allReview($id, 'product');
    // Determine routes based on user type
    $dashboardRoute = $user->user_type == 'Store'
      ? route('store.dashboard')
      : route('admin.dashboard');

    $productIndexRoute = $user->user_type == 'Store'
      ? url('store/products')
      : route('products.index');

    $breadcrumbs = [
      'Home' => $dashboardRoute,
      'Product' => $productIndexRoute,
      $product->title => null, // last one is active
    ];
    return view('/store/products/show', compact('product', 'id', 'reviewData', 'breadcrumbs'));
  }
  public function create()
  {
    $categories = Store::where('user_id', auth()->id())->first()?->categories()->whereNotNull('parent_id') // Only parent categories
      ->get();
    $parentCategoryIds = $categories->pluck('parent_id')->unique();
    $parentCategories = Category::whereIn('id', $parentCategoryIds)->get();
    return view('/store/products/create', compact('categories', 'parentCategories'));
  }
  public function edit(int $id)
  {

    $categories = Store::where('user_id', auth()->id())->first()?->categories()->whereNotNull('parent_id') // Only parent categories
      ->get();
    $parentCategoryIds = $categories->pluck('parent_id')->unique();
    $parentCategories = Category::whereIn('id', $parentCategoryIds)->get();
    $product = Product::with('store', 'images')->findOrFail($id);
    return view('store.products.create', compact('product', 'categories', 'parentCategories'));
  }
  public function getProductsByStore(Request $request)
  {
    $product = $this->productRepository->getProductsByStore($request);
    return $this->successResponse($product, __(''));
  }
  public function getProductsByCategory(Request $request)
  {
    $product = $this->productRepository->getProductsByCategory($request);
    return $this->successResponse($product, __(''));
  }
  public function store(ProductRequest $request)
  {
    $product = $this->productRepository->store($request, null);
    return $this->successResponse($product, __('locale.Product Created Successfully!'));
  }
  public function update(ProductRequest $request, int $id)
  {
    $product = $this->productRepository->store($request, $id);
    return $this->successResponse($product, __('locale.Product Updated Successfully!'));
  }

  public function changeStatus(Request $request)
  {
    $id = $request->post('id');
    $status = $request->post('status');
    $this->productRepository->toggleStatus($id, $status);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Product Status Changed Successfully'));
  }

  public function changeStock(Request $request)
  {
    $id = $request->post('id');
    $status = $request->post('status');
    $this->productRepository->changeStock($id, $status);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Product Stock Status Changed Successfully'));
  }

  public function destroy($id)
  {
    $this->productRepository->delete($id);
    return $this->successResponse($id, __('locale.Product Deleted Successfully!'));
  }

  public function uploadTempImage(Request $request)
  {
    if ($request->hasFile('file')) {
      $path = $request->file('file')->store('products/temp', 'public');
      return $this->successResponse(['filename' => $path], __(''));
    }
    return $this->errorResponse(__('locale.Something went wrong'), 422);
  }

  public function deleteTemp(Request $request)
  {
    Storage::disk('public')->delete($request->filename);
    return response()->json(['success' => true]);
  }

  public function deleteImage(Request $request, $id)
  {
    $image = ProductImage::findOrFail($id);
    Storage::disk('public')->delete($image->file);
    $image->delete();
    return $this->successResponse([], __('locale.Image Delete successfully'));
  }

  public function loadImages(Request $request)
  {
    $images = ProductImage::where('product_id', $request->product_id)->get();
    $data = $images->map(function ($img) {
      return [
        'id' => $img->id,
        'file' => asset('storage/' . $img->file),
      ];
    });
    return $this->successResponse($data, __(''));
  }

}
