<?php
namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use App\Models\KanbanBoard;
use App\Models\KanbanTask;
use Illuminate\Support\Str;
use App\Traits\JsonResponseTrait;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;
class KanbanController extends Controller
{

  use JsonResponseTrait;
  function __construct()
  {
    $this->middleware('permission:kanban-list|kanban-create|kanban-edit|kanban-delete', ['only' => ['index', 'show', 'fetch']]);
    $this->middleware('permission:kanban-create', ['only' => ['create', 'store', 'storeBoard', 'storeTask']]);
    $this->middleware('permission:kanban-edit', ['only' => ['edit', 'update', 'changeStatus', 'updateTask']]);
    $this->middleware('permission:kanban-delete', ['only' => ['deleteTask', 'deleteBoard']]);
  }
  public function index()
  {
    return view('common.app-kanban');
  }

  public function fetch()
  {
    $boards = KanbanBoard::with('tasks')->get();
    $response = $boards->map(function ($board) {
      return [
        'id' => $board->slug,
        'title' => $board->title,
        'item' => $board->tasks->map(function ($task) {
          return [
            'id' => $task->slug ?? ('task-' . $task->id), // optional: if you store slugs
            'title' => $task->title,
            'comments' => $task->comments ?? '0', // add real or dummy data
            'badge-text' => $task->label ?? 'Notes',
            'badge' => $task->label_color ?? 'secondary',
            'due-date' => optional($task->due_date)->format('j F') ?? '',
            'image' => $task->attachments ? asset('storage/Files/' . $task->attachments) : '',
            'attachments' => $task->attachments ? 1 : 0,
            'assigned' => $task->assigned ?? [],  // should be array of image names
            'members' => $task->members ?? []      // should be array of names
          ];
        })->toArray()
      ];
    });

    return response()->json($response);
  }

  public function storeBoard(Request $request)
  {
    $slug = Str::slug($request->title);
    $input = $request->all();
    $input['slug'] = $slug;
    // Validate input (including unique slug)
    $validator = Validator::make($input, [
      'title' => 'required|string|max:255',
      'slug' => 'required|unique:kanban_boards,slug'
    ]);
    if ($validator->fails()) {
      $errors = $validator->errors()->all();
      $message = implode(' ', $errors);
      return $this->errorResponse($message, 422);
    }
    $board = KanbanBoard::create(['title' => $request->title, 'slug' => $slug]);
    return $this->successResponse($board, __('locale.Kanban Board added successfully.'));
  }

  public function storeTask(Request $request)
  {
    $validator = Validator::make($request->all(), [
      'title' => 'required|string|max:255',
      'board_id' => 'required|exists:kanban_boards,slug'
    ]);
    if ($validator->fails()) {
      $errors = $validator->errors()->all(); // array of error messages
      $message = implode(' ', $errors);      // combine into a single string
      return $this->errorResponse($message, 422); // use helper
    }
    $board = KanbanBoard::where('slug', $request->board_id)->firstOrFail();
    $task = KanbanTask::create([
      'title' => $request->title,
      'kanban_board_id' => $board->id
    ]);

    return response()->json($task); // include id and title in JSON
  }

  public function updateTask(Request $request, $id)
  {
    $validated = $request->validate([
      'title' => 'required|string|max:255',
      'due_date' => 'nullable|date',
      'label' => 'nullable|string|max:100',
      'label_color' => 'nullable|string',
      'comments' => 'nullable|string',
      'attachment' => 'nullable|image|max:2048'
    ]);

    $task = KanbanTask::findOrFail($id);
    $task->fill($validated);
    $task->save();

    // Handle new attachments
    $uploadedFiles = [];
    if (!empty($request->file('attachment'))) {
      $manager = new ImageManager(new Driver());
      $image = $request->file('attachment');
      $imagename = 'KB-' . time() . '.' . $image->extension();
      $destinationPath = public_path('storage/Files/');
      $manager->read($image)->save($destinationPath . $imagename);
      $task->attachments = $imagename;
    }
    $task->save();

    return $this->successResponse($task, __('locale.Task updated successfully'));

  }

  public function deleteTask($id)
  {
    $id = str_replace('task-', '', $id);
    KanbanTask::findOrFail($id)->delete();
    return response()->json(['success' => true]);
  }

  public function deleteBoard($slug)
  {
    KanbanBoard::where('slug', $slug)->delete();
    return response()->json(['success' => true]);
  }
}
