<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Auth;
use App\Models\User;
use App\Models\Order;
use App\Models\Product;
use App\Models\Transaction;
use App\Models\Ride;
use App\Models\Wallet;
use Carbon\Carbon;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\DB;

class HomeController extends Controller
{


  public function logout(Request $request)
  {
    $user = Auth::user(); // store user before logout
    if ($user) {
      $user->updatePushSubscription()->delete(); // remove push subscription
    }
    Auth::guard('web')->logout();
    // Invalidate session and regenerate CSRF token
    $request->session()->invalidate();
    $request->session()->regenerateToken();
    return redirect()->route('index');
  }

  public function login()
  {
    if (Auth::check()) {
      $user = Auth::user();
      $redirectRoute = match ($user->user_type) {
        'Admin' => route('admin.dashboard'),
        'Store' => route('store.dashboard'),
        'User', 'Driver' => route('frontend'),
        default => route('dashboard'), // fallback
      };

      return redirect($redirectRoute);
    } else {
      $pageConfigs = ['pageHeader' => false];
      return view('/auth/login', ['pageConfigs' => $pageConfigs]);
    }
  }

  public function dashboard()
  {
    // Cache users stats
    $userStats = cacheRemember('dashboard_user_stats', 60, function () {
      return User::selectRaw("
            COUNT(*) as totalUser,
            COUNT(email_verified_at) as verified,
            SUM(CASE WHEN email_verified_at IS NULL THEN 1 ELSE 0 END) as emailNotVerified,
            SUM(CASE WHEN is_verified = 0 THEN 1 ELSE 0 END) as notVerified,
            SUM(CASE WHEN status = 0 THEN 1 ELSE 0 END) as userDisabled
        ")->where('user_type', 'User')->first();
    }, ['dashboard']);

    // Cache drivers stats
    $driverStats = cacheRemember('dashboard_driver_stats', 60, function () {
      return User::selectRaw("
            COUNT(*) as totalUser,
            COUNT(email_verified_at) as emailVerified,
            SUM(CASE WHEN is_verified = 1 THEN 1 ELSE 0 END) as verified,
            SUM(CASE WHEN email_verified_at IS NULL THEN 1 ELSE 0 END) as emailNotVerified,
            SUM(CASE WHEN is_verified = 0 THEN 1 ELSE 0 END) as notVerified,
            SUM(CASE WHEN status = 0 THEN 1 ELSE 0 END) as userDisabled,
            SUM(CASE WHEN drivers.is_active = 1 AND is_verified = 1 THEN 1 ELSE 0 END) AS activeDrivers
        ")
        ->leftJoin('drivers', 'drivers.user_id', '=', 'users.id')
        ->where('user_type', 'Driver')
        ->first();
    }, ['dashboard']);

    // Cache stores stats
    $storeStats = cacheRemember('dashboard_store_stats', 60, function () {
      return User::selectRaw("
            COUNT(*) as totalUser,
            SUM(CASE WHEN stores.is_active = 1 THEN 1 ELSE 0 END) as activeStores
        ")
        ->leftJoin('stores', 'stores.user_id', '=', 'users.id')
        ->where('user_type', 'Store')
        ->first();
    }, ['dashboard']);

    // Latest transactions (cache for 30 seconds)
    $transactions = cacheRemember('dashboard_transactions', 60, function () {
      return Transaction::latest()->limit(10)->get();
    }, ['dashboard']);

    // Latest ongoing rides (cache for 10 seconds, very dynamic)
    $rides = cacheRemember('dashboard_ongoing_rides', 10, function () {
      return Ride::where('status', 'Ongoing')->latest()->limit(10)->get();
    }, ['dashboard']);

    return view('admin.dashboard', [
      'totalUser' => $userStats->totalUser,
      'verified' => $userStats->verified,
      'notVerified' => $userStats->notVerified,
      'userDisabled' => $userStats->userDisabled,
      'totalDriver' => $driverStats->totalUser,
      'driverVerified' => $driverStats->verified,
      'driverActive' => $driverStats->activeDrivers,
      'totalStore' => $storeStats->totalUser,
      'storeActive' => $storeStats->activeStores,
      'transactions' => $transactions,
      'rides' => $rides,
    ]);
  }

  public function getOrdersLastWeek()
  {
    return cacheRemember('orders_last_week', 600, function () {

      $now = Carbon::now();

      // Current week range (last 7 days)
      $startDate = $now->copy()->subDays(6)->startOfDay();
      $endDate = $now->copy()->endOfDay();

      // Previous week range
      $prevStart = $startDate->copy()->subDays(7);
      $prevEnd = $startDate->copy()->subSecond();

      // Query current week counts grouped by day
      $orders = DB::ordersWithoutPendingOnline()
        ->selectRaw("DAYNAME(created_at) as day, COUNT(*) as total")
        ->whereBetween('created_at', [$startDate, $endDate])
        ->groupBy('day')
        ->get()
        ->pluck('total', 'day');

      // Prepare fixed day order for the chart
      $days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
      $series = [];
      foreach ($days as $day) {
        $series[] = $orders[$day] ?? 0;
      }

      // Total orders last week
      $totalLastWeek = array_sum($series);

      // Total orders previous week
      $prevWeekTotal = DB::ordersWithoutPendingOnline()
        ->whereBetween('created_at', [$prevStart, $prevEnd])
        ->count();

      // Calculate percentage change
      $percentageChange = $prevWeekTotal > 0
        ? (($totalLastWeek - $prevWeekTotal) / $prevWeekTotal) * 100
        : 0;

      return [
        'categories' => ['M', 'T', 'W', 'T', 'F', 'S', 'S'],
        'series' => $series,
        'total' => $totalLastWeek,
        'percentage' => round($percentageChange, 1) // e.g., 12.6
      ];

    }, ['dashboard']); // tag for easy flushing
  }

  public function getSalesLastYear()
  {
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));

    return cacheRemember('sales_last_year', 600, function () use ($currency) {

      $now = Carbon::now();

      // Current year range
      $startDate = $now->copy()->startOfYear();
      $endDate = $now->copy()->endOfYear();

      // Previous year range
      $prevStart = $startDate->copy()->subYear();
      $prevEnd = $endDate->copy()->subYear();

      // Monthly sales for last year
      $salesData = DB::ordersWithoutPendingOnline()
        ->selectRaw('MONTH(created_at) as month, SUM(grand_total) as total')
        ->whereBetween('created_at', [$startDate, $endDate])
        ->groupBy('month')
        ->orderBy('month')
        ->pluck('total', 'month');

      // Prepare series for all 12 months
      $series = [];
      for ($m = 1; $m <= 12; $m++) {
        $series[] = round($salesData[$m] ?? 0, 2);
      }

      // Total sales for last year
      $totalSalesLastYear = array_sum($series);

      // Total sales for previous year
      $prevYearTotal = DB::ordersWithoutPendingOnline()
        ->whereBetween('created_at', [$prevStart, $prevEnd])
        ->sum('grand_total');

      // Calculate percentage change
      $percentageChange = $prevYearTotal > 0
        ? (($totalSalesLastYear - $prevYearTotal) / $prevYearTotal) * 100
        : 0;

      return [
        'series' => $series,
        'total' => $currency . $totalSalesLastYear,
        'percentage' => round($percentageChange, 1) // e.g., -16.2
      ];

    }, ['dashboard']); // tag for easy cache clearing
  }

  public function getRidesDashboardData()
  {
    $currency = cache()->remember('currency_symbol', 3600, fn() => Helpers::setting('currency_symbol', 'currency'));

    return cacheRemember('rides_dashboard_data', 600, function () use ($currency) {

      $now = Carbon::now();

      /*
      |----------------------------------------------------------------------
      | 1. Rides Last Week
      |----------------------------------------------------------------------
      */
      $startDate = $now->copy()->subDays(6)->startOfDay();
      $endDate = $now->copy()->endOfDay();

      $prevStart = $startDate->copy()->subDays(7);
      $prevEnd = $startDate->copy()->subSecond();

      $ridesWeekData = DB::table('rides')
        ->selectRaw("DAYNAME(created_at) as day, COUNT(*) as total")
        ->whereBetween('created_at', [$startDate, $endDate])
        ->groupBy('day')
        ->pluck('total', 'day');

      $days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
      $ridesWeekSeries = [];
      foreach ($days as $day) {
        $ridesWeekSeries[] = $ridesWeekData[$day] ?? 0;
      }

      $totalRidesLastWeek = array_sum($ridesWeekSeries);

      $prevWeekRides = DB::table('rides')
        ->whereBetween('created_at', [$prevStart, $prevEnd])
        ->count();

      $ridesWeekPercentage = $prevWeekRides > 0
        ? (($totalRidesLastWeek - $prevWeekRides) / $prevWeekRides) * 100
        : 0;

      /*
      |----------------------------------------------------------------------
      | 2. Ride Sales Last Year
      |----------------------------------------------------------------------
      */
      $salesStart = $now->copy()->startOfYear();
      $salesEnd = $now->copy()->endOfYear();

      $prevSalesStart = $salesStart->copy()->subYear();
      $prevSalesEnd = $salesEnd->copy()->subYear();

      $ridesSalesData = DB::table('rides')
        ->selectRaw('MONTH(created_at) as month, SUM(fare) as total')
        ->whereBetween('created_at', [$salesStart, $salesEnd])
        ->groupBy('month')
        ->orderBy('month')
        ->pluck('total', 'month');

      $salesSeries = [];
      for ($m = 1; $m <= 12; $m++) {
        $salesSeries[] = round($ridesSalesData[$m] ?? 0, 2);
      }

      $totalSalesLastYear = array_sum($salesSeries);

      $prevYearSales = DB::table('rides')
        ->whereBetween('created_at', [$prevSalesStart, $prevSalesEnd])
        ->sum('fare');

      $salesPercentage = $prevYearSales > 0
        ? (($totalSalesLastYear - $prevYearSales) / $prevYearSales) * 100
        : 0;

      /*
      |----------------------------------------------------------------------
      | Return Combined Data
      |----------------------------------------------------------------------
      */
      return [
        'rides_last_week' => [
          'categories' => ['M', 'T', 'W', 'T', 'F', 'S', 'S'],
          'series' => $ridesWeekSeries,
          'total' => $totalRidesLastWeek,
          'percentage' => round($ridesWeekPercentage, 1)
        ],
        'sales_last_year' => [
          'series' => $salesSeries,
          'total' => $currency . round($totalSalesLastYear, 2),
          'percentage' => round($salesPercentage, 1)
        ]
      ];

    }, ['dashboard']); // tag for easy cache clearing
  }

  public function getEarningReports()
  {
    return cacheRemember('earning_reports', 600, function () {
      $months = range(1, 9); // Jan–Sep, adjust if you want 12 months
      $monthNames = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep'];

      // Orders count
      $ordersData = DB::ordersWithoutPendingOnline()
        ->selectRaw('MONTH(created_at) as month, COUNT(*) as total')
        ->whereYear('created_at', Carbon::now()->year)
        ->groupBy('month')
        ->pluck('total', 'month');

      // Rides count
      $ridesData = DB::table('rides')
        ->selectRaw('MONTH(created_at) as month, COUNT(*) as total')
        ->whereYear('created_at', Carbon::now()->year)
        ->groupBy('month')
        ->pluck('total', 'month');

      // Profit (commission from both)
      $profitData = DB::ordersWithoutPendingOnline()
        ->selectRaw('MONTH(created_at) as month, SUM(platform_commission) as total')
        ->whereYear('created_at', Carbon::now()->year)
        ->groupBy('month')
        ->pluck('total', 'month')
        ->toArray();

      $ridesProfitData = DB::table('rides')
        ->selectRaw('MONTH(created_at) as month, SUM(platform_commission) as total')
        ->whereYear('created_at', Carbon::now()->year)
        ->groupBy('month')
        ->pluck('total', 'month')
        ->toArray();

      // Merge profits
      foreach ($ridesProfitData as $m => $val) {
        $profitData[$m] = ($profitData[$m] ?? 0) + $val;
      }

      // Income (full income from rides + orders)
      $incomeData = DB::ordersWithoutPendingOnline()
        ->selectRaw('MONTH(created_at) as month, SUM(grand_total) as total')
        ->whereYear('created_at', Carbon::now()->year)
        ->groupBy('month')
        ->pluck('total', 'month')
        ->toArray();

      $ridesIncomeData = DB::table('rides')
        ->selectRaw('MONTH(created_at) as month, SUM(fare) as total')
        ->whereYear('created_at', Carbon::now()->year)
        ->groupBy('month')
        ->pluck('total', 'month')
        ->toArray();

      foreach ($ridesIncomeData as $m => $val) {
        $incomeData[$m] = ($incomeData[$m] ?? 0) + $val;
      }

      // Format into chart JSON
      $ordersArray = $ordersData instanceof \Illuminate\Support\Collection ? $ordersData->toArray() : (array) $ordersData;
      $ridesArray = $ridesData instanceof \Illuminate\Support\Collection ? $ridesData->toArray() : (array) $ridesData;
      $profitArray = (array) $profitData;
      $incomeArray = (array) $incomeData;

      $chartData = [
        [
          'chart_data' => array_map(fn($m) => $ordersArray[$m] ?? 0, $months),
          'active_option' => !empty($ordersArray) ? array_search(max($ordersArray), $ordersArray) : null,
        ],
        [
          'chart_data' => array_map(fn($m) => $ridesArray[$m] ?? 0, $months),
          'active_option' => !empty($ridesArray) ? array_search(max($ridesArray), $ridesArray) : null,
        ],
        [
          'chart_data' => array_map(fn($m) => round(($profitArray[$m] ?? 0), 1), $months),
          'active_option' => !empty($profitArray) ? array_search(max($profitArray), $profitArray) : null,
        ],
        [
          'chart_data' => array_map(fn($m) => round(($incomeArray[$m] ?? 0), 1), $months),
          'active_option' => !empty($incomeArray) ? array_search(max($incomeArray), $incomeArray) : null,
        ],
      ];



      return [
        'data' => $chartData,
        'categories' => $monthNames
      ];

    }, ['dashboard']); // tag for easy cache clearing
  }

  public function salesLastSixMonths()
  {
    return cacheRemember('sales_last_six_months', 600, function () {

      $startDate = Carbon::now()->subMonths(5)->startOfMonth();

      $orders = DB::ordersWithoutPendingOnline()
        ->selectRaw("MONTH(created_at) as month, COUNT(*) as total")
        ->where('created_at', '>=', $startDate)
        ->groupBy('month')
        ->pluck('total', 'month');

      $rides = DB::table('rides')
        ->selectRaw("MONTH(created_at) as month, COUNT(*) as total")
        ->where('created_at', '>=', $startDate)
        ->groupBy('month')
        ->pluck('total', 'month');

      // Build last 6 months list
      $labels = [];
      $orderData = [];
      $rideData = [];
      for ($i = 5; $i >= 0; $i--) {
        $month = Carbon::now()->subMonths($i);
        $monthNum = $month->month;

        $labels[] = $month->format('M');
        $orderData[] = $orders[$monthNum] ?? 0;
        $rideData[] = $rides[$monthNum] ?? 0;
      }

      return [
        'labels' => $labels,
        'orders' => $orderData,
        'rides' => $rideData
      ];

    }, ['dashboard']); // dashboard tag for easy clearing
  }


  public function getSupportTracker()
  {
    $stats = DB::table('contacts')
      ->selectRaw("
            COUNT(*) AS totalTickets,
            SUM(CASE WHEN status = 'Open' THEN 1 ELSE 0 END) AS openTickets,
            SUM(CASE WHEN status = 'Close' THEN 1 ELSE 0 END) AS closedTickets
        ")
      ->first();

    // Calculate completion percentage
    $completionPercent = $stats->totalTickets > 0
      ? round(($stats->closedTickets / $stats->totalTickets) * 100)
      : 0;

    return response()->json([
      'total' => (int) $stats->totalTickets,
      'open' => (int) $stats->openTickets,
      'closed' => (int) $stats->closedTickets,
      'completionPercent' => $completionPercent
    ]);
  }


  public function storeDashboard()
  {
    $storeId = User::find(auth()->id())?->storeId;
    // Pre-calculate date ranges
    $now = now();
    $currentMonthStart = $now->copy()->startOfMonth();
    $currentMonthEnd = $now->copy()->endOfMonth();
    $previousMonthStart = $now->copy()->subMonth()->startOfMonth();
    $previousMonthEnd = $now->copy()->subMonth()->endOfMonth();
    $currentWeekStart = $now->copy()->startOfWeek();
    $currentWeekEnd = $now->copy()->endOfWeek();
    $lastWeekStart = $now->copy()->subWeek()->startOfWeek();
    $lastWeekEnd = $now->copy()->subWeek()->endOfWeek();

    // === 1. Single Aggregate Query ===
    $agg = DB::ordersWithoutPendingOnline()
      ->selectRaw("
            -- Current month
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN grand_total ELSE 0 END) as current_sales,
            COUNT(CASE WHEN created_at BETWEEN ? AND ? THEN 1 END) as current_orders,

            -- Previous month
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN grand_total ELSE 0 END) as previous_sales,
            COUNT(CASE WHEN created_at BETWEEN ? AND ? THEN 1 END) as previous_orders,

            -- Current week
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN grand_total ELSE 0 END) as current_earnings,
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN grand_total - platform_commission ELSE 0 END) as current_profit,
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN platform_commission ELSE 0 END) as current_expenses,

            -- Last week
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN grand_total ELSE 0 END) as last_earnings,
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN grand_total - platform_commission ELSE 0 END) as last_profit,
            SUM(CASE WHEN created_at BETWEEN ? AND ? THEN platform_commission ELSE 0 END) as last_expenses,

            -- Total revenue & sales
            SUM(grand_total) as total_revenue,
            COUNT(*) as total_sales
        ", [
        // Current month
        $currentMonthStart,
        $currentMonthEnd,
        $currentMonthStart,
        $currentMonthEnd,

        // Previous month
        $previousMonthStart,
        $previousMonthEnd,
        $previousMonthStart,
        $previousMonthEnd,

        // Current week
        $currentWeekStart,
        $currentWeekEnd,
        $currentWeekStart,
        $currentWeekEnd,
        $currentWeekStart,
        $currentWeekEnd,

        // Last week
        $lastWeekStart,
        $lastWeekEnd,
        $lastWeekStart,
        $lastWeekEnd,
        $lastWeekStart,
        $lastWeekEnd,
      ])
      ->where('store_id', $storeId)
      ->first();

    // === 2. Weekly earnings + yearly earnings/expenses in one query ===
    $daysOfWeek = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];

    $chartData = DB::ordersWithoutPendingOnline()
      ->selectRaw("
        DAYNAME(created_at) as day_name,
        WEEK(created_at, 3) as week_num,
        SUM(grand_total) as total_earning,
        SUM(platform_commission) as total_expense
    ")
      ->where('store_id', $storeId)
      ->whereYear('created_at', $now->year)
      ->groupByRaw("week_num, day_name, DAYOFWEEK(created_at)")
      ->orderByRaw("FIELD(day_name, 'Monday','Tuesday','Wednesday','Thursday','Friday','Saturday','Sunday')")
      ->get();

    // ---- Extract current week's daily totals (with missing days = 0) ----
    $weeklyEarnings = array_fill_keys($daysOfWeek, 0);
    foreach ($chartData->where('week_num', $now->weekOfYear) as $row) {
      if (isset($weeklyEarnings[$row->day_name])) {
        $weeklyEarnings[$row->day_name] = (float) $row->total_earning;
      }
    }
    $weeklyEarnings = array_values($weeklyEarnings);
    // ---- Extract full year earnings/expenses by week ----
    $weeklyGrouped = $chartData
      ->groupBy('week_num')
      ->map(function ($week) {
        return [
          'total_earning' => $week->sum('total_earning'),
          'total_expense' => $week->sum('total_expense')
        ];
      });

    $earningsBase = array_fill_keys($daysOfWeek, 0);
    $expensesBase = array_fill_keys($daysOfWeek, 0);

    foreach ($chartData->where('week_num', $now->weekOfYear) as $row) {
      if (isset($earningsBase[$row->day_name])) {
        $earningsBase[$row->day_name] = (float) $row->total_earning;
        $expensesBase[$row->day_name] = (float) $row->total_expense;
      }
    }

    $earningsList = array_values($earningsBase);
    $expensesList = array_values($expensesBase);


    // Percentage calculations
    $sales_percent = ($agg->previous_sales > 0)
      ? (($agg->current_sales - $agg->previous_sales) / $agg->previous_sales) * 100
      : 0;
    $orders_percent = ($agg->previous_orders > 0)
      ? (($agg->current_orders - $agg->previous_orders) / $agg->previous_orders) * 100
      : 0;

    $percentageChange = ($agg->last_earnings > 0)
      ? (($agg->current_earnings - $agg->last_earnings) / $agg->last_earnings) * 100
      : 0;
    $percentageChangeProfit = ($agg->last_profit > 0)
      ? (($agg->current_profit - $agg->last_profit) / $agg->last_profit) * 100
      : 0;
    $percentageChangeExpense = ($agg->last_expenses > 0)
      ? (($agg->current_expenses - $agg->last_expenses) / $agg->last_expenses) * 100
      : 0;

    $thisWeekEarning = $weeklyGrouped[$now->weekOfYear]['total_earning'] ?? 0;
    $lastWeekEarning = $weeklyGrouped[$now->copy()->subWeek()->weekOfYear]['total_earning'] ?? 0;
    $percentageChangeList = $lastWeekEarning != 0
      ? (($thisWeekEarning - $lastWeekEarning) / abs($lastWeekEarning)) * 100
      : 0;
    $orderCount = Order::where('store_id', $storeId)
      ->withCount('items')
      ->get(['id', 'items_count']);
    $pendingOrderCount = Order::where('store_id', $storeId)
      ->where('status', 'Placed')
      ->count();
    $pendingDues = Wallet::where('owner_type', 'store')->where('owner_id', $storeId)->first()?->balance;
    $productCount = Product::where('store_id', $storeId)->count();

    $pendingOrders = Order::with('user')->where('store_id', $storeId)
      ->where('status', 'Placed')
      ->get();
    return view('store.dashboard', [
      'sales' => (object) [
        'total_sales' => $agg->current_sales,
        'total_orders' => $agg->current_orders
      ],
      'sales_percent' => round($sales_percent, 2),
      'orders_percent' => round($orders_percent, 2),
      'earnings' => $agg->current_earnings,
      'profit' => $agg->current_profit,
      'expenses' => $agg->current_expenses,
      'percentageChange' => round($percentageChange, 2),
      'percentageChangeProfit' => round($percentageChangeProfit, 2),
      'percentageChangeExpense' => round($percentageChangeExpense, 2),
      'weeklyEarnings' => $weeklyEarnings,
      'earningsList' => $earningsList,
      'expensesList' => $expensesList,
      'percentageChangeList' => $percentageChangeList,
      'totalRevenue' => (float) $agg->total_revenue,
      'totalSales' => $agg->total_sales,
      'orderCount' => $orderCount,
      'pendingOrderCount' => $pendingOrderCount,
      'productCount' => $productCount,
      'pendingDues' => $pendingDues ?? 0,
      'pendingOrders' => $pendingOrders,
    ]);
  }

  public function averageDailySales(Request $request)
  {
    $storeId = User::find(auth()->id())?->storeId;

    // Example: Fetch last 30 days sales for store
    $sales = DB::ordersWithoutPendingOnline()
      ->selectRaw('DATE(created_at) as date, SUM(grand_total) as total')
      ->where('store_id', $storeId)
      ->whereMonth('created_at', now()->month)
      ->groupBy('date')
      ->orderBy('date')
      ->pluck('total')
      ->toArray();

    $totalSales = array_sum($sales); // sum all daily totals
    return response()->json([
      'data' => array_values($sales),
      'total' => $totalSales
    ]);
  }

  public function paymentMessage($txnId = null)
  {
    return view('webview.payment-message', compact('txnId'));
  }

}
