<?php
namespace App\Http\Controllers\admin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Driver;
use App\Traits\JsonResponseTrait;
use App\Repositories\Contracts\DriverRepositoryInterface;
use App\Repositories\Contracts\ReviewRepositoryInterface;
use App\Http\Requests\DriverRequest;
class DriverController extends Controller
{
  use JsonResponseTrait;
  protected $driverRepository;
  protected $reviewRepository;
  function __construct(DriverRepositoryInterface $driverRepository, ReviewRepositoryInterface $reviewRepository)
  {
    $this->middleware('permission:driver-list|driver-create|driver-edit|driver-delete', ['only' => ['index', 'show', 'driverListAll', 'driverMaps', 'showDetails']]);
    $this->middleware('permission:driver-create', ['only' => ['create', 'store']]);
    $this->middleware('permission:driver-edit', ['only' => ['edit', 'update', 'changeStatus']]);
    $this->middleware('permission:driver-delete', ['only' => ['destroy']]);
    $this->driverRepository = $driverRepository;
    $this->reviewRepository = $reviewRepository;
  }
  public function index()
  {
    $userStats = User::selectRaw("
        COUNT(*) as totalUser,
        COUNT(email_verified_at) as verified,
        SUM(CASE WHEN email_verified_at IS NULL THEN 1 ELSE 0 END) as emailNotVerified,
        SUM(CASE WHEN is_verified = 0 THEN 1 ELSE 0 END) as notVerified,
        SUM(CASE WHEN status = 0 THEN 1 ELSE 0 END) as userDisabled
    ")
      ->where('user_type', 'Driver')
      ->first();

    return view('admin.drivers.index', [
      'totalUser' => $userStats->totalUser,
      'verified' => $userStats->verified,
      'notVerified' => $userStats->notVerified,
      'userDisabled' => $userStats->userDisabled,
    ]);
  }

  public function driverListAll(Request $request)
  {
    return $this->driverRepository->allList($request);
  }

  public function driverMaps(Request $request)
  {
    $drivers = $this->driverRepository->allMaps($request);
    return view('admin.drivers.maps', compact('drivers'));
  }
  public function showDetails($id)
  {
    $user = $this->driverRepository->find($id);
    return view('_partials\_modals\modal-driver-approval', compact('user'));

  }
  public function show(Request $request, $id)
  {
    $user = $this->driverRepository->find($id);
    $reviewData = $this->reviewRepository->allReview($id, 'driver');
    $breadcrumbs = [
      'Home' => route('admin.dashboard'),
      'Driver' => route('drivers.index'),
      $user->name => null, // last one is active
    ];
    return view('/admin/drivers/show', compact('user', 'id', 'reviewData', 'breadcrumbs'));
  }
  public function create()
  {
    return view('/admin/drivers/create');
  }
  public function edit(int $id)
  {
    $driver = Driver::with('user', 'vehicle')->findOrFail($id);
    return view('admin.drivers.create', compact('driver'));
  }
  public function store(DriverRequest $request)
  {
    $driver = $this->driverRepository->store($request, null);
    return $this->successResponse($driver, __('locale.Driver Created Successfully!'));
  }
  public function update(DriverRequest $request, int $id)
  {
    $driver = $this->driverRepository->store($request, $id);
    return $this->successResponse($driver, __('locale.Driver Updated Successfully!'));
  }

  public function changeStatus(Request $request, $id)
  {
    $status = $request->post('status');
    $this->driverRepository->changeStatus($id, $status);
    return $this->successResponse(['id' => $id, 'status' => $status], __('locale.Driver Status Changed Successfully'));
  }
  public function activeDrivers()
  {
    $data = $this->driverRepository->activeDrivers('Delivery');
    return $this->successResponse($data, __('locale.Driver List'));
  }

}
