<?php
namespace App\Http\Controllers\admin;
use Auth;
use Chat;
use Carbon\Carbon;
use App\Models\User;
use App\Helpers\Helpers;
use Illuminate\Http\Request;
use Musonza\Chat\Models\Message;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Cache;
use Musonza\Chat\Models\Participation;
class ChatAppController extends Controller
{
  public function chatApp(Request $request)
  {
    $authUser = Auth::user();
    $selectedUserId = $request->get('user');

    // Get all conversations where the auth user is a participant
    $conversations = Chat::conversations()
      ->setParticipant($authUser)
      ->setPaginationParams([
        'page' => 1,
        'perPage' => 100,
        'sorting' => 'desc',
      ])
      ->get();

    $conversationIds = $conversations->pluck('conversation.id')->toArray();
    $unreadCounts = Message::select('chat_messages.conversation_id', DB::raw('COUNT(*) as unread'))
      ->join('chat_message_notifications as n', 'n.message_id', '=', 'chat_messages.id')
      ->whereIn('chat_messages.conversation_id', $conversationIds)
      ->where('n.messageable_id', $authUser->id)
      ->where('n.is_seen', 0)
      ->groupBy('chat_messages.conversation_id')
      ->pluck('unread', 'conversation_id'); // [ convo_id => unread_count ]
    $users = [];
    $chattedUserIds = [];

    foreach ($conversations as $conversation) {
      $data = $conversation->conversation;

      // Exclude self and non-Driver participants
      $participants = collect($data['participants'])->filter(function ($participant) use ($authUser) {
        return $participant['messageable_id'] != $authUser->id &&
          ($participant['messageable']['user_type'] ?? '') === 'Driver';  //Driver
      });

      foreach ($participants as $participant) {
        $messageable = $participant['messageable'] ?? null;
        if (!$messageable || empty($messageable['name'])) {
          continue;
        }

        $userId = $messageable['id'];
        if (in_array($userId, $chattedUserIds)) {
          continue;
        }
        $chattedUserIds[] = $userId;

        $lastMessage = $data['last_message'] ?? null;
        $lastCreated = $lastMessage['created_at'] ?? null;

        $users[] = [
          'id' => $userId,
          'convo_id' => $data['id'],
          'name' => $messageable['name'],
          'user_id' => $userId,
          'last_message' => $lastMessage['body'] ?? '',
          'last_message_type' => $lastMessage['type'] ?? '',
          'last_time' => $lastCreated ? Carbon::parse($lastCreated)->format('h:i A') : '',
          'last_date' => $lastCreated ? Carbon::parse($lastCreated)->format('d M Y') : '',
          'count' => $unreadCounts[$data['id']] ?? '', // unreadCount can be array with convo_id as key
          'block' => $participant['settings']['mute_mentions'] ?? false,
        ];
      }
    }

    // Get other active Drivers not yet chatted with
    $excludeIds = array_merge($chattedUserIds, [$authUser->id]);

    $new_users = User::where('status', 1)
      ->where('user_type', 'Driver')
      ->whereNotIn('id', $excludeIds)
      ->get();

    if ($selectedUserId && !in_array($selectedUserId, $excludeIds)) {
      $selectedUser = User::where('id', $selectedUserId)->first();
      if ($selectedUser && !$new_users->contains('id', $selectedUser->id)) {
        $new_users->push($selectedUser);
      }
    }

    return view('common.app-chat', compact('users', 'new_users', 'selectedUserId'));
  }

  public function chat($userId = null, $conversationId = null, $load = null, $myUserId = null)
  {
    $url = config('app.url') . 'admin/';
    $asset = url('');
    $page = request()->get('page', 1); // Default to page 1
    $perPage = 50;
    $authUser = empty($myUserId) ? Auth::user() : User::find($myUserId);
    $authUserId = $authUser->id;
    $myUser = User::find($authUserId);
    $otherUser = User::find($userId);

    $conversation = Chat::conversations()->getById($conversationId);
    $messages = Chat::conversation($conversation)->setParticipant($myUser)->setPaginationParams([
      'page' => $page,
      'perPage' => $perPage,
      'sorting' => 'desc'
    ])->getMessages();
    Chat::conversation($conversation)->setParticipant($myUser)->readAll();
    //dd($messages);
    $userImage = $otherUser->profile_photo_url;
    $adminImage = $authUser->profile_photo_url;

    $participation = Chat::conversation($conversation)->getParticipation($myUser);
    $settings = $participation->settings ?? [];
    $isBlocked = !empty($settings['mute_mentions']);

    $blockAction = $isBlocked ? 'UnBlock' : 'Blocked';
    $blockLabel = $isBlocked ? 'UnBlock Contact' : 'Block Contact';
    $blockUrl = url("/chat/block-user/{$conversationId}/{$blockAction}");
    $blockHtml = "<a class='dropdown-item d-none' href='{$blockUrl}'>{$blockLabel}</a>";

    $isOnline = Cache::has("user-is-online-{$otherUser->id}");
    $status = $isBlocked ? 'busy' : ($isOnline ? 'online' : 'offline');
    $lastSeen = $isOnline ? $otherUser->last_seen : $otherUser->created_at;
    $lastSeenText = Carbon::parse($lastSeen)->diffForHumans();

    $colorCodes = ['#dcf8c6', '#e1f3fb', '#c6e1fa;', '#fafad2', '#ffcccb', '#e6e6fa', '#d4e8a6', '#d4e5d1', '#d1e7dd', '#e0f7fa', '#ffe4e1', '#d8c7f7', '#f5f5dc', '#f5f5f5', '#fff1a3'];
    $senderColors = [];
    $users = User::where('id', $authUser->id)->get();
    foreach ($users as $key => $user) {
      $assignedColor = $colorCodes[$key % count($colorCodes)];
      $senderColors[str_replace(' ', '_', strtolower($user->name))] = $assignedColor;
    }
    $html = '';
    if ($load != 'load') {
      // Start building HTML
      $html .= <<<HTML
      <div class="chat-history-wrapper">
         <div class="chat-history-header border-bottom">
            <div class="d-flex justify-content-between align-items-center">
              <div class="d-flex overflow-hidden align-items-center">
                <i class="icon-base ti tabler-menu-2 icon-lg cursor-pointer d-lg-none d-block me-4"
                  data-bs-toggle="sidebar" data-overlay data-target="#app-chat-contacts"></i>
                <div class="flex-shrink-0 avatar avatar-{$status}">
                  <img src="{$userImage}" alt="Avatar" class="rounded-circle"
                    data-bs-toggle="sidebar" data-overlay data-target="#app-chat-sidebar-right" />
                </div>
                <div class="chat-contact-info flex-grow-1 ms-4">
                  <h6 class="m-0 fw-normal">{$otherUser->name}</h6>
                  <small class="user-status text-body">Last Seen: {$lastSeenText}</small>
                </div>
              </div>
              <div class="d-flex align-items-center">
                <a href="tel:{$otherUser->phone}" class="btn btn-text-secondary cursor-pointer d-sm-inline-flex d-none me-1 btn-icon rounded-pill">
                  <i class="icon-base ti tabler-phone icon-22px"></i>
                </a>

                <div class="dropdown">
                  <button class="btn btn-icon btn-text-secondary text-secondary rounded-pill dropdown-toggle hide-arrow"
                    data-bs-toggle="dropdown" aria-expanded="true" id="chat-header-actions"><i
                      class="icon-base ti tabler-dots-vertical icon-22px"></i></button>
                  <div class="dropdown-menu dropdown-menu-end" aria-labelledby="chat-header-actions">
                    <a class="dropdown-item"  target="_blank" href="{$url}users/{$otherUser->id}">View Contact</a>
                    <a class="dropdown-item d-none" href="javascript:void(0);">Mute Notifications</a>
                        {$blockHtml}
                    <a class="dropdown-item" href="{$url}chat/clear/{$conversationId}">Clear Chat</a>
                    <a class="dropdown-item d-none" href="javascript:void(0);">Report</a>
                  </div>
                </div>
              </div>
            </div>
          </div>

    HTML;
    }
    $html .= <<<HTML
    <div class="chat-history-body" id="userChats" >
            <ul class="list-unstyled chat-history">
        <input type="hidden" id="conversationid" value="{$conversationId}">
        <input type="hidden" id="userid" value="{$userId}">
        <input type="hidden" id="typeid" value="User">
    HTML;

    $oldDate = '';
    $previousSender = null;
    $groupOpen = false;

    foreach ($messages->reverse() as $message) {
      $isSender = $message->messageable_id == $authUserId && $message->is_sender;
      $alignClass = $isSender ? 'chat-message-right' : 'chat-message';
      $senderImage = $isSender ? $adminImage : $userImage;
      $createdAt = $message->created_at;
      $dateGroup = $createdAt->format('Ymd');
      $time = $createdAt->format('h:i A');

      // Date Divider
      if ($oldDate !== $dateGroup) {
        if ($groupOpen) {
          $html .= '</div></li>';
          $groupOpen = false;
        }
        $html .= "<div class='chat-date-wrapper text-center my-1 w-100'><div class='chat-date-divider'>{$createdAt->format('d M Y')}</div></div>";
        $oldDate = $dateGroup;
      }

      // Group Break
      if ($previousSender !== $isSender) {
        if ($groupOpen) {
          $html .= '</div></li>'; // Close previous group
        }

        $html .= "<li class='chat-message {$alignClass}'>";
        $html .= "<div class='d-flex overflow-hidden' style='max-width: 75%; width: fit-content;min-width:35%'>";

        // Left-aligned message: Avatar first
        if (!$isSender) {
          $html .= <<<HTML
                <div class="user-avatar flex-shrink-0 me-4">
                    <div class="avatar avatar-sm">
                        <img src="{$senderImage}" class="rounded-circle" alt="Avatar" />
                    </div>
                </div>
HTML;
        }

        $html .= '<div class="chat-message-wrapper flex-grow-1">';
        $groupOpen = true;
        $previousSender = $isSender;
      }

      // Message Body Handling
      $attachmentPath = "{$asset}/storage/Chat/{$conversationId}/";
      $body = $this->renderMessageHtml($message, $attachmentPath);

      $readIndicator = '';
      $clr = 'color:#lightgrey !important';
      // Last message from me: show read status
      if (isset($message) && $message->messageable_id == $authUserId && $message->is_sender) {
        $read = $message->is_seen ?? false;
        $clr = 'color:#fff !important';
        $readIndicator = '<i class="icon-base ti tabler-checks icon-16px ' . ($read ? 'text-success' : 'text-muted') . ' me-1"></i>';
      }
      $readIndicatorClass = 'text-' . $isSender ? 'end' : 'start';
      // Chat Message Output
      $html .= <<<HTML
        <div class="chat-message-text mt-2" id="message-{$message->id}" data-message-id="{$message->id}"
     data-message-type="{$message->type}">{$body}
          <div class="text-{$readIndicatorClass} text-body-secondary mt-0" style="{$clr}">
            {$readIndicator}<small>{$time}</small>
        </div></div>
HTML;
    }

    // Close last message group
    if ($groupOpen) {

      $html .= <<<HTML

    </div> <!-- close chat-message-wrapper -->
HTML;

      if ($isSender) {
        $html .= <<<HTML
            <div class="user-avatar flex-shrink-0 ms-4">
                <div class="avatar avatar-sm">
                    <img src="{$senderImage}" class="rounded-circle" alt="Avatar" />
                </div>
            </div>
HTML;
      }

      $html .= '</div></li>'; // Close d-flex & li
    }


    return [
      'block' => $isBlocked,
      'message' => 'Chat loaded successfully',
      'code' => 200,
      'data' => $html,
      'convo' => $conversationId,
      'user_id' => $userId,
      'load_data' => $messages->isNotEmpty() ? true : false,
    ];
  }

  protected function renderMessageHtml($message, $attachmentPath)
  {
    // Message Body Handling
    $body = '';
    switch ($message->type) {
      case 'Video':
        $src = $attachmentPath . $message->body;
        $body = "<video style='max-width:55%' controls><source src='{$src}' type='video/mp4'></video>";
        $body .= "<div class='mt-1 copyIcons'>
                        <a href='{$src}' download style='background:none;border:none;color:#999;'>
                            <i class='icon-base ti tabler-download'></i>
                        </a>
                      </div>";
        break;
      case 'Image':
        $src = $attachmentPath . $message->body;
        $body = "<img src='{$src}' class='img-fluid rounded' alt='Image' />";
        $body .= "<div class='mt-1 copyIcons'>
                        <a href='{$src}' download style='background:none;border:none;color:#999;'>
                            <i class='icon-base ti tabler-download'></i>
                        </a>
                      </div>";
        break;
      case 'attachment':
        $data = $message->data ?? [];
        $file = $data['file_name'] ?? '';
        $fileType = $data['file_type'] ?? '';
        $src = $attachmentPath . $file;
        $body = '<p class="mb-1">' . e($message->body) . '</p>';
        $body .= $fileType === 'Video'
          ? "<video style='max-width:55%' controls><source src='{$src}' type='video/mp4'></video>"
          : "<img src='{$src}' class='img-fluid rounded' alt='Attachment' />";
        $body .= "<div class='mt-1 copyIcons'>
                        <button class='copy-btn' data-message-id='message-{$message->id}' style='background:none;border:none;color:#999;'>
                            <i class='icon-base ti tabler-copy'></i>
                        </button>
                        <a href='{$src}' download style='background:none;border:none;color:#999;'>
                            <i class='icon-base ti tabler-download'></i>
                        </a>
                      </div>";
        break;
      case 'reply':
        // message->data may be an array or a JSON string
        $messageData = $message->data;
        if (is_string($messageData)) {
          $decoded = json_decode($messageData, true);
          $messageData = is_array($decoded) ? $decoded : [];
        }
        $messageData = (array) ($messageData ?? []);

        // Fields that may exist depending on how reply was stored
        $replyId = $messageData['reply_to_message_id'] ?? null;
        $replyUserId = $messageData['reply_to_user_id'] ?? null;
        $replyUserName = $messageData['reply_to_user_name'] ?? 'Unknown';
        $replyTitle = $messageData['reply_to_body'] ?? null;
        $replyMediaUrl = $messageData['reply_to_media'] ?? null;
        $replyType = $messageData['reply_to_type'] ?? 'text';
        $replyText = $messageData['reply_to_body'] ?? '';

        // sanitize
        $replyUserNameSafe = e($replyUserName);
        $replyTextSafe = e($replyText);
        $replyTitleSafe = e($replyTitle);

        // Build quoted block (reply preview)
        $quoted = '<div class="reply-preview p-1 rounded" style="background:#f6f6f8;border-left:4px solid #2ecc71;">';
        $quoted .= "<div class='reply-preview-meta'><p class='text-dark'><b>{$replyUserNameSafe}</b></p></div>";

        if (!empty($replyMediaUrl)) {
          // If media is present, try to show image/video
          $ext = pathinfo(parse_url($replyMediaUrl, PHP_URL_PATH), PATHINFO_EXTENSION);
          $ext = strtolower($ext);
          if (in_array($ext, ['mp4', 'webm', 'ogg']) || stripos($replyType, 'video') !== false) {
            $quoted .= "<div class='reply-media mb-1'><video style='max-width:100%;height:auto;display:block;' controls preload='metadata'><source src=\"" . e($replyMediaUrl) . "\" type='video/mp4'>Your browser does not support the video tag.</video></div>";
          } elseif (in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'svg', 'webp']) || stripos($replyType, 'image') !== false) {
            $quoted .= "<div class='reply-media mb-1'><img src=\"" . e($replyMediaUrl) . "\" alt='Reply image' style='max-width:100%;height:auto;border-radius:6px;display:block;' /></div>";
          } else {
            // unknown file type, show link
            $quoted .= "<div class='reply-media mb-1'><a href=\"" . e($replyMediaUrl) . "\" target='_blank' rel='noopener noreferrer'>" . e(basename($replyMediaUrl)) . "</a></div>";
          }
        } elseif (!empty($replyTitle)) {
          // no media, but have a title/body
          $quoted .= "<div class='reply-text mb-1' style='color:#555; font-size:13px;'>" . $replyTitleSafe . "</div>";
        } else {
          // fallback to reply text or empty
          $quoted .= "<div class='reply-text mb-1' style='color:#555; font-size:13px;'>" . ($replyTextSafe ?: '&nbsp;') . "</div>";
        }

        // clickable area to jump to original message (if you have JS scrollToMessage function)
        if ($replyId) {
          $quoted = "<div class='replyClick' data-id=\"" . e($replyId) . "\" onclick=\"if(typeof scrollToMessage === 'function'){scrollToMessage(" . e(json_encode($replyId)) . ")}\">{$quoted}</div>";
        }

        $quoted .= '</div>'; // .reply-preview

        // Compose the final body: quoted preview + actual message body + controls
        $body = '<div class="reply-wrapper" style="margin:8px 0;">';
        // Quoted preview
        $body .= $quoted;

        // The main replied message body (what user typed as reply)
        $body .= '<div class="reply-main mt-2">';
        $body .= '<p class="mb-1">' . e($message->body) . '</p>';
        $body .= '</div>';

        // Footer: time + copy/download buttons (similar to your other cases)
        $time = $message->created_at ? $message->created_at->format('H:i') : '';
        $body .= '<div class="d-flex justify-content-between align-items-center mt-2" style="font-size:12px;color:#999;">';

        $body .= '<div class="reply-actions copyIcons">';
        // copy button
        $body .= '<button class="copy-btn btn btn-sm" data-message-id="message-' . e($message->id) . '" style="background:none;border:none;color:#999;margin-right:8px;"><i class="icon-base ti tabler-copy"></i></button>';
        // if reply included a media file, show download link
        if (!empty($replyMediaUrl)) {
          $body .= '<a href="' . e($replyMediaUrl) . '" download style="color:#999;text-decoration:none;"><i class="icon-base ti tabler-download"></i></a>';
        }
        $body .= '</div></div>'; // end footer

        $body .= '</div>'; // end .reply-wrapper

        break;

      default:
        $body = '<p class="mb-0">' . e($message->body) . '</p>';
        $body .= "<div class='mt-1 copyIcons'>
                        <button class='copy-btn' data-message-id='message-{$message->id}' style='background:none;border:none;color:#999;'>
                            <i class='icon-base ti tabler-copy'></i>
                        </button>

                      </div>";
    }
    return $body;
  }

  public function chatStart($id, $page = null)
  {
    $user = User::findOrFail($id);
    $myUser = Auth::user();
    // Find any existing private conversation between the two users
    $existingConversation = DB::table('chat_participation')->select('conversation_id', DB::raw('COUNT(*) as count'))->whereIn('messageable_id', [$myUser->id, $user->id])->groupBy('conversation_id')->having('count', '>', 1)->pluck('conversation_id')->first();
    if (!$existingConversation) {
      $participants = User::where('id', $myUser->id)->where('status', '!=', 2)->get();
      // Add $user to participants if not already included
      if (!$participants->contains('id', $user->id)) {
        $participants->push($user);
      }
      $participantArray = $participants->all();
      //$participants = [$myUser, $user];
      $conversation = Chat::createConversation($participantArray)->makePrivate(true);
      $conversation->update([
        'data' => ['title' => 'Laravel Channel', 'description' => 'Laravel Channel Description',]
      ]);
      $conversationId = $conversation->id;
    } else {
      $conversationId = $existingConversation;
    }
    return $this->chat($id, $conversationId, null, null);
  }

  public function sendMessage(Request $request)
  {
    $myUser = Auth::check() ? Auth::user() : User::find($request->sender_id);
    $recipient = User::find($request->userid);
    $conversationId = $request->conversation_id;
    if (empty($conversationId)) {
      // Check if a conversation already exists between the two users
      $existing = DB::table('chat_participation')->select('conversation_id', DB::raw('COUNT(*) as count'))->whereIn('messageable_id', [$myUser->id, $recipient->id])->groupBy('conversation_id')->having('count', '>', 1)->pluck('conversation_id')->first();

      if (!$existing) {
        // Create a new private conversation
        $participants = User::where('id', $myUser->id)->where('status', '!=', 2)->get();
        // Add $user to participants if not already included
        if (!$participants->contains('id', $recipient->id)) {
          $participants->push($recipient);
        }
        $participantArray = $participants->all();
        //$participants = [$myUser, $recipient];
        $conversation = Chat::createConversation($participantArray)->makePrivate(true);
        $conversation->update(['data' => ['title' => 'Laravel Channel', 'description' => 'Laravel Channel Description']]);
        $conversationId = $conversation->id;
      } else {
        $conversationId = $existing;
      }
    }
    // Define message type and text
    if (!empty($request->message) && ($request->hasFile('icon'))) {
      $textMessage = $request->message;
      $prepareFile = $this->prepareFile($request);
      $ext = Helpers::typ(pathinfo($prepareFile, PATHINFO_EXTENSION));
      $messageType = 'attachment';
      $data = ['file_name' => $prepareFile, 'file_type' => $ext];
    } else {
      $prepareFile = $this->prepareFile($request);
      $textMessage = $request->message ?? $prepareFile;
      $messageType = $request->message ? 'text' : Helpers::typ(pathinfo($textMessage, PATHINFO_EXTENSION));
      $data = array();
    }
    if ($request->reply_to_message_id) {
      $msg = Chat::messages()->getById($request->reply_to_message_id);
      $participant = Participation::find($msg->participation_id);
      // Reply reference details
      $media = in_array($msg->type, ['Image', 'Video'])
        ? asset('storage/Chat/' . $conversation->id . '/' . $msg->body)
        : null;
      $sub_type = in_array($msg->type, ['Image', 'Video']) ? $msg->type : 'text';
      $data = [
        'reply_to_message_id' => $msg->id,
        'reply_to_user_id' => $participant->messageable_id,
        'reply_to_user_name' => $participant->messageable?->name ?? 'Unknown',
        'reply_to_body' => $msg->body,
        'reply_to_media' => $media,
        'reply_to_type' => $msg->type,
        'sub_type' => $sub_type,
        'message' => $textMessage,
      ];
      $messageType = 'reply';
    }
    // Send message
    $conversation = Chat::conversations()->getById($conversationId);
    $message = Chat::message($textMessage)
      ->type($messageType)
      ->data($data)
      ->from($myUser)
      ->to($conversation)
      ->send();

    // Store file if applicable
    if ($request->hasFile('icon')) {
      $this->storeFile($request->file('icon'), $conversationId, $prepareFile);
    }
    $asset = url('');
    $attachmentPath = "{$asset}/storage/Chat/{$conversationId}/";
    $body = $this->renderMessageHtml($message, $attachmentPath);
    return response()->json([
      'status' => 200,
      'message' => $body,
      'time' => date('h:i A'),
      'file_name' => $data['file_name'] ?? null,
      'file_type' => $data['file_type'] ?? null,
      'type' => $messageType,
      'conversation_id' => $conversationId
    ]);
  }

  private function prepareFile(Request $request)
  {
    if ($request->hasFile('icon')) {
      $image = $request->file('icon');
      return 'MGC-' . time() . '.' . $image->extension();
    }
    return null;
  }


  private function storeFile($file, $conversationId, $prepareFile)
  {
    $path = storage_path("app/public/Chat/{$conversationId}");
    if (!File::exists($path)) {
      File::makeDirectory($path, 0755, true);
    }
    $file->move($path, $prepareFile);
  }

  public function unreadMesg($convo_id)
  {
    $user = User::find(Auth::user()->id);
    $conversation = Chat::conversations()->getById($convo_id);
    Chat::conversation($conversation)->setParticipant($user)->readAll();
  }

  public function chatBlock($conversationid, $status)
  {
    $user = User::find(Auth::user()->id);
    if ($status == 'Blocked') {
      $settings = ['mute_mentions' => true];
    } else {
      $settings = ['mute_mentions' => false];
    }
    $conversation = Chat::conversations()->getById($conversationid);
    $con = Chat::conversation($conversation)->getParticipation($user)->update(['settings' => $settings]);
    return redirect()->route('chat');
  }

  public function chatClear($conversationid)
  {
    $user = User::find(Auth::user()->id);
    $conversation = Chat::conversations()->getById($conversationid);
    Chat::conversation($conversation)->setParticipant($user)->clear();
    return redirect()->route('chat');
  }


}
