<?php
namespace App\Http\Controllers\admin;
use App\Http\Controllers\Controller;
use App\Http\Requests\CategoryRequest;
use App\Traits\JsonResponseTrait;
use App\Traits\ImageUploadTrait;
use Illuminate\Http\Request;
use App\Repositories\Contracts\CategoryRepositoryInterface;
class CategoryController extends Controller
{
  use JsonResponseTrait, ImageUploadTrait;

  protected $categoryRepository;

  public function __construct(CategoryRepositoryInterface $categoryRepository)
  {
    $this->middleware('permission:category-list|category-create|category-edit|category-delete', ['only' => ['index', 'show', 'subCategory', 'categoryListAll']]);
    $this->middleware('permission:category-create', ['only' => ['create', 'store']]);
    $this->middleware('permission:category-edit', ['only' => ['edit', 'update', 'changeStatus']]);
    $this->middleware('permission:category-delete', ['only' => ['destroy']]);

    $this->categoryRepository = $categoryRepository;
  }

  public function index()
  {
    $allCategories = $this->categoryRepository->allCategories();
    return view('admin.categories.index', compact('allCategories'));
  }
  public function subCategory()
  {
    $allCategories = $this->categoryRepository->allCategories();
    return view('admin.categories.subcategory', compact('allCategories'));
  }

  public function categoryListAll(Request $request)
  {
    $query = $request->query('type') ?? null;
    $categories = $this->categoryRepository->allList($query);
    return \DataTables::of($categories)->addIndexColumn()->make(true);
  }

  public function store(CategoryRequest $request)
  {
    $data = $request->validated();
    if ($request->hasFile('image')) {
      $data['image'] = $this->uploadImage($request->file('image'), 'categories');
    }
    $category = $this->categoryRepository->store($data);

    return $this->successResponse($category, __('locale.Category Uploaded Successfully'));
  }

  public function update(CategoryRequest $request, $id)
  {
    $data = $request->validated();
    $data['prescriptions'] = $request->has('prescriptions') ? 1 : 0;
    $category = $this->categoryRepository->update($id, $data);
    if ($request->hasFile('image')) {
      $path = $this->uploadImage($request->file('image'), 'categories', $category->image);
      $category->image = $path;
      $category->save();
    }
    return $this->successResponse($category->load('parent'), __('locale.Category Edited Successfully'));
  }

  public function dropdown()
  {
    return $this->categoryRepository->dropdown();
  }

  public function destroy($id)
  {
    $this->categoryRepository->delete($id);
    return $this->successResponse($id, __('locale.Category Deleted Successfully'));
  }

  public function changeStatus()
  {
    $category = $this->categoryRepository->toggleStatus(request('id'), request('status'));
    return $this->successResponse($category, __('locale.Category Status Changes Successfully'));
  }
  public function subcategories($id)
  {
    return $this->categoryRepository->subCategories($id);
  }

}
