<?php

namespace App\Http\Controllers;
use App\Traits\JsonResponseTrait;
use App\Models\Subscription;
use App\Models\Package;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Stripe\Stripe;
use Stripe\PaymentMethod;
use Stripe\Customer;
use Stripe\Checkout\Session as CheckoutSession;
use App\Services\SubscriptionService;
use App\Events\SubscriptionCancelled;
use App\Events\SubscriptionCreated;
use Stripe\Exception\InvalidRequestException;
use Illuminate\Support\Facades\Log;

class SubscriptionController extends Controller
{
  use JsonResponseTrait;
  protected SubscriptionService $service;
  public function __construct(SubscriptionService $service)
  {
    $this->service = $service;
    Stripe::setApiKey(config('cashier.secret'));
  }

  public function profile()
  {

    $customerDeleted = false;
    $user = Auth::user();
    $subscription = Subscription::with('package')
      ->where('user_id', $user->id)
      ->first();
    $transactions = Transaction::where('subscription_id', $subscription->id ?? 0)
      ->orderByDesc('start_date')->orderByDesc('id')
      ->get();
    $paymentMethods = collect(); // default empty
    try {
      // Try fetching payment methods from Stripe
      $paymentMethods = PaymentMethod::all([
        'customer' => $user->stripe_id,
        'type' => 'card',
      ]);
    } catch (InvalidRequestException $e) {
      if (str_contains($e->getMessage(), 'No such customer')) {
        Log::warning('Stripe customer not found for user ID ' . $user->id . ': ' . $e->getMessage());
        $customerDeleted = true;
      } else {
        Log::error('Stripe API error while fetching payment methods: ' . $e->getMessage());
      }
    }

    $setupIntent = \Stripe\SetupIntent::create([
      'customer' => $user->stripe_id,
    ]);
    $clientSecret = $setupIntent->client_secret;

    return view('store.subscription_profile', compact('subscription', 'transactions', 'paymentMethods', 'user', 'clientSecret', 'customerDeleted'));

  }

  public function subscribe(Request $request)
  {
    $data = $request->validate([
      'package_id' => 'required|integer|exists:packages,id',
    ]);

    $package = Package::with('duration')->findOrFail($data['package_id']);
    $durationMap = [
      'Days' => 'day',
      'Weeks' => 'week',
      'Months' => 'month',
      'Years' => 'year',
    ];
    $interval_count = $package['duration']?->name ?? 1;
    $interval = $durationMap[$package['duration']?->week ?? 'Months'] ?? 'month';
    $quantity = 1;
    $unitAmount = (int) round(($package->price) * 100); // in cents

    Stripe::setApiKey(config('cashier.secret'));

    // Ensure we have a Stripe customer
    $user = Auth::user();
    if (!$user->stripe_id) {
      $user->createAsStripeCustomer();
    }

    // Create Checkout Session
    $session = CheckoutSession::create([
      'customer' => $user->stripe_id,
      'payment_method_types' => ['card'],
      'mode' => 'subscription',
      'line_items' => [
        [
          'price_data' => [
            'currency' => 'usd',
            'product_data' => [
              'name' => $package->name,
            ],
            'unit_amount' => $unitAmount,
            'recurring' => [
              'interval' => $interval,
              'interval_count' => $interval_count
            ]
          ],
          'quantity' => $quantity,
        ]
      ],
      'metadata' => [
        'package_id' => $package->id,
        'license_count' => $quantity,
      ],
      'success_url' => route('subscription.success') . '?session_id={CHECKOUT_SESSION_ID}',
      'cancel_url' => route('subscription.cancel'),
    ]);

    // Redirect the user to Stripe Checkout
    return redirect($session->url);
  }

  public function success(Request $request)
  {
    Stripe::setApiKey(config('cashier.secret'));

    $sessionId = $request->query('session_id');
    if (!$sessionId) {
      return redirect()->route('subscription.profile')
        ->withErrors(__('locale.No session ID provided.'));
    }
    // Retrieve the Checkout Session with its metadata
    $session = CheckoutSession::retrieve($sessionId);
    // Grab metadata
    $meta = $session->metadata;
    $packageId = $meta->package_id;
    $stripeSubId = $session->subscription;
    $stripeSubscription = \Stripe\Subscription::retrieve($stripeSubId);
    $periodEnd = \Carbon\Carbon::createFromTimestamp($stripeSubscription->current_period_end)->format('Y-m-d');

    // Lookup package & company
    $user = Auth::user();
    $package = Package::findOrFail($packageId);
    $amountPaid = $session->amount_total / 100;

    $subscription = Subscription::updateOrCreate(
      ['user_id' => $user->id],
      [
        'name' => 'main',
        'package_id' => $package->id,
        'amount' => $amountPaid,
        'status' => 'complete',
        'canceled_at' => null,
        'next_billing_date' => $periodEnd ?? now()->addMonth(),
        'stripe_subscription_id' => $stripeSubId,
      ]
    );
    $subscription->refresh();
    event(new SubscriptionCreated($subscription, $stripeSubscription));  //handle with webhook
    return redirect()->route('subscription.profile')->with('success', __('locale.Subscription activated!'));
  }

  public function cancel()
  {
    $user = Auth::user();
    $subscription = Subscription::where('user_id', $user->id)->first();
    if (!$subscription) {
      return $this->errorResponse(__('locale.Subscription not found.'), 400);
    }
    event(new SubscriptionCancelled($subscription));
    return $this->successResponse(null, __('locale.Subscription canceled successfully.'));
  }

  public function cancell()
  {
    return redirect()->route('subscription.profile')->with('error', __('locale.Payment Cancelled successfully!'));
  }


  public function deleteCard($paymentMethodId)
  {
    try {
      $paymentMethod = PaymentMethod::retrieve($paymentMethodId);
      $paymentMethod->detach(); // Detaches from customer
      return $this->successResponse(null, __('locale.Card deleted successfully'));
    } catch (\Exception $e) {
      return $this->errorResponse($e->getMessage(), 400);
    }
  }

  public function savePaymentMethod(Request $request)
  {
    $user = auth()->user(); // or however you get the user
    $paymentMethodId = $request->payment_method;

    // Attach to customer
    PaymentMethod::retrieve($paymentMethodId)->attach([
      'customer' => $user->stripe_id,
    ]);

    // Set as default
    Customer::update($user->stripe_id, [
      'invoice_settings' => [
        'default_payment_method' => $paymentMethodId,
      ],
    ]);
    $user->default_payment_method = $paymentMethodId;
    $user->save();
    if ($request->type == 'default') {
      return back()->with('success', __('locale.Card added as default successfully.'));
    } else {
      return back()->with('success', __('locale.Card added successfully.'));
    }

  }


  public function updatePackage(Request $request)
  {
    $data = $request->validate([
      'package_id' => 'required|integer|exists:packages,id',
    ]);

    Stripe::setApiKey(config('cashier.secret'));
    $user = auth()->user();
    $subscription = Subscription::where('user_id', $user->id)->first();
    if (!$subscription) {
      return $this->errorResponse(__('locale.Subscription not found.'), 400);
    }
    $package = Package::findOrFail($data['package_id']);
    $durationMap = [
      'Days' => 'day',
      'Weeks' => 'week',
      'Months' => 'month',
      'Years' => 'year',
    ];
    $interval_count = $package['duration']?->name ?? 1;
    $interval = $durationMap[$package['duration']?->week ?? 'Months'] ?? 'month';
    $quantity = 1;
    $unitAmount = (int) round(($package->price) * 100); // in cents

    //if we want to cancel old one and create new subscription without adjustment
    if ($subscription) {
      event(new SubscriptionCancelled($subscription));
    }
    // Create Checkout Session
    $session = CheckoutSession::create([
      'customer' => $user->stripe_id,
      'payment_method_types' => ['card'],
      'mode' => 'subscription',
      'line_items' => [
        [
          'price_data' => [
            'currency' => 'usd',
            'product_data' => [
              'name' => $package->name,
            ],
            'unit_amount' => $unitAmount,
            'recurring' => [
              'interval' => $interval,
              'interval_count' => $interval_count
            ]
          ],
          'quantity' => $quantity,
        ]
      ],
      'metadata' => [
        'package_id' => $package->id,
        'license_count' => $quantity,
      ],
      'success_url' => route('subscription.success') . '?session_id={CHECKOUT_SESSION_ID}',
      'cancel_url' => route('subscription.cancel'),
    ]);
    // Redirect the user to Stripe Checkout
    return response()->json(['checkout_url' => $session->url]);

  }
}
