<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Order;
use App\Models\Ride;
use App\Models\User;
use App\Models\Store;
use App\Models\CouponUsage;
use Illuminate\Support\Facades\Auth;
use App\Events\PaymentSuccessEvent;
use App\Helpers\Helpers;
use App\Models\WalletTransaction;
use App\Models\Transaction;
use App\Services\WalletService;
use Illuminate\Support\Str;
use App\Traits\ApiResponseTrait;
use App\Http\Requests\OrderPaymentRequest;
use App\Http\Requests\RidePaymentRequest;
use App\Http\Requests\WalletPaymentRequest;
use App\Repositories\Contracts\UserRepositoryInterface;
use App\Repositories\Contracts\TransactionRepositoryInterface;
use App\Helpers\PaystackHelper;
class PaystackController extends Controller
{
  use ApiResponseTrait;
  protected $transactionRepository;
  protected $userRepository;
  public function __construct(UserRepositoryInterface $userRepository, TransactionRepositoryInterface $transactionRepository)
  {
    $this->transactionRepository = $transactionRepository;
    $this->userRepository = $userRepository;
  }

  public function addWallet(WalletPaymentRequest $request)
  {
    $user = Auth::user();
    $amountKobo = $request->amount * 100; // NGN in kobo
    try {
      $metadata = [
        'user_id' => $user->id,
        'amount' => $request->amount
      ];
      $paystackResponse = PaystackHelper::initializeTransaction($amountKobo, $user->email, route('wallet.callback'), $metadata);
      return $this->successResponse(['authorization_url' => $paystackResponse->url], __('Redirect Url'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }
  public function payOrder(OrderPaymentRequest $request)
  {

    $user = Auth::user();
    $order = DB::table('orders')->find($request->order_id);
    $amountKobo = round($order->grand_total * 100);
    try {
      $metadata = [
        'user_id' => $user->id,
        'order_id' => $order->id,
        'amount' => $order->grand_total,
      ];
      $paystackResponse = PaystackHelper::initializeTransaction($amountKobo, $user->email, route('order.callback'), $metadata);
      return $this->successResponse(['authorization_url' => $paystackResponse->url], __('Redirect Url'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }
  public function payRide(RidePaymentRequest $request)
  {
    $user = Auth::user();
    $ride = DB::table('rides')->find($request->ride_id);
    $amountKobo = round($ride->fare * 100);
    try {
      $metadata = [
        'user_id' => $user->id,
        'ride_id' => $ride->id,
        'amount' => $ride->fare,
      ];
      $paystackResponse = PaystackHelper::initializeTransaction($amountKobo, $user->email, route('ride.callback'), $metadata);
      return $this->successResponse(['authorization_url' => $paystackResponse->url], __('Redirect Url'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 400);
    }
  }
  public function walletCallback(Request $request)
  {
    try {
      //$paymentDetails = Paystack::getPaymentData();
      $reference = $request->query('reference');
      $paymentDetails = PaystackHelper::verifyTransaction($reference);
      $data = $paymentDetails['data'];
      $currency = Helpers::setting('currency', 'currency');
      if ($data['status'] === 'success') {
        $transaction = Transaction::where('txn_id', $data['reference'])->first();
        // If transaction already processed, skip
        if (!$transaction) {
          $metadata = $data['metadata']; //json_decode($data['metadata'], true);
          $user = User::find($metadata['user_id']);
          $wallets = new WalletService;
          $idempotencyKey = Str::uuid()->toString();
          $userType = strtolower($user->user_type);
          $ownerId = $userType == 'driver' ? $user->driverId : ($userType == 'store' ? $user->storeId : $user->id);
          $userWallet = $wallets->getOrCreate($userType, $ownerId);
          $wallets->creditEarning($userWallet, $metadata['amount'], 'wallet', $user->id, 'online', $idempotencyKey, 'Money Added to Wallet');
          $lastTxnId = WalletTransaction::where('wallet_id', $userWallet->id)
            ->latest('id')
            ->first()?->id ?? 0;
          DB::transaction(function () use ($user, $lastTxnId, $data, $metadata, $currency) {
            $this->transactionRepository->store([
              'user_id' => $user->id,
              'transactionable_type' => 'wallet',
              'transactionable_id' => $lastTxnId,
              'amount' => $metadata['amount'],
              'sub_amount' => $metadata['amount'],
              'tax' => 0,
              'discount' => 0,
              'currency' => $user->currency ?? $currency,
              'txn_id' => $data['reference'],
              'txn_details' => json_encode($data),
              'payment_mode' => 'paystack',
              'status' => 'success',
              'contact_details' => json_encode([
                'email' => $user->email ?? '',
                'phone' => $user->phone ?? '',
                'message' => 'Money added to wallet',
              ]),
            ]);
          });
          return redirect()->to('paystack-message/' . $data['reference'])->with('success', __('locale.Payment successfull'));
        }
        return redirect()->to('paystack-message')->with('success', __('locale.Transaction already saved'));
      }
      return redirect()->to('paystack-message')->with('warning', __('locale.Payment failed'));
    } catch (\Exception $e) {
      return redirect()->to('paystack-message')->with('warning', $e->getMessage());
    }
  }
  public function orderCallback(Request $request)
  {
    try {
      $reference = $request->query('reference');
      $paymentDetails = PaystackHelper::verifyTransaction($reference);
      $data = $paymentDetails['data'];
      $currency = Helpers::setting('currency', 'currency');
      if ($data['status'] === 'success') {
        $transaction = Transaction::where('txn_id', $data['reference'])->first();
        // If transaction already processed, skip
        if (!$transaction) {
          $metadata = $data['metadata'];
          $user = User::find($metadata['user_id']);
          $order = Order::find($metadata['order_id']);
          $store = Store::find($order->store_id);
          DB::transaction(function () use ($user, $order, $data, $metadata, $store, $currency) {
            $this->transactionRepository->store([
              'user_id' => $user->id,
              'transactionable_type' => 'order',
              'transactionable_id' => $order->id,
              'amount' => $metadata['amount'],
              'sub_amount' => $order->subtotal + $order->shipping,
              'tax' => $order->tax,
              'discount' => $order->discount,
              'currency' => $order->currency ?? $currency,
              'txn_id' => $data['reference'],
              'txn_details' => json_encode($data),
              'payment_mode' => 'paystack',
              'status' => 'success',
              'contact_details' => json_encode([
                'email' => $user->email ?? '',
                'phone' => $user->phone ?? '',
                'message' => 'Order Payment',
              ]),
            ]);
            if ($order->payment_status == 'Pending') {
              $message = "You have received a new order #{$order->order_id} amounting to {$order->currency}" . number_format($order->grand_total, 2);
              Helpers::notifications('Order', 'App\Notifications\Order', $message, $store->user_id, $order->id, 'orders', 'tabler-basket-check');
            }
            Order::where('id', $order->id)
              ->update([
                'payment_status' => 'Paid',
                'payment_type' => 'Online',
              ]);
            if (!empty($order->coupon_id)) {
              $couponUsage = new CouponUsage;
              $couponUsage->user_id = $order->user_id;
              $couponUsage->coupon_id = $order->coupon_id;
              $couponUsage->discount = $order->discount;
              $couponUsage->save();
            }
            if ($order->status == 'Out for Delivery') {
              $ride = Ride::where('order_id', $order->id)->first();
              if ($ride) {
                broadcast(new PaymentSuccessEvent($ride))->toOthers();
              }
            }
          });
          return redirect()->to('paystack-message/' . $data['reference'])->with('success', __('locale.Payment successfull'));
        }
        return redirect()->to('paystack-message')->with('success', __('locale.Transaction already saved'));
      }
      return redirect()->to('paystack-message')->with('warning', __('locale.Payment failed'));
    } catch (\Exception $e) {
      return redirect()->to('paystack-message')->with('warning', $e->getMessage());
    }
  }
  public function rideCallback(Request $request)
  {
    try {
      $reference = $request->query('reference');
      $paymentDetails = PaystackHelper::verifyTransaction($reference);
      $data = $paymentDetails['data'];
      $currency = Helpers::setting('currency', 'currency');
      if ($data['status'] === 'success') {
        $transaction = Transaction::where('txn_id', $data['reference'])->first();
        // If transaction already processed, skip
        if (!$transaction) {
          $metadata = $data['metadata'];
          $user = User::find($metadata['user_id']);
          $ride = Ride::find($metadata['ride_id']);
          DB::transaction(function () use ($user, $ride, $data, $metadata, $currency) {
            $invoice = json_decode($ride->invoice_details);
            $this->transactionRepository->store([
              'user_id' => $user->id,
              'transactionable_type' => 'ride',
              'transactionable_id' => $ride->id,
              'amount' => $metadata['amount'],
              'sub_amount' => $invoice->sub_total ?? $ride->fare,
              'tax' => 0,
              'discount' => $invoice->discount ?? 0,
              'currency' => $ride->currency ?? $currency,
              'txn_id' => $data['reference'],
              'txn_details' => json_encode($data),
              'payment_mode' => 'stripe',
              'status' => 'success',
              'contact_details' => json_encode([
                'email' => $user->email ?? '',
                'phone' => $user->phone ?? '',
                'message' => $ride->type == 'Ride' ? 'Ride Payment' : 'Parcel Ride Payment',
              ]),
            ]);

            Ride::where('id', $ride->id)
              ->update([
                'payment_type' => 'online',
                'payment_status' => 'Paid',
              ]);
            broadcast(new PaymentSuccessEvent($ride))->toOthers();
            // coupon usage alrealy in endRide
          });
          return redirect()->to('paystack-message/' . $data['reference'])->with('success', __('locale.Payment successfull'));
        }
        return redirect()->to('paystack-message')->with('success', __('locale.Transaction already saved'));
      }
      return redirect()->to('paystack-message')->with('warning', __('locale.Payment failed'));
    } catch (\Exception $e) {
      return redirect()->to('paystack-message')->with('warning', $e->getMessage());
    }
  }

  public function walletWebhook(Request $request)
  {
    // Get Paystack signature
    $paystackSignature = $request->header('x-paystack-signature');

    // Verify signature
    $secretKey = config('paystack.secret');
    $computedSignature = hash_hmac('sha512', $request->getContent(), $secretKey);

    if ($paystackSignature !== $computedSignature) {
      return response()->json(['status' => false, 'message' => 'Invalid signature'], 400);
    }

    $event = $request->all();
    $data = $event['data'] ?? null;

    if ($data && $data['status'] === 'success') {
      $metadata = $data['metadata'];

      $userId = $metadata['user_id'] ?? null;
      $amount = $metadata['amount'] ?? null;
      $reference = $data['reference'] ?? null;

      if ($userId && $amount && $reference) {
        $transaction = Transaction::where('txn_id', $reference)->first();

        // If transaction already processed, skip
        if (!$transaction) {
          $user = User::find($metadata['user_id']);
          $wallets = new WalletService;
          $idempotencyKey = Str::uuid()->toString();
          $userType = strtolower($user->user_type);
          $ownerId = $userType == 'driver' ? $user->driverId : ($userType == 'store' ? $user->storeId : $user->id);
          $userWallet = $wallets->getOrCreate($userType, $ownerId);
          $wallets->creditEarning($userWallet, $metadata['amount'], 'wallet', $user->id, 'online', $idempotencyKey, 'Money Added to Wallet');
          $lastTxnId = WalletTransaction::where('wallet_id', $userWallet->id)
            ->latest('id')
            ->first()?->id ?? 0;
          DB::transaction(function () use ($user, $lastTxnId, $data, $metadata) {
            $this->transactionRepository->store([
              'user_id' => $user->id,
              'transactionable_type' => 'wallet',
              'transactionable_id' => $lastTxnId,
              'amount' => $metadata['amount'],
              'sub_amount' => $metadata['amount'],
              'tax' => 0,
              'discount' => 0,
              'currency' => $user->currency,
              'txn_id' => $data['reference'],
              'txn_details' => json_encode($data),
              'payment_mode' => 'stripe',
              'status' => 'success',
              'contact_details' => json_encode([
                'email' => $user->email ?? '',
                'phone' => $user->phone ?? '',
                'message' => 'Money added to wallet',
              ]),
            ]);
          });
        }
      }
    }

    return response()->json(['status' => true]);
  }
  public function orderWebhook(Request $request)
  {
    // Get Paystack signature
    $paystackSignature = $request->header('x-paystack-signature');

    // Verify signature
    $secretKey = config('paystack.secret');
    $computedSignature = hash_hmac('sha512', $request->getContent(), $secretKey);

    if ($paystackSignature !== $computedSignature) {
      return response()->json(['status' => false, 'message' => 'Invalid signature'], 400);
    }

    $event = $request->all();
    $data = $event['data'] ?? null;

    if ($data && $data['status'] === 'success') {
      $metadata = $data['metadata'];

      $userId = $metadata['user_id'] ?? null;
      $amount = $metadata['amount'] ?? null;
      $reference = $data['reference'] ?? null;

      if ($userId && $amount && $reference) {
        $transaction = Transaction::where('txn_id', $reference)->first();

        // If transaction already processed, skip
        if (!$transaction) {
          $user = User::find($metadata['user_id']);
          $order = Order::find($metadata['order_id']);
          DB::transaction(function () use ($user, $order, $data, $metadata) {
            $this->transactionRepository->store([
              'user_id' => $user->id,
              'transactionable_type' => 'order',
              'transactionable_id' => $order->id,
              'amount' => $metadata['amount'],
              'sub_amount' => $order->subtotal + $order->shipping,
              'tax' => $order->tax,
              'discount' => $order->discount,
              'currency' => $order->currency,
              'txn_id' => $data['reference'],
              'txn_details' => json_encode($data),
              'payment_mode' => 'paystack',
              'status' => 'success',
              'contact_details' => json_encode([
                'email' => $user->email ?? '',
                'phone' => $user->phone ?? '',
                'message' => 'Order Payment',
              ]),
            ]);

            Order::where('id', $order->id)
              ->update([
                'payment_status' => 'Paid',
                'payment_type' => 'Online',
              ]);
            if (!empty($order->coupon_id)) {
              $couponUsage = new CouponUsage;
              $couponUsage->user_id = $order->user_id;
              $couponUsage->coupon_id = $order->coupon_id;
              $couponUsage->discount = $order->discount;
              $couponUsage->save();
            }
            if ($order->status == 'Out for Delivery') {
              $ride = Ride::where('order_id', $order->id)->first();
              if ($ride) {
                broadcast(new PaymentSuccessEvent($ride))->toOthers();
              }
            }
          });
        }
      }
    }

    return response()->json(['status' => true]);
  }

  public function rideWebhook(Request $request)
  {
    // Get Paystack signature
    $paystackSignature = $request->header('x-paystack-signature');

    // Verify signature
    $secretKey = config('paystack.secret');
    $computedSignature = hash_hmac('sha512', $request->getContent(), $secretKey);

    if ($paystackSignature !== $computedSignature) {
      return response()->json(['status' => false, 'message' => 'Invalid signature'], 400);
    }

    $event = $request->all();
    $data = $event['data'] ?? null;

    if ($data && $data['status'] === 'success') {
      $metadata = $data['metadata'];

      $userId = $metadata['user_id'] ?? null;
      $amount = $metadata['amount'] ?? null;
      $reference = $data['reference'] ?? null;

      if ($userId && $amount && $reference) {
        $transaction = Transaction::where('txn_id', $reference)->first();

        // If transaction already processed, skip
        if (!$transaction) {
          $user = User::find($metadata['user_id']);
          $ride = Ride::find($metadata['ride_id']);
          DB::transaction(function () use ($user, $ride, $data, $metadata) {
            $invoice = json_decode($ride->invoice_details);
            $this->transactionRepository->store([
              'user_id' => $user->id,
              'transactionable_type' => 'ride',
              'transactionable_id' => $ride->id,
              'amount' => $metadata['amount'],
              'sub_amount' => $invoice->sub_total ?? $ride->fare,
              'tax' => 0,
              'discount' => $invoice->discount ?? 0,
              'currency' => $ride->currency,
              'txn_id' => $data['reference'],
              'txn_details' => json_encode($data),
              'payment_mode' => 'stripe',
              'status' => 'success',
              'contact_details' => json_encode([
                'email' => $user->email ?? '',
                'phone' => $user->phone ?? '',
                'message' => $ride->type == 'Ride' ? 'Ride Payment' : 'Parcel Ride Payment',
              ]),
            ]);

            Ride::where('id', $ride->id)
              ->update([
                'payment_type' => 'online',
                'payment_status' => 'Paid',
              ]);
            broadcast(new PaymentSuccessEvent($ride))->toOthers();
            // coupon usage alrealy in endRide
          });
        }
      }
    }

    return response()->json(['status' => true]);
  }

}
