<?php
namespace App\Http\Controllers\API;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use Auth, Chat;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use App\Helpers\Helpers;
use App\Traits\ApiResponseTrait;
use Carbon\Carbon;
use App\Events\ChatMessage;
use Musonza\Chat\Models\Participation;


class ChatAppController extends Controller
{
  use ApiResponseTrait;
  public function index(Request $request)
  {
    $user = User::find(auth('sanctum')->id());
    $conversations = Chat::conversations()->setParticipant($user)
      ->setPaginationParams([
        'page' => 1,
        'perPage' => 200,
        'sorting' => 'desc',
        'columns' => ['*'],
        'pageName' => ''
      ])
      ->get()
      ->load('conversation');  // Ensure conversation model is loaded
    $usersData = [];
    $totalUnreadCount = $unreadCount = 0;
    foreach ($conversations as $conversation) {
      $conversationData = $conversation->data;

      if (!empty($conversationData)) {
        $conversationModel = $conversation->conversation;

        $unreadCount = Chat::conversation($conversationModel)->setParticipant($user)->unreadCount();
        $totalUnreadCount += $unreadCount;

        $lastMessage = $this->getLastMessageDetails($conversationModel);

        $usersData = [
          'id' => $conversation->id,
          'convo_id' => $conversationModel->id,
          'last_message' => $lastMessage['body'],
          'last_message_type' => $lastMessage['type'],
          'last_time' => $lastMessage['time'],
          'last_date' => $lastMessage['date'],
          'dateTime' => $lastMessage['dateTime'],
          'count' => $unreadCount,
          'user_id' => $user->id,
        ];
      }
    }
    // Sort users by dateTime in descending order
    array_multisort(array_column($usersData, 'dateTime'), SORT_DESC, $usersData);
    return $this->successResponse(['unread' => $totalUnreadCount, 'users' => $usersData,], !empty($usersData) ? 'User Chat List' : 'No Chat List Found', 200);
  }

  /* Get user profile image based on user type and availability of profile photo. */
  private function getUserImage($user)
  {
    if ($user->user_type === 'Admin') {
      return $user->profile_photo_path
        ? $user->profile_photo_url
        : asset('assets/img/branding/logo.png');
    } else {
      return !empty($user->profile_photo_path)
        ? asset('storage/users') . '/' . $user->profile_photo_path
        : ($user->profile_photo_url ?: asset("assets/img/default.png"));
    }
  }

  /**  * Get the last message details, including body, type, and formatted timestamps. */
  private function getLastMessageDetails($conversationModel)
  {
    $lastMessage = $conversationModel['last_message'];
    $body = $type = $time = $date = $dateTime = '';
    if (!empty($lastMessage)) {
      $body = $lastMessage['body'];
      $type = $lastMessage['type'];
      $time = $lastMessage['created_at']->format('H:i');
      $date = $lastMessage['created_at']->format('d M Y');
      $dateTime = date('YmdHis', strtotime($lastMessage['created_at']));
    }

    return compact('body', 'type', 'time', 'date', 'dateTime');
  }

  /** * Check if the current user is blocked by or has blocked the other user. */
  private function isUserBlocked($currentUser, $otherUser)
  {
    return DB::table('block_lists')
      ->where(function ($query) use ($currentUser, $otherUser) {
        $query->where('user_id', $currentUser->id)->where('blocked_id', $otherUser->id)
          ->orWhere(function ($query) use ($currentUser, $otherUser) {
            $query->where('user_id', $otherUser->id)->where('blocked_id', $currentUser->id);
          });
      })->exists();
  }


  public function chatStart($user_id, $conversationid = null)
  {
    $page = request()->get('page', 1);
    $user = User::findOrFail($user_id);
    $myUser = User::find(Auth::id());
    // Find any existing private conversation between the two users
    $existingConversation = DB::table('chat_participation')->select('conversation_id', DB::raw('COUNT(*) as count'))->whereIn('messageable_id', [$myUser->id, $user->id])->groupBy('conversation_id')->having('count', '>', 1)->pluck('conversation_id')->first();
    if (!$existingConversation) {
      /*   $participants = $user;
        // Add $user to participants if not already included
        if (!$participants->contains('id', $myUser->id)) {
          $participants->push($myUser);
        }
      $participantArray = $participants->all();
      */
      $participants = [$myUser, $user];
      $conversation = Chat::createConversation($participants)->makePrivate(true);
      $conversation->update([
        'data' => ['title' => 'Laravel Channel', 'description' => 'Laravel Channel Description',]
      ]);
      $conversationId = $conversation->id;
    } else {
      $conversationId = $existingConversation;
    }
    return $this->chat($user->id, $conversationId, $page);
  }

  public function chat($user_id = null, $conversationid = null, $page = 1)
  {
    $user = User::find($user_id);
    $myuser = User::find(auth('sanctum')->id());
    $conversation = Chat::conversations()->getById($conversationid);
    $perPage = 10;
    // Fetch messages
    $messages = Chat::conversation($conversation)->setParticipant($myuser)->setPaginationParams([
      'page' => $page,
      'perPage' => $perPage,
      'sorting' => 'desc'
    ])->getMessages();
    $lastPage = $messages->lastPage();
    $currentPage = $messages->currentPage();
    $messages = $messages->sortBy('created_at');
    // Mark messages as read
    Chat::conversation($conversation)->setParticipant($myuser)->readAll();


    // Check if the conversation is blocked/muted
    $participation = Chat::conversation($conversation)->getParticipation($myuser);
    $isBlocked = $participation && isset($participation->settings['mute_mentions'])
      ? $participation->settings['mute_mentions'] == 1
      : false;
    $blockStatus = $isBlocked ? 'Blocked' : 'Unblock';
    $blockUrl = url('api/block-user') . '/' . $conversationid . ($isBlocked ? '/UnBlocked' : '/Blocked');
    $onlineStatus = Cache::has('user-is-online-' . $user->id) ? 'online' : 'offline';
    $lastSeen = $onlineStatus === 'online' ? Carbon::parse($user->last_seen)->diffForHumans() : Carbon::parse($user->created_at)->diffForHumans();
    $image = $this->getUserImage($user);

    // Prepare chat header
    $header = [
      'image' => $image,
      'online_status' => $onlineStatus,
      'name' => $user->name,
      'phone' => $user->phone,
      'email' => $user->email,
      'last_seen' => $lastSeen,
      'block_url' => $blockUrl,
      'block_status' => $blockStatus,
      'clear_chat_url' => url('api/clear-chat') . '/' . $conversationid,
    ];

    // Process messages
    $htmlMessages = [];
    $previousDate = '';
    $adminToggle = $userToggle = 'No';
    $htmlMessages = $messages->map(function ($message) use (&$previousDate, &$adminToggle, &$userToggle, $conversationid) {
      $messageData = $message->type === 'reply' || $message->type === 'attachment' ? $message->data : '';
      $body = $message->body;
      $replyMedia = $message->type === 'reply' ? (!empty($messageData['media']) ? $messageData['media'] : $messageData['title']) : null;
      $attachmentImage = !empty($messageData['file_name']) ? asset('storage/Chat') . '/' . $conversationid . '/' . $messageData['file_name'] : '';
      $attachmentType = !empty($messageData['file_type']) ? $messageData['file_type'] : '';

      $messageDetails = [
        '_id' => $message->id,
        'text' => in_array($message->type, ['Video', 'Image'])
          ? url('storage/Chat/' . $conversationid . '/' . $message->body)
          : $body,
        'createdAt' => $message->created_at->toIso8601String(),
        'type' => $message->type,
        'attachment_image' => $attachmentImage,
        'attachment_type' => $attachmentType,
        'message_time' => $message->created_at->format('H:i'),
        'divider_date' => $previousDate != $message->created_at->format('Ymd') ? $message->created_at->format('d M Y') : '',
        'reply_media' => $replyMedia,
        'reply_type' => $messageData['sub_type'] ?? null,
        'reply_user_name' => $messageData['user_name'] ?? null,
        'reply_user_id' => $messageData['user_id'] ?? null,
        'link_id' => $messageData['id'] ?? null,
      ];

      $otherUser = User::find($message->sender->id);
      $userImage = $this->getUserImage($otherUser);

      // Determine user details for the message
      if ($message->is_sender == 1 && $message->sender->id == Auth::id()) {
        $messageDetails['user'] = [
          'id' => Auth::id(),
          'name' => Auth::user()->name,
          'avatar' => $adminToggle == 'Yes' ? '' : $userImage,
        ];
        $adminToggle = 'Yes';
        $userToggle = 'No';
      } else {
        $sender = User::find($message->sender->id);
        $messageDetails['user'] = [
          // 'data'=>$message,
          'id' => $sender->id,
          'name' => $sender->name ?? '',
          'avatar' => $userToggle == 'Yes' ? '' : $userImage,
        ];
        $adminToggle = 'No';
        $userToggle = 'Yes';
      }

      $previousDate = $message->created_at->format('Ymd');
      return $messageDetails;
    })->filter()->values()->all();


    return $this->successResponse([
      'chat' => [
        'header' => $header,
        'messages' => $htmlMessages
      ],
      'convo_id' => $conversationid,
      'current_page' => !empty($currentPage) ? $currentPage : 1,
      'next_page_url' => (!empty($currentPage) && $currentPage < $lastPage)
        ? url('api/start-chat') . '/' . $user->id . '?page=' . ($currentPage + 1)
        : '',
    ], 'Chat Uploaded Successfully', 200);

  }

  public function sendMessage(Request $request)
  {
    $myUser = User::find(auth('sanctum')->id());
    $conversationId = $request->conversation_id;
    // Define message type and text
    if (!empty($request->message) && ($request->hasFile('icon'))) {
      $textMessage = $request->message;
      $prepareFile = $this->prepareFile($request);
      $ext = Helpers::typ(pathinfo($prepareFile, PATHINFO_EXTENSION));
      $messageType = 'attachment';
      $data = ['file_name' => $prepareFile, 'file_type' => $ext];
    } else {
      $prepareFile = $this->prepareFile($request);
      $textMessage = $request->message ?? $prepareFile;
      $messageType = $request->message ? 'text' : Helpers::typ(pathinfo($textMessage, PATHINFO_EXTENSION));
      $data = array();
    }
    // Send message
    $conversation = Chat::conversations()->getById($conversationId);
    $message = Chat::message($textMessage)
      ->type($messageType)
      ->data($data)
      ->from($myUser)
      ->to($conversation)
      ->send();

    // Store file if applicable
    if ($request->hasFile('icon')) {
      $this->storeFile($request->file('icon'), $conversationId, $prepareFile);
    }
    $otherUser = User::find($message->sender->id);
    $response = $this->prepareMessageData($message, $myUser, $conversation, $textMessage, $otherUser);
    ChatMessage::dispatch($response);
    return $this->successResponse([], 'Message sent successfully', 200);
  }

  private function prepareFile(Request $request)
  {
    if ($request->hasFile('icon')) {
      $image = $request->file('icon');
      return 'MGC-' . time() . '.' . $image->extension();
    }
    return null;
  }


  private function storeFile($file, $conversationId, $prepareFile)
  {
    $path = storage_path("app/public/Chat/{$conversationId}");
    if (!File::exists($path)) {
      File::makeDirectory($path, 0755, true);
    }
    $file->move($path, $prepareFile);
  }

  private function prepareMessageData($message, $myUser, $conversation, $messageText, $otherUser, $title = null)
  {
    $url = config('app.url');
    $avatar = $this->getUserImage($myUser);
    //for single user chat
    //$count = Chat::conversation($conversation)->setParticipant($otherUser)->unreadCount();
    //$unreadCount = Chat::messages()->setParticipant($otherUser)->unreadCount();
    //for single user chat
    $createdAt = now();
    //for mobile
    $messageData = $message->type === 'reply' || $message->type === 'attachment' ? $message->data : '';
    $body = $message->body;
    $replyMedia = $message->type === 'reply' ? (!empty($messageData['media']) ? $messageData['media'] : $messageData['title']) : null;
    $attachmentImage = !empty($messageData['file_name']) ? asset('storage/Chat') . '/' . $conversation->id . '/' . $messageData['file_name'] : '';
    $attachmentType = !empty($messageData['file_type']) ? $messageData['file_type'] : '';
    //for mobile
    return [
      'userid' => $myUser->id,
      'conversationid' => $conversation->id,
      'name' => $myUser->name,
      'image' => $avatar,
      'message' => $messageText,
      'count' => !empty($count) ? $count : 1,
      'reciever_id' => $otherUser->id,
      'total_count' => !empty($unreadCount) ? $count : 1,
      'created_at' => $createdAt->format('H:i'),
      'title' => $title,
      /*  'data' => [
         '_id' => $message->id,
         'text' => $messageText,
         'createdAt' => $message->created_at->toIso8601String(),
         'message_type' => $message->type,
         'data' => $message->data,
         'type' => filter_var($messageText, FILTER_VALIDATE_URL) ? Helpers::typ(pathinfo($messageText, PATHINFO_EXTENSION)) : 'text',
         'message_time' => $createdAt->format('H:i'),
         'divider_date' => '', //$createdAt->format('d M Y'),
         'user' => [
           'id' => $myUser->id,
           '_id' => $myUser->id,
           'name' => $myUser->name,
           'avatar' => $avatar,
         ]
       ], */
      'data' => [
        '_id' => $message->id,
        'text' => in_array($message->type, ['Video', 'Image'])
          ? url('storage/Chat/' . $conversation->id . '/' . $message->body)
          : $body,
        'createdAt' => $message->created_at->toIso8601String(),
        'type' => $message->type,
        'attachment_image' => $attachmentImage,
        'attachment_type' => $attachmentType,
        'message_time' => $message->created_at->format('H:i'),
        'divider_date' => '',
        'reply_media' => $replyMedia,
        'reply_type' => $messageData['sub_type'] ?? null,
        'reply_user_name' => $messageData['user_name'] ?? null,
        'reply_user_id' => $messageData['user_id'] ?? null,
        'link_id' => $messageData['id'] ?? null,
        'user' => [
          'id' => $myUser->id,
          '_id' => $myUser->id,
          'name' => $myUser->name,
          'avatar' => $avatar,
        ]
      ],
    ];
  }

  public function unreadMesg($convo_id)
  {
    $user = User::find(auth('sanctum')->id());
    $conversation = Chat::conversations()->getById($convo_id);
    Chat::conversation($conversation)->setParticipant($user)->readAll();
    return $this->successResponse([], 'Mark all messages read', 200);
  }

  public function countUnreadMesg()
  {
    $user = User::find(auth('sanctum')->id());
    $count = Chat::messages()->setParticipant($user)->unreadCount();
    return $this->successResponse(['count' => $count], 'All Unread Message', 200);
  }

  public function chatBlock($conversationid, $status)
  {
    $user = User::find(auth('sanctum')->id());
    if ($status == 'Blocked') {
      $settings = ['mute_mentions' => true];
    } else {
      $settings = ['mute_mentions' => false];
    }
    $conversation = Chat::conversations()->getById($conversationid);
    $con = Chat::conversation($conversation)->getParticipation($user)->update(['settings' => $settings]);
    return $this->successResponse([], __(' locale.You blocked user successfully'), 200);
  }

  public function chatClear($conversationid)
  {
    $user = User::find(auth('sanctum')->id());
    $conversation = Chat::conversations()->getById($conversationid);
    Chat::conversation($conversation)->setParticipant($user)->clear();
    return $this->successResponse([], __(' locale.Your chat history is cleared'), 200);
  }

  public function messageDelete($id)
  {
    $user = User::find(auth('sanctum')->id());
    $message = Chat::messages()->getById($id);
    //Chat::message($message)->setParticipant($user)->delete();  //delete for me
    DB::table('chat_messages')->where('id', $id)->delete(); //delete from all
    return $this->successResponse([], __(' locale.Message delete successfully'), 200);
  }

  public function replyMessage(Request $request, $message_id)
  {
    try {
      $msg = Chat::messages()->getById($message_id);
      if (!$msg) {
        return $this->errorResponse([], __('locale.Message not found'), 400);
      }

      $myuser = User::find(auth('sanctum')->id());
      if (!$myuser) {
        return $this->errorResponse([], __('locale.Unauthorized'), 401);
      }

      $textMessage = $request->message;

      // Get conversation
      $conversation = Chat::conversations()->getById($msg->conversation_id);
      if (!$conversation) {
        return $this->errorResponse([], __('locale.Conversation not found'), 400);
      }

      // Get the original participation (sender)
      $participant = Participation::find($msg->participation_id);
      if (!$participant) {
        return $this->errorResponse([], __('locale.Participation not found'), 400);
      }

      // Reply reference details
      $media = in_array($msg->type, ['Image', 'Video'])
        ? asset('storage/Chat/' . $conversation->id . '/' . $msg->body)
        : null;
      $sub_type = in_array($msg->type, ['Image', 'Video']) ? $msg->type : 'text';

      $post_details = [
        'reply_to_message_id' => $msg->id,
        'reply_to_user_id' => $participant->messageable_id,
        'reply_to_user_name' => $participant->messageable?->name ?? 'Unknown',
        'reply_to_body' => $msg->body,
        'reply_to_media' => $media,
        'reply_to_type' => $msg->type,
        'sub_type' => $sub_type,
        'message' => $textMessage,
      ];

      // ✅ Send reply
      $message = Chat::message($textMessage)
        ->type('reply')
        ->data($post_details)
        ->from($myuser)
        ->to($conversation) // pass participant here
        ->send();
      return $this->successResponse($message, __('locale.Reply on message sent successfully'), 200);
    } catch (\Exception $e) {
      return $this->errorResponse([], $e->getMessage(), 500);
    }
  }


}
