<?php
namespace App\Http\Controllers\API;
use App\Http\Controllers\Controller;
use App\Models\Contact;
use App\Models\Setting;
use Exception;
use App\Mail\SupportEmail;
use App\Traits\ApiResponseTrait;
use App\Models\Country;
use App\Models\State;
use App\Models\City;
use App\Models\Faq;
use App\Models\VehicleModel;
use App\Helpers\Helpers;
use Illuminate\Support\Facades\Mail;
use App\Http\Requests\StoreTicketRequest;
use App\Http\Requests\WithdrawalFormRequest;
use App\Repositories\Contracts\WithdrawalRequestRepositoryInterface;
use App\Repositories\Contracts\VehicleBrandRepositoryInterface;
use App\Repositories\Contracts\VehicleCategoryRepositoryInterface;
use App\Repositories\Contracts\VehicleModelRepositoryInterface;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
class ApiController extends Controller
{
  use ApiResponseTrait;
  use ApiResponseTrait;
  protected $walletRepository, $vehicleCategoryRepository, $vehicleBrandRepository, $vehicleModelRepository;
  function __construct(WithdrawalRequestRepositoryInterface $walletRepository, VehicleCategoryRepositoryInterface $vehicleCategoryRepository, VehicleBrandRepositoryInterface $vehicleBrandRepository, VehicleModelRepositoryInterface $vehicleModelRepository)
  {
    $this->walletRepository = $walletRepository;
    $this->vehicleCategoryRepository = $vehicleCategoryRepository;
    $this->vehicleBrandRepository = $vehicleBrandRepository;
    $this->vehicleModelRepository = $vehicleModelRepository;
  }
  public function createTicket(StoreTicketRequest $request)
  {
    $category = $request->category;
    $contact = new Contact;
    $date = date('Y-m-d H:i:s'); //this returns the current date time
    $middle = strtotime($date);
    $un_id = 'T' . $middle;
    $contact->ticket_no = $un_id;
    $contact->user_id = !empty($request->user_id) ? $request->user_id : 0;
    $contact->email = $request->email;
    $contact->username = $request->name;
    $contact->category = $category;
    $contact->subject = $request->subject;
    $contact->body = $request->message;
    $contact->status = 'Open';
    $contact->save();
    $message = "Dear admin, A new support ticket No. $un_id has been issued to a user $request->email dated $date. Please review this ticket and take action.";
    Helpers::adminNotification('Support Tickets', 'App\Models\Contact', $message, $contact->id, 'tickets', $request->user_id ?? 0);
    //send mail
    $array['view'] = 'emails.supportTicket';
    $array['subject'] = $request->subject;
    $array['name'] = $request->person_name;
    $array['data'] = $contact;
    $array['content'] = "Your ticket for issue '" . $category . "' generated with ticket no " . $un_id . " Admin will contact you at the earlist.  ";
    try {
      Mail::to($request->person_email)->queue(new SupportEmail($array));
    } catch (Exception $e) {
    }
    return $this->successResponse([], __('Ticket no: ' . $un_id . ' created successfully. Admin will contact you at the earlist'), 200);
  }
  public function getTickets()
  {
    $contacts = Contact::where('user_id', auth('sanctum')->id())->get();
    $contacts = $contacts->map(function ($contact) {
      $contact->date = date('d-m-Y', strtotime($contact->created_at));
      return $contact;
    });
    if ($contacts->isNotEmpty()) {
      return $this->successResponse($contacts, __('locale.Support Ticket'), 200);
    } else {
      return $this->successResponse([], __('locale.No Support Ticket Yet'), 200);
    }
  }
  public function countries()
  {
    $countries = Country::select('name', 'id')->orderBy('name', 'asc')->get();
    $countries->map(function ($results) {
      return [
        'id' => $results->id,
        'name' => $results->name,
      ];
    });
    return $this->successResponse($countries, '', 200);
  }

  public function states($country_id)
  {
    $states = State::where('country_id', $country_id)->select('name', 'id')->orderBy('name', 'asc')->get();
    $states->map(function ($results) {
      return [
        'id' => $results->id,
        'name' => $results->name,
      ];
    });
    return $this->successResponse($states, '', 200);
  }

  public function cities($state_id)
  {
    $cities = City::where('state_id', $state_id)->select('name', 'id')->orderBy('name', 'asc')->get();
    $cities->map(function ($results) {
      return [
        'id' => $results->id,
        'name' => $results->name,
      ];
    });
    return $this->successResponse($cities, '', 200);
  }
  public function support()
  {
    $keys = ['contact_number', 'email', 'address', 'admin-chat'];
    $settings = Setting::whereIn('meta_key', $keys)
      ->get(['meta_key', 'meta_value'])
      ->mapWithKeys(function ($item) {
        return [$item->meta_key => $item->meta_value];
      });
    return $this->successResponse($settings, '', 200);
  }

  public function faqs()
  {
    $faqs = Faq::where('status', 1)->get();
    return $this->successResponse($faqs, '', 200);
  }

  public function store(WithdrawalFormRequest $request)
  {
    $wallet = $this->walletRepository->store($request, null);
    return $this->successResponse($wallet, __('locale.Withdrawal Request Created Successfully!'));
  }

  public function categories(Request $request)
  {
    $id = $request->input('id') ?? null;
    return $this->successResponse($this->vehicleCategoryRepository->subCategories($id, $request->input('type')), '', 200);
  }
  public function brandDropdown()
  {
    return $this->successResponse($this->vehicleBrandRepository->dropdown(), '', 200);
  }
  public function modelDropdown(Request $request)
  {
    $brandId = $request->vehicle_brand_id;
    $categoryId = $request->vehicle_category_id;
    $parentCategoryId = $request->vehicle_parent_category_id;
    if (!$brandId || !$categoryId || !$parentCategoryId) {
      return $this->successResponse([], '', 200);
    }
    $models = VehicleModel::query()
      ->where('vehicle_brand_id', $brandId)
      ->where('vehicle_category_id', $categoryId);
    if (!empty($parentCategoryId)) {
      $models->where('vehicle_parent_id', $parentCategoryId);
    }
    $models = $models->select('id', 'name')->orderBy('name')->get();
    return $this->successResponse($models, '', 200);
  }
  public function colorDropdown()
  {
    $colors = DB::table('colors')->select('id', 'name', 'code')->orderBy('name')->get();
    return $this->successResponse($colors, '', 200);
  }

}
