<?php
namespace App\Helpers;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Str;
use App\Models\Setting;
use App\Models\User;
use App\Models\Order;
use App\Models\WithdrawalRequest;
use App\Models\Contact;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use App\Notifications\UserNotification;
use App\Events\NotificationEvent;
use Musonza\Chat\Models\MessageNotification;
use Auth, Chat;
class Helpers
{
  /**
   * Generate menu attributes for semi-dark mode
   *
   * @param bool $semiDarkEnabled Whether semi-dark mode is enabled
   * @return array HTML attributes for the menu element
   */
  public static function getMenuAttributes($semiDarkEnabled)
  {
    $attributes = [];

    if ($semiDarkEnabled) {
      $attributes['data-bs-theme'] = 'dark';
    }

    return $attributes;
  }

  public static function appClasses()
  {

    $data = config('custom.custom');


    // default data array
    $DefaultData = [
      'myLayout' => 'vertical',
      'myTheme' => 'light',
      'mySkins' => 'default',
      'hasSemiDark' => false,
      'myRTLMode' => true,
      'hasCustomizer' => true,
      'showDropdownOnHover' => true,
      'displayCustomizer' => true,
      'contentLayout' => 'compact',
      'headerType' => 'fixed',
      'navbarType' => 'sticky',
      'menuFixed' => true,
      'menuCollapsed' => false,
      'footerFixed' => false,
      'customizerControls' => [
        'color',
        'theme',
        'skins',
        'semiDark',
        'layoutCollapsed',
        'layoutNavbarOptions',
        'headerType',
        'contentLayout',
        'rtl'
      ],
      //   'defaultLanguage'=>'en',
    ];

    // if any key missing of array from custom.php file it will be merge and set a default value from dataDefault array and store in data variable
    $data = array_merge($DefaultData, $data);

    // All options available in the template
    $allOptions = [
      'myLayout' => ['vertical', 'horizontal', 'blank', 'front'],
      'menuCollapsed' => [true, false],
      'hasCustomizer' => [true, false],
      'showDropdownOnHover' => [true, false],
      'displayCustomizer' => [true, false],
      'contentLayout' => ['compact', 'wide'],
      'headerType' => ['fixed', 'static'],
      'navbarType' => ['sticky', 'static', 'hidden'],
      'myTheme' => ['light', 'dark', 'system'],
      'mySkins' => ['default', 'bordered', 'raspberry'],
      'hasSemiDark' => [true, false],
      'myRTLMode' => [true, false],
      'menuFixed' => [true, false],
      'footerFixed' => [true, false],
      'customizerControls' => [],
      // 'defaultLanguage'=>array('en'=>'en','fr'=>'fr','de'=>'de','ar'=>'ar'),
    ];

    //if myLayout value empty or not match with default options in custom.php config file then set a default value
    foreach ($allOptions as $key => $value) {
      if (array_key_exists($key, $DefaultData)) {
        if (gettype($DefaultData[$key]) === gettype($data[$key])) {
          // data key should be string
          if (is_string($data[$key])) {
            // data key should not be empty
            if (isset($data[$key]) && $data[$key] !== null) {
              // data key should not be exist inside allOptions array's sub array
              if (!array_key_exists($data[$key], $value)) {
                // ensure that passed value should be match with any of allOptions array value
                $result = array_search($data[$key], $value, 'strict');
                if (empty($result) && $result !== 0) {
                  $data[$key] = $DefaultData[$key];
                }
              }
            } else {
              // if data key not set or
              $data[$key] = $DefaultData[$key];
            }
          }
        } else {
          $data[$key] = $DefaultData[$key];
        }
      }
    }
    $themeVal = $data['myTheme'] == "dark" ? "dark" : "light";
    $themeUpdatedVal = $data['myTheme'] == "dark" ? "dark" : $data['myTheme'];

    // Determine if the layout is admin or front based on template name
    $layoutName = $data['myLayout'];
    $isAdmin = !Str::contains($layoutName, 'front');

    $modeCookieName = $isAdmin ? 'admin-mode' : 'front-mode';
    $colorPrefCookieName = $isAdmin ? 'admin-colorPref' : 'front-colorPref';
    $primaryColorCookieName = $isAdmin ? 'admin-primaryColor' : 'front-primaryColor';

    // Get primary color from custom.php if explicitly set
    $primaryColor = null;  //null
    if (array_key_exists('primaryColor', $data)) {
      $primaryColor = $data['primaryColor'];
    }

    // Check for primary color in cookie
    if (isset($_COOKIE[$primaryColorCookieName])) {
      $primaryColor = $_COOKIE[$primaryColorCookieName];
    }

    // Determine style based on cookies, only if not 'blank-layout'
    if ($layoutName !== 'blank') {
      if (isset($_COOKIE[$modeCookieName])) {
        $themeVal = $_COOKIE[$modeCookieName];
        if ($themeVal === 'system') {
          $themeVal = isset($_COOKIE[$colorPrefCookieName]) ? $_COOKIE[$colorPrefCookieName] : 'light';
        }
        $themeUpdatedVal = $_COOKIE[$modeCookieName];
      }
    }

    // Define standardized cookie names
    $skinCookieName = 'customize_skin';
    $semiDarkCookieName = 'customize_semi_dark';

    // Process skin and semi-dark settings only for admin layouts
    if ($isAdmin) {
      // Get skin from cookie or fall back to config
      $skinFromCookie = isset($_COOKIE[$skinCookieName]) ? $_COOKIE[$skinCookieName] : null;
      $configSkin = isset($data['mySkins']) ? $data['mySkins'] : 'default';
      $skinName = $skinFromCookie ?: $configSkin;

      // Get semi-dark setting from cookie or fall back to config
      $semiDarkFromCookie = isset($_COOKIE[$semiDarkCookieName]) ? $_COOKIE[$semiDarkCookieName] : null;
      // Ensure we have a proper boolean conversion
      $semiDarkEnabled = $semiDarkFromCookie !== null ?
        filter_var($semiDarkFromCookie, FILTER_VALIDATE_BOOLEAN) :
        (bool) $data['hasSemiDark'];
    } else {
      // For front-end layouts, use defaults
      $skinName = 'default';
      $semiDarkEnabled = false;
    }

    // Get menu Collapsed state from cookie or fall back to config
    $menuCollapsedFromCookie = isset($_COOKIE['LayoutCollapsed']) ? $_COOKIE['LayoutCollapsed'] : $data['menuCollapsed'];

    // Get content layout from cookie or fall back to config
    $contentLayoutFromCookie = isset($_COOKIE['contentLayout']) ? $_COOKIE['contentLayout'] : $data['contentLayout'];

    // Get header type from cookie or fall back to config
    $navbarTypeFromCookie = isset($_COOKIE['navbarType']) ? $_COOKIE['navbarType'] : $data['navbarType'];

    // Get Header type from cookie or fall back to config
    $headerTypeFromCookie = isset($_COOKIE['headerType']) ? $_COOKIE['headerType'] : $data['headerType'];

    $directionVal = isset($_COOKIE['direction']) ? ($_COOKIE['direction'] === 'true' ? 'rtl' : 'ltr') : $data['myRTLMode'];

    //layout classes
    $layoutClasses = [
      'layout' => $data['myLayout'],
      'skins' => $data['mySkins'],
      'skinName' => $skinName,
      'semiDark' => $semiDarkEnabled,
      'color' => $primaryColor,
      'theme' => $themeVal,
      'themeOpt' => $data['myTheme'],
      'themeOptVal' => $themeUpdatedVal,
      'rtlMode' => $data['myRTLMode'],
      'textDirection' => $directionVal,
      'menuCollapsed' => $menuCollapsedFromCookie,
      'hasCustomizer' => $data['hasCustomizer'],
      'showDropdownOnHover' => $data['showDropdownOnHover'],
      'displayCustomizer' => $data['displayCustomizer'],
      'contentLayout' => $contentLayoutFromCookie,
      'headerType' => $headerTypeFromCookie,
      'navbarType' => $navbarTypeFromCookie,
      'menuFixed' => $data['menuFixed'],
      'footerFixed' => $data['footerFixed'],
      'customizerControls' => $data['customizerControls'],
      'menuAttributes' => self::getMenuAttributes($semiDarkEnabled),
    ];

    // sidebar Collapsed
    if ($layoutClasses['menuCollapsed'] === 'true' || $layoutClasses['menuCollapsed'] === true) {
      $layoutClasses['menuCollapsed'] = 'layout-menu-collapsed';
    } else {
      $layoutClasses['menuCollapsed'] = '';
    }

    // Header Type
    if ($layoutClasses['headerType'] == 'fixed') {
      $layoutClasses['headerType'] = 'layout-menu-fixed';
    }
    // Navbar Type
    if ($layoutClasses['navbarType'] == 'sticky') {
      $layoutClasses['navbarType'] = 'layout-navbar-fixed';
    } elseif ($layoutClasses['navbarType'] == 'static') {
      $layoutClasses['navbarType'] = '';
    } else {
      $layoutClasses['navbarType'] = 'layout-navbar-hidden';
    }

    // Menu Fixed
    if ($layoutClasses['menuFixed'] == true) {
      $layoutClasses['menuFixed'] = 'layout-menu-fixed';
    }


    // Footer Fixed
    if ($layoutClasses['footerFixed'] == true) {
      $layoutClasses['footerFixed'] = 'layout-footer-fixed';
    }

    // RTL Layout/Mode
    if ($layoutClasses['rtlMode'] == true) {
      $layoutClasses['rtlMode'] = 'rtl';
      $layoutClasses['textDirection'] = isset($_COOKIE['direction']) ? ($_COOKIE['direction'] === 'true' ? 'rtl' : 'ltr') : 'rtl';
    } else {
      $layoutClasses['rtlMode'] = 'ltr';
      $layoutClasses['textDirection'] = isset($_COOKIE['direction']) && $_COOKIE['direction'] === 'true' ? 'rtl' : 'ltr';
    }

    // Show DropdownOnHover for Horizontal Menu
    if ($layoutClasses['showDropdownOnHover'] == true) {
      $layoutClasses['showDropdownOnHover'] = true;
    } else {
      $layoutClasses['showDropdownOnHover'] = false;
    }

    // To hide/show display customizer UI, not js
    if ($layoutClasses['displayCustomizer'] == true) {
      $layoutClasses['displayCustomizer'] = true;
    } else {
      $layoutClasses['displayCustomizer'] = false;
    }

    return $layoutClasses;
  }

  public static function updatePageConfig($pageConfigs)
  {
    $demo = 'custom';
    if (isset($pageConfigs)) {
      if (count($pageConfigs) > 0) {
        foreach ($pageConfigs as $config => $val) {
          Config::set('custom.' . $demo . '.' . $config, $val);
        }
      }
    }
  }

  /**
   * Generate CSS for primary color
   *
   * @param string $color Hex color code for primary color
   * @return string CSS for primary color
   */
  public static function generatePrimaryColorCSS($color)
  {
    if (!$color)
      return '';

    // Check if the color actually came from a cookie or explicit configuration
    // Don't generate CSS if there's no specific need for a custom color
    $configColor = config('custom.custom.primaryColor', null);
    $isFromCookie = isset($_COOKIE['admin-primaryColor']) || isset($_COOKIE['front-primaryColor']);

    if (!$configColor && !$isFromCookie)
      return '';

    $r = hexdec(substr($color, 1, 2));
    $g = hexdec(substr($color, 3, 2));
    $b = hexdec(substr($color, 5, 2));

    // Calculate contrast color based on YIQ formula
    $yiq = (($r * 299) + ($g * 587) + ($b * 114)) / 1000;
    $contrastColor = ($yiq >= 150) ? '#000' : '#fff';

    return <<<CSS
:root, [data-bs-theme=light], [data-bs-theme=dark] {
  --bs-primary: {$color};
  --bs-primary-rgb: {$r}, {$g}, {$b};
  --bs-primary-bg-subtle: rgba({$r}, {$g}, {$b}, 0.1);
  --bs-primary-border-subtle: rgba({$r}, {$g}, {$b}, 0.3);
  --bs-primary-contrast: {$contrastColor};
}
CSS;
  }


  public static function typ($mime)
  {
    $mime = strtolower($mime);
    switch ($mime) {
      case 'png':
        $res = 'Image';
        break;
      case 'jpg':
        $res = 'Image';
        break;
      case 'jpeg':
        $res = 'Image';
        break;
      case 'webp':
        $res = 'Image';
        break;
      case 'mp4':
        $res = 'Video';
        break;
      case 'mov':
        $res = 'Video';
        break;
      case 'mpeg':
        $res = 'Video';
        break;
      case 'ogg':
        $res = 'Video';
        break;
      case 'pdf':
        $res = 'Pdf';
        break;
      default:
        $res = 'Image';
    }
    return $res;
  }

  public static function fcm_notifications($title, $message, $token, $data = [], $image = null)
  {
    $fcmUrl = 'https://fcm.googleapis.com/v1/projects/topsy-4d463/messages:send';
    $payload = [
      'message' => [
        'notification' => [
          'title' => $title,
          'body' => str_replace('&nbsp;', ' ', $message),
          'image' => $image,
        ],
        'data' => [
          'type' => $data['type'] ?? '',
          'type_id' => (string) $data['type_id'] ?? '',
          'title' => $title,
          'body' => str_replace('&nbsp;', ' ', $message),
          'image' => $image,
        ],
      ],
    ];

    // Determine if the tokens are an array or a single token
    if (is_array($token)) {
      if (count($token) > 1) {
        $payload['message']['tokens'] = $token; // Use registration_ids for multiple tokens
      } else {
        $payload['message']['token'] = $token[0]; // Single token
      }
    } else {
      $payload['message']['token'] = $token; // Single token
    }

    try {
      $client = new Client();
      $response = $client->post($fcmUrl, [
        'headers' => [
          'Authorization' => 'Bearer ' . self::getAccessToken(),
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
      ]);

      $responseBody = json_decode($response->getBody()->getContents(), true);
      //Log::error('FCM Notification sent', ['response' => $responseBody]);

      return $responseBody;
    } catch (\Exception $e) {
      Log::error('FCM Notification error: ' . $e->getMessage(), [
        'exception' => $e,
        'response' => $responseBody ?? null,
      ]);
    }
  }

  private static function getAccessToken()
  {
    // Use Google service account credentials to generate access token
    $client = new \Google_Client();
    $client->setAuthConfig(storage_path('app/topsy-4d463-firebase-adminsdk-fbsvc-1cb9bf9265.json'));
    $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
    $client->useApplicationDefaultCredentials();
    Log::error('FCM Access Token', [
      'token' => $client->fetchAccessTokenWithAssertion()['access_token'],
    ]);
    return $client->fetchAccessTokenWithAssertion()['access_token'];
  }

  public static function notifications($head, $type, $message, $user_id, $table_id, $url, $icon = 'tabler-mail')
  {
    $user = User::find($user_id);
    $details = [
      'body' => $message,
      'head' => $head,
      'request_id' => $table_id,
      'user_id' => $user->id,
      'type' => $type,
      'notifiable_type' => 'App\Models\User',
      'class' => 'icon-base ti ' . $icon . ' icon-lg warning',
      'color' => 'primary media-heading',
      'url' => $url,
      'sender_id' => auth('sanctum')->id(),
    ];
    Notification::send($user, new UserNotification($details));
    event(new NotificationEvent($details));
  }
  public static function adminNotification($head, $type, $message, $table_id, $url, $sender_id = null)
  {
    $admins = User::whereHas('roles', function ($query) {
      $query->where('name', 'Admin');
    })->get();
    $details = [
      'body' => $message,
      'head' => $head,
      'request_id' => $table_id,
      'user_id' => null, // will set for each admin inside loop
      'type' => $type,
      'notifiable_type' => 'App\Models\User',
      'class' => 'icon-base ti tabler-briefcase icon-lg warning',
      'color' => 'primary media-heading',
      'url' => $url,
      'sender_id' => $sender_id ?? auth('sanctum')->id(),
    ];
    // Send notification to each admin
    foreach ($admins as $admin) {
      $details['user_id'] = $admin->id;
      Notification::send($admin, new UserNotification($details));
      event(new NotificationEvent($details));
    }
  }

  public static function setting($key, $type)
  {
    $cr = Setting::where('meta_key', $key)->where('meta_type', $type)->first()?->meta_value;
    return $cr;
  }
  public static function unreadCounts()
  {
    $data = array();
    $id = Auth::id();
    $user = User::find($id);
    if (!empty($user)) {
      $data['unread_msg'] = Chat::messages()->setParticipant($user)->unreadCount();
      $data['notifications'] = $user->unreadNotifications;
      $data['prefix'] = $user->user_type == 'Admin' ? 'admin/' : 'store/';
    }
    return $data;
  }

  public static function random_strings($length_of_string)
  {
    $str_result = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'; //'0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
    return substr(str_shuffle($str_result), 0, $length_of_string);
  }

  public static function addDynamicMenuBadges(array|object $menu): array
  {
    $menu = json_decode(json_encode($menu), true);
    foreach ($menu as &$item) {
      // Show pending user count badge
      if (isset($item['url']) && $item['url'] === 'users') {
        $pendingUsers = User::whereNull('email_verified_at')->where('user_type', 'User')->count();
        if ($pendingUsers > 0) {
          $item['badge'] = ['danger', $pendingUsers];
        }
      }
      if (isset($item['url']) && $item['url'] === 'drivers') {
        $pendingUsers = User::where('is_verified', 0)->where('user_type', 'Driver')->count();
        if ($pendingUsers > 0) {
          $item['badge'] = ['danger', $pendingUsers];
        }
      }
      if (isset($item['url']) && $item['url'] === 'drivers') {
        $pendingUsers = User::where('is_verified', 0)->where('user_type', 'Store')->count();
        if ($pendingUsers > 0) {
          $item['badge'] = ['danger', $pendingUsers];
        }
      }

      $user = Auth::user();
      if (isset($item['url']) && $item['url'] === 'chat') {
        $notifications = MessageNotification::where('is_seen', 0)
          ->where('is_sender', 0)
          ->where('messageable_id', $user->id)
          ->count();
        if ($notifications > 0) {
          $item['badge'] = ['success', $notifications];
        }
      }
      if ($user->user_type == 'Admin' && isset($item['url']) && $item['url'] === 'withdrawal_requests') {
        $withdrawalRequest = WithdrawalRequest::where('status', 'Pending')
          ->count();
        if ($withdrawalRequest > 0) {
          $item['badge'] = ['primary', $withdrawalRequest];
        }
      }
      if ($user->user_type == 'Admin' && isset($item['url']) && $item['url'] === 'tickets') {
        $suportTicket = Contact::where('status', 'Open')
          ->count();
        if ($suportTicket > 0) {
          $item['badge'] = ['info', $suportTicket];
        }
      }
      if ($user->user_type == 'Store' && isset($item['url']) && $item['url'] === 'orders') {
        $storeId = User::find($user->id)?->storeId;
        $newOrders = Order::where('store_id', $storeId)
          ->where('status', 'Placed')
          ->count();
        if ($newOrders > 0) {
          $item['badge'] = ['primary', $newOrders];
        }
      }

      // Recursively apply to submenu
      if (isset($item['submenu']) && is_array($item['submenu'])) {
        $item['submenu'] = self::addDynamicMenuBadges($item['submenu']);
      }
    }
    return $menu;
  }
  public static function markNotificationsAsRead($heads): int
  {
    //markNotificationsAsRead(['Support Tickets', 'Orders']);
    //markNotificationsAsRead('Support Tickets');
    $user = Auth::guard('web')->user();
    if (!$user) {
      return 0;
    }
    // Convert to array if a single string is passed
    $heads = is_array($heads) ? $heads : [$heads];
    $count = 0;
    foreach ($user->unreadNotifications as $notification) {
      $head = $notification['data']['head'] ?? null;
      if ($head && in_array($head, $heads, true)) {
        $notification->markAsRead();
        $count++;
      }
    }
    return $count;
  }

  public static function getStatusMessage($status)
  {
    return match (strtolower($status)) {
      'placed' => 'Your order has been placed successfully',
      'accepted' => 'Order has been accepted by the store',
      'preparing' => 'Order is being prepared',
      'ready' => 'Order is ready for dispatch',
      'dispatched' => 'Order has been dispatched from the store',
      'out for delivery' => 'Courier is out for delivery',
      'delivered' => 'Order has been delivered',
      'cancelled' => 'Order was cancelled',
      default => ucfirst($status),
    };
  }

  public static function formatNumberShort($number): string
  {
    if ($number >= 1000000) {
      return round($number / 1000000, 1) . 'M';
    }
    if ($number >= 1000) {
      return round($number / 1000, 1) . 'k';
    }
    return (string) $number;
  }
  public static function estimateDeliveryTime(?float $distance, int $storeDeliveryTime): string
  {
    if (!$distance || !$storeDeliveryTime) {
      return '';
    }
    // Average speed in km/min (optional, use distance-based calculation)
    $avgSpeed = 0.5; // 30 km/h => 0.5 km/min

    // Calculate travel time based on distance
    $travelTime = $distance / $avgSpeed;

    // Use the greater of store's delivery time or distance-based time as base
    $baseTime = max($storeDeliveryTime, $travelTime);

    // Create min/max with ±20% variance
    $minTime = ceil(($baseTime * 1) / 5) * 5;
    $maxTime = ceil(($baseTime * 1.2) / 5) * 5;

    return "{$minTime}-{$maxTime} Mins";
  }

  public static function haversine($lat1, $lon1, $lat2, $lon2)
  {
    $earthRadius = 6371; // km
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    $a = sin($dLat / 2) * sin($dLat / 2) +
      cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
      sin($dLon / 2) * sin($dLon / 2);
    $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
    return $earthRadius * $c; // km
  }
  public static function formatSecondsToMinutes($seconds)
  {
    $minutes = floor($seconds / 60);
    $remainingSeconds = $seconds % 60;

    $parts = [];
    if ($minutes > 0) {
      $parts[] = $minutes . ' minute' . ($minutes > 1 ? 's' : '');
    }
    if ($remainingSeconds > 0 || $minutes == 0) {
      $parts[] = $remainingSeconds . ' second' . ($remainingSeconds > 1 ? 's' : '');
    }

    return implode(' ', $parts);
  }

  public static function getStatusLabel(string $status, string $type): string
  {
    $statuses = [
      'Ride' => [
        'Pending' => 'Pending',
        'Accepted' => 'Accepted',
        'Arrived' => 'Arrived',
        'Ongoing' => 'On Trip',
        'Invoiced' => 'Invoiced',
        'Completed' => 'Completed',
        'Cancelled' => 'Cancelled',
      ],
      'Parcel' => [
        'Pending' => 'Pending',
        'Accepted' => 'Accepted',
        'Arrived' => 'Arrived',
        'PickedUp' => 'Picked Up',
        'Ongoing' => 'In Transit',
        'Invoiced' => 'Invoiced',
        'Completed' => 'Delivered',
        'Cancelled' => 'Cancelled',
        'Failed' => 'Delivery Failed',
      ],
      'Delivery' => [ // Food / Order delivery
        'Pending' => 'Pending',
        'Accepted' => 'Accepted',
        'PickedUp' => 'Order Picked Up',
        'Ongoing' => 'Out for Delivery',
        'Invoiced' => 'Invoiced',
        'Completed' => 'Delivered',
        'Cancelled' => 'Cancelled',
        'Failed' => 'Delivery Failed',
      ],
    ];

    // Fallback: if type or status not found
    return $statuses[$type][$status] ?? $status;
  }


}
