<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Rental;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use App\Helpers\Helpers;

class SendInstallmentReminders extends Command
{
  /**
   * The name and signature of the console command.
   *
   * @var string
   */
  protected $signature = 'app:send-installment-reminders';

  /**
   * The console command description.
   *
   * @var string
   */
  protected $description = 'Send installment due and overdue reminders daily at 6am';

  /**
   * Execute the console command.
   */
  public function handle()
  {
    $today = Carbon::today();
    $twoDaysBefore = $today->copy()->addDays(2);

    // 1. Upcoming (due in 2 days)
    Rental::with([
      'car:id,vehicle_model',
      'driver.user:id,fcm_token'
    ])
      ->select('id', 'car_id', 'driver_id', 'due_date')
      ->whereDate('due_date', $twoDaysBefore)
      ->where('status', 'Active')
      ->chunk(100, function ($rentals) {
        foreach ($rentals as $inst) {
          $message = "Reminder: Installment #{$inst->id} for Rental #{$inst->id} ({$inst->car?->vehicle_model}) is due in 2 days.";
          $token = $inst->driver?->user?->fcm_token;
          if ($token) {
            Helpers::fcm_notifications('Installment Due', $message, $token, ['type' => 'rentals', 'type_id' => $inst->id]);
          }
        }
      });

    // 2. Overdue (past due date, still pending)
    Rental::with([
      'car:id,vehicle_model',
      'driver.user:id,fcm_token'
    ])
      ->select('id', 'car_id', 'driver_id', 'due_date')
      ->where('due_date', '<', $today)
      ->where('status', 'Active')
      ->chunk(100, function ($rentals) {
        foreach ($rentals as $inst) {
          $message = "Overdue: Installment #{$inst->id} for Rental #{$inst->id} ({$inst->car?->vehicle_model}) is overdue.";
          $token = $inst->driver?->user?->fcm_token;
          if ($token) {
            Helpers::fcm_notifications('Installment OverDue', $message, $token, ['type' => 'rentals', 'type_id' => $inst->id]);
          }
        }
      });

    $this->info("Installment reminders processed.");
  }
}
