<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\App;

class DealModel extends Model
{
    use HasFactory;
    protected $table = 'deals';
    protected $fillable = [
        'deal_title',
        'deal_type',
        'deal_amount',
        'deal_status',
        'completed_by_type',
        'completed_by_id',
        'from_customer_id',
        'to_customer_id',
        'created_at',
        'status_datetime',
        'updated_at',
        'is_release',
        'is_hold',
        'cancel_by',
        'cancel_reason',
        'cancel_type',
        'transfer_fees_in_percent',
        'admin_earnings',
        'release_amount',
        'dispute_request_by_type',
        'dispute_request_by_id',
        'dispute_request_date',
        'dispute_reason',
        'amount_released_to',
        'deal_number',
        'is_paid',
        'dispute_reason_id',
        'is_read',
        'total_deal_Amount',
        'payment_proof',
        'end_reason',
        'shipping_proof',
        'payment_status',
        'shipping_status',
        'currency_code'
    ];

    protected $appends = ['buyer_name', 'seller_name'];

    public function getBuyerNameAttribute()
    {
        return $this->deal_type === 'buy' ?
            optional($this->fromCustomer)->full_name
            : optional($this->toCustomer)->full_name;
    }
    public function getSellerNameAttribute()
    {
        return $this->deal_type === 'sell' ?
            optional($this->fromCustomer)->full_name
            : optional($this->toCustomer)->full_name;
    }
    public function completedByAdmin()
    {
        return $this->belongsTo(AdminModel::class, 'completed_by_id');
    }

    public function cancelByAdmin()
    {
        return $this->belongsTo(AdminModel::class, 'cancel_by');
    }

    public function disputeByAdmin()
    {
        return $this->belongsTo(AdminModel::class, 'dispute_request_by_id');
    }



    public function fromCustomer()
    {
        return $this->belongsTo(CustomerModel::class, 'from_customer_id', 'id')->withTrashed();
    }

    public function toCustomer()
    {
        return $this->belongsTo(CustomerModel::class, 'to_customer_id', 'id')->withTrashed();
    }

    public function amountReleasedTo()
    {
        return $this->belongsTo(CustomerModel::class, 'amount_released_to');
    }

    public function canceledDeal()
    {
        return $this->hasOne(CanceledDealModel::class, 'deal_id');
    }
    public function disputeDocs()
    {
        return $this->hasMany(disputDdealDocumentModel::class, 'deal_id');
    }
    public function images()
    {
        return $this->hasMany(DealImages::class, 'deal_id');
    }
    // All rejections (history)
    public function paymentRejections()
    {
        return $this->hasMany(DealDocumentRejection::class, 'deal_id')
            ->where('document_type', 'payment');
    }

    public function shippingRejections()
    {
        return $this->hasMany(DealDocumentRejection::class, 'deal_id')
            ->where('document_type', 'shipping');
    }

    // Only latest rejection
    public function latestPaymentRejection()
    {
        return $this->hasOne(DealDocumentRejection::class, 'deal_id')
            ->where('document_type', 'payment')
            ->orderByDesc('id');
    }

    public function latestShippingRejection()
    {
        return $this->hasOne(DealDocumentRejection::class, 'deal_id')
            ->where('document_type', 'shipping')
            ->orderByDesc('id');
    }
    public function dealUpdateValidation($inputs = [], $id = null)
    {
        $validStatus = array_keys(dealStatusFilter(false));
        $rule = ['deal_status' => 'required|in:' . implode(',', $validStatus)];
        $validator = Validator::make($inputs, $rule);
        return $validator;
    }
    public static function ongoingDealsPagination(Request $request)
    {
        $ongoingDeals = [PENDING, ACCEPTED, TRANSIT, DELIVERED, COMPLETED];
        $filter = 1;
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'deals.id';


        $fields = "
        deals.*," . buyerSellerFields() . "
        ";


        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                deals.deal_title like '%" . addslashes($request->get('keyword')) . "%'
                or deals.deal_status like '%" . addslashes($request->get('keyword')) . "%'
                or f.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or t.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or f.email like '%" . addslashes($request->get('keyword')) . "%'
                or t.email like '%" . addslashes($request->get('keyword')) . "%'
                or f.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or t.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or f.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or t.dial_code like '%" . addslashes($request->get('keyword')) . "%'
               
                )";
        }

        if ($request->has('payment_status') && $request->get('payment_status') != '') {
            switch ($request->get('payment_status')) {
                case PAYMENT_RECEIVED:
                    $filter .= " and deals.is_paid = 1 and deals.is_release = 0";
                    break;

                case PAYMENT_PENDING:
                    $filter .= " and deals.is_paid = 0";
                    break;
            }
        }

        if ($request->has('deal_status') && $request->get('deal_status') != '') {
            $filter .= " and deals.deal_status = '" . addslashes($request->get('deal_status')) . "'";
        }

        if ($request->has('deal_type') && $request->get('deal_type') != '') {
            $filter .= " and deals.deal_type = '" . addslashes($request->get('deal_type')) . "'";
        }
        if ($request->has('currency_type') && $request->get('currency_type') != '') {
            $filter .= " and UPPER(deals.currency_code) = '" . strtoupper(addslashes($request->get('currency_type'))) . "'";
        }

        if ($request->has('date_range') && !empty($request->get('date_range'))) {
            $dateRange = $request->get('date_range');
            $dates = explode(' to ', $dateRange);

            if (count($dates) === 2) {
                $startDate = $dates[0];
                $endDate = $dates[1];
                $startDate = date('Y-m-d', strtotime($startDate));
                $endDate = date('Y-m-d', strtotime($endDate));
                // Apply the filter for the date range
                $filter .= " and DATE(deals.created_at) BETWEEN '" . addslashes($startDate) . "' AND '" . addslashes($endDate) . "'";
            } elseif (count($dates) === 1) {
                $startDate = $dates[0];
                $startDate = date('Y-m-d', strtotime($startDate));
                $filter .= " and DATE(deals.created_at) ='" . addslashes($startDate) . "'";
            }
        }

        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }

        return self::select(DB::raw($fields))
            ->leftJoin('customers as f', 'f.id', '=', 'deals.from_customer_id')
            ->leftJoin('customers as t', 't.id', '=', 'deals.to_customer_id')
            ->whereIn('deals.deal_status', $ongoingDeals)
            ->whereRaw($filter)
            ->orderBy($sortEntity, $sortOrder);
    }
    public static function disputedDealsPagination(Request $request)
    {
        $disputed = [ON_DISPUTE, DISPUTE_ACCEPTED, DISPUTE_REJECTED, DISPUTE_UNDER_REVIEW];
        $filter = 1;
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'deals.id';


        $fields = "
        deals.*," . buyerSellerFields() . "
        ";



        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                deals.deal_title like '%" . addslashes($request->get('keyword')) . "%'
                or deals.deal_status like '%" . addslashes($request->get('keyword')) . "%'
                or f.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or t.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or f.email like '%" . addslashes($request->get('keyword')) . "%'
                or t.email like '%" . addslashes($request->get('keyword')) . "%'
                or f.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or t.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or f.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or t.dial_code like '%" . addslashes($request->get('keyword')) . "%'
               
                )";
        }

        if ($request->has('deal_status') && $request->get('deal_status') != '') {
            $filter .= " and deals.deal_status = '" . addslashes($request->get('deal_status')) . "'";
        }

        if ($request->has('deal_type') && $request->get('deal_type') != '') {
            $filter .= " and deals.deal_type = '" . addslashes($request->get('deal_type')) . "'";
        }

        if ($request->has('payment_status') && $request->get('payment_status') != '') {
            switch ($request->get('payment_status')) {
                case PAYMENT_RECEIVED:
                    $filter .= " and deals.is_paid = 1 and deals.is_release = 0";
                    break;

                case MONEY_RETURNED_BUYER:
                    $filter .= " and deals.is_release=1 and  ((deals.deal_type = 'buy' and deals.amount_released_to = deals.from_customer_id)
                    or (deals.deal_type = 'sell' and deals.amount_released_to = deals.to_customer_id))";
                    break;

                case MONEY_DEPOSITED_SELLER:
                    $filter .= " and deals.is_release=1 and  ((deals.deal_type = 'buy' and deals.amount_released_to = deals.to_customer_id)
                                or(deals.deal_type = 'sell' and deals.amount_released_to = deals.from_customer_id))";
                    break;
            }
        }

        if ($request->has('date_range') && !empty($request->get('date_range'))) {
            $dateRange = $request->get('date_range');
            $dates = explode(' to ', $dateRange);

            if (count($dates) === 2) {
                $startDate = $dates[0];
                $endDate = $dates[1];
                $startDate = date('Y-m-d', strtotime($startDate));
                $endDate = date('Y-m-d', strtotime($endDate));
                // Apply the filter for the date range
                $filter .= " and DATE(deals.created_at) BETWEEN '" . addslashes($startDate) . "' AND '" . addslashes($endDate) . "'";
            } elseif (count($dates) === 1) {
                $startDate = $dates[0];
                $startDate = date('Y-m-d', strtotime($startDate));
                $filter .= " and DATE(deals.created_at) ='" . addslashes($startDate) . "'";
            }
        }


        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }
        if ($request->has('currency_type') && $request->get('currency_type') != '') {
            $filter .= " and UPPER(deals.currency_code) = '" . strtoupper(addslashes($request->get('currency_type'))) . "'";
        }
        return self::select(DB::raw($fields))
            ->leftJoin('customers as f', 'f.id', '=', 'deals.from_customer_id')
            ->leftJoin('customers as t', 't.id', '=', 'deals.to_customer_id')
            ->whereIn('deals.deal_status', $disputed)
            ->whereRaw($filter)
            ->orderBy($sortEntity, $sortOrder);
    }

    public static function cancelledDealsPagination(Request $request)
    {
        $disputed = [CANCEL];
        $filter = 1;
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'deals.id';


        $fields = "
        deals.*," . buyerSellerFields() . "
        ";


        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                deals.deal_title like '%" . addslashes($request->get('keyword')) . "%'
                or deals.deal_status like '%" . addslashes($request->get('keyword')) . "%'
                or f.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or t.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or f.email like '%" . addslashes($request->get('keyword')) . "%'
                or t.email like '%" . addslashes($request->get('keyword')) . "%'
                or f.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or t.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or f.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or t.dial_code like '%" . addslashes($request->get('keyword')) . "%'
               
                )";
        }



        if ($request->has('deal_type') && $request->get('deal_type') != '') {
            $filter .= " and deals.deal_type = '" . addslashes($request->get('deal_type')) . "'";
        }
        if ($request->has('currency_type') && $request->get('currency_type') != '') {
            $filter .= " and UPPER(deals.currency_code) = '" . strtoupper(addslashes($request->get('currency_type'))) . "'";
        }
        if ($request->has('payment_status') && $request->get('payment_status') != '') {
            switch ($request->get('payment_status')) {
                case PAYMENT_RECEIVED:
                    $filter .= " and deals.is_paid = 1 and deals.is_release = 0";
                    break;

                case PAYMENT_PENDING:
                    $filter .= " and deals.is_paid = 0";
                    break;

                case MONEY_RETURNED_BUYER:
                    $filter .= " and deals.is_release=1 and  ((deals.deal_type = 'buy' and deals.amount_released_to = deals.from_customer_id)
                    or (deals.deal_type = 'sell' and deals.amount_released_to = deals.to_customer_id))";
                    break;

                case MONEY_DEPOSITED_SELLER:
                    $filter .= " and deals.is_release=1 and  ((deals.deal_type = 'buy' and deals.amount_released_to = deals.to_customer_id)
                                or(deals.deal_type = 'sell' and deals.amount_released_to = deals.from_customer_id))";
                    break;
            }
        }

        if ($request->has('date_range') && !empty($request->get('date_range'))) {
            $dateRange = $request->get('date_range');
            $dates = explode(' to ', $dateRange);

            if (count($dates) === 2) {
                $startDate = $dates[0];
                $endDate = $dates[1];
                $startDate = date('Y-m-d', strtotime($startDate));
                $endDate = date('Y-m-d', strtotime($endDate));
                // Apply the filter for the date range
                $filter .= " and DATE(deals.created_at) BETWEEN '" . addslashes($startDate) . "' AND '" . addslashes($endDate) . "'";
            } elseif (count($dates) === 1) {
                $startDate = $dates[0];
                $startDate = date('Y-m-d', strtotime($startDate));
                $filter .= " and DATE(deals.created_at) ='" . addslashes($startDate) . "'";
            }
        }


        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }

        return self::select(DB::raw($fields))
            ->leftJoin('customers as f', 'f.id', '=', 'deals.from_customer_id')
            ->leftJoin('customers as t', 't.id', '=', 'deals.to_customer_id')
            ->whereIn('deals.deal_status', $disputed)
            ->whereRaw($filter)
            ->orderBy($sortEntity, $sortOrder);
    }

    public static function holdAmountDealsPagination(Request $request)
    {
        $filter = 1;
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'deals.id';


        $fields = "
        deals.*," . buyerSellerFields() . "
        ";



        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                deals.deal_title like '%" . addslashes($request->get('keyword')) . "%'
                or deals.deal_status like '%" . addslashes($request->get('keyword')) . "%'
                or f.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or t.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or f.email like '%" . addslashes($request->get('keyword')) . "%'
                or t.email like '%" . addslashes($request->get('keyword')) . "%'
                or f.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or t.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or f.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or t.dial_code like '%" . addslashes($request->get('keyword')) . "%'
               
                )";
        }



        if ($request->has('deal_type') && $request->get('deal_type') != '') {
            $filter .= " and deals.deal_type = '" . addslashes($request->get('deal_type')) . "'";
        }
        if ($request->has('currency_type') && $request->get('currency_type') != '') {
            $filter .= " and UPPER(deals.currency_code) = '" . strtoupper(addslashes($request->get('currency_type'))) . "'";
        }
        if ($request->has('date_range') && !empty($request->get('date_range'))) {
            $dateRange = $request->get('date_range');
            $dates = explode(' to ', $dateRange);

            if (count($dates) === 2) {
                $startDate = $dates[0];
                $endDate = $dates[1];
                $startDate = date('Y-m-d', strtotime($startDate));
                $endDate = date('Y-m-d', strtotime($endDate));
                // Apply the filter for the date range
                $filter .= " and DATE(deals.created_at) BETWEEN '" . addslashes($startDate) . "' AND '" . addslashes($endDate) . "'";
            } elseif (count($dates) === 1) {
                $startDate = $dates[0];
                $startDate = date('Y-m-d', strtotime($startDate));
                $filter .= " and DATE(deals.created_at) ='" . addslashes($startDate) . "'";
            }
        }


        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }


        return self::select(DB::raw($fields))
            ->leftJoin('customers as f', 'f.id', '=', 'deals.from_customer_id')
            ->leftJoin('customers as t', 't.id', '=', 'deals.to_customer_id')
            ->where('deals.is_hold', 1)
            ->whereRaw($filter)
            ->orderBy($sortEntity, $sortOrder);
    }
    public static function declinedDealsPagination(Request $request)
    {
        $declined = [DECLINED];
        $filter = 1;
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'deals.id';


        $fields = "
        deals.*," . buyerSellerFields() . "
        ";



        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                deals.deal_title like '%" . addslashes($request->get('keyword')) . "%'
                or deals.deal_status like '%" . addslashes($request->get('keyword')) . "%'
                or f.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or t.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or f.email like '%" . addslashes($request->get('keyword')) . "%'
                or t.email like '%" . addslashes($request->get('keyword')) . "%'
                or f.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or t.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or f.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or t.dial_code like '%" . addslashes($request->get('keyword')) . "%'
               
                )";
        }



        if ($request->has('deal_type') && $request->get('deal_type') != '') {
            $filter .= " and deals.deal_type = '" . addslashes($request->get('deal_type')) . "'";
        }
        if ($request->has('currency_type') && $request->get('currency_type') != '') {
            $filter .= " and UPPER(deals.currency_code) = '" . strtoupper(addslashes($request->get('currency_type'))) . "'";
        }
        if ($request->has('date_range') && !empty($request->get('date_range'))) {
            $dateRange = $request->get('date_range');
            $dates = explode(' to ', $dateRange);

            if (count($dates) === 2) {
                $startDate = $dates[0];
                $endDate = $dates[1];
                $startDate = date('Y-m-d', strtotime($startDate));
                $endDate = date('Y-m-d', strtotime($endDate));
                // Apply the filter for the date range
                $filter .= " and DATE(deals.created_at) BETWEEN '" . addslashes($startDate) . "' AND '" . addslashes($endDate) . "'";
            } elseif (count($dates) === 1) {
                $startDate = $dates[0];
                $startDate = date('Y-m-d', strtotime($startDate));
                $filter .= " and DATE(deals.created_at) ='" . addslashes($startDate) . "'";
            }
        }


        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }


        return self::select(DB::raw($fields))
            ->leftJoin('customers as f', 'f.id', '=', 'deals.from_customer_id')
            ->leftJoin('customers as t', 't.id', '=', 'deals.to_customer_id')
            ->whereIn('deals.deal_status', $declined)
            ->whereRaw($filter)
            ->orderBy($sortEntity, $sortOrder);
    }
    public static function closedDealsPagination(Request $request)
    {
        $closed = [CLOSED];
        $filter = '1 ';
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'deals.updated_at';


        $fields = "
        deals.*," . buyerSellerFields() . "
        ";



        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                deals.deal_title like '%" . addslashes($request->get('keyword')) . "%'
                or deals.deal_status like '%" . addslashes($request->get('keyword')) . "%'
                or f.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or t.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or f.email like '%" . addslashes($request->get('keyword')) . "%'
                or t.email like '%" . addslashes($request->get('keyword')) . "%'
                or f.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or t.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or f.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or t.dial_code like '%" . addslashes($request->get('keyword')) . "%'
               
                )";
        }



        if ($request->has('deal_type') && $request->get('deal_type') != '') {
            $filter .= " and deals.deal_type = '" . addslashes($request->get('deal_type')) . "'";
        }
        if ($request->has('currency_type') && $request->get('currency_type') != '') {
            $filter .= " and UPPER(deals.currency_code) = '" . strtoupper(addslashes($request->get('currency_type'))) . "'";
        }
        if ($request->has('date_range') && !empty($request->get('date_range'))) {
            $dateRange = $request->get('date_range');
            $dates = explode(' to ', $dateRange);

            if (count($dates) === 2) {
                $startDate = $dates[0];
                $endDate = $dates[1];
                $startDate = date('Y-m-d', strtotime($startDate));
                $endDate = date('Y-m-d', strtotime($endDate));
                // Apply the filter for the date range
                $filter .= " and DATE(deals.created_at) BETWEEN '" . addslashes($startDate) . "' AND '" . addslashes($endDate) . "'";
            } elseif (count($dates) === 1) {
                $startDate = $dates[0];
                $startDate = date('Y-m-d', strtotime($startDate));
                $filter .= " and DATE(deals.created_at) ='" . addslashes($startDate) . "'";
            }
        }
        //get deals as  closure type
        if ($request->has('closure_type') && $request->get('closure_type') != '') {
            $filter .= "and deals.end_reason = '" . addslashes($request->get('closure_type')) . "' and deals.deal_status = '" . CLOSED . "'";
        }

        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }

        return self::select(DB::raw($fields))
            ->leftJoin('customers as f', 'f.id', '=', 'deals.from_customer_id')
            ->leftJoin('customers as t', 't.id', '=', 'deals.to_customer_id')
            ->whereIn('deals.deal_status', $closed)
            ->whereRaw($filter)
            ->orderBy($sortEntity, $sortOrder);
    }
    public function pagination(Request $request)
    {
        $filter = 1;
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'deals.id';


        $fields = "
        deals.*,
        f.full_name as from_customer_name,
        f.email as from_customer_email,
        f.mobile_number as from_customer_mobile_number,
        f.dial_code as from_customer_dial_code,
        t.full_name as to_customer_name,
        t.email as to_customer_email,
        t.mobile_number as to_customer_mobile_number,
        t.dial_code as to_customer_dial_code
    ";

        if ($request->has('per_page') && $request->get('per_page') != '') {
            $perPage = $request->get('per_page');
        }

        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                deals.deal_title like '%" . addslashes($request->get('keyword')) . "%'
                or deals.deal_status like '%" . addslashes($request->get('keyword')) . "%'
                or f.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or t.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or f.email like '%" . addslashes($request->get('keyword')) . "%'
                or t.email like '%" . addslashes($request->get('keyword')) . "%'
                or f.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or t.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or f.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or t.dial_code like '%" . addslashes($request->get('keyword')) . "%'
               
                )";
        }

        if ($request->has('deal_status') && $request->get('deal_status') != '') {
            $filter .= " and deals.deal_status = '" . addslashes($request->get('deal_status')) . "'";
        }

        if ($request->has('deal_type') && $request->get('deal_type') != '') {
            $filter .= " and deals.deal_type = '" . addslashes($request->get('deal_type')) . "'";
        }

        // Check if 'date_range' is provided and handle it
        if ($request->has('date_range') && !empty($request->get('date_range'))) {
            $dateRange = $request->get('date_range');
            $dates = explode(' to ', $dateRange);  // Split the range into two parts: start_date and end_date

            if (count($dates) === 2) {
                $startDate = $dates[0];  // First part is the start date
                $endDate = $dates[1];   // Second part is the end date
                $startDate = date('Y-m-d', strtotime($startDate));
                $endDate = date('Y-m-d', strtotime($endDate));
                // Apply the filter for the date range
                $filter .= " and DATE(deals.created_at) BETWEEN '" . addslashes($startDate) . "' AND '" . addslashes($endDate) . "'";
            }
        }

        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }

        return $this
            ->select(DB::raw($fields))
            ->leftJoin('customers as f', 'f.id', '=', 'deals.from_customer_id')
            ->leftJoin('customers as t', 't.id', '=', 'deals.to_customer_id')
            ->whereRaw($filter)
            ->orderBy($sortEntity, $sortOrder)
            ->paginate($perPage);
    }

    public static function getDealstats($statuses)
    {
        $query = self::whereIn('deal_status', $statuses);
        $count = (clone $query)->count();
        $totals = (clone $query)->selectRaw('currency_code,  COUNT(*) as deal_count,SUM(deal_amount) as total_amount')
            ->groupBy('currency_code')
            ->get()
            ->keyBy('currency_code');
        return [
            'count'        => $count, // total deals regardless of currency
            'mxn_deals'    => $totals['MXN']->deal_count ?? 0,
            'usd_deals'    => $totals['USD']->deal_count ?? 0,
            'mxn_total'    => $totals['MXN']->total_amount ?? 0,
            'usd_total'    => $totals['USD']->total_amount ?? 0,
        ];
    }
    public static function getHoldDeals()
    {
        $query =  self::where('is_hold', 1);
        $count = (clone $query)->count();
        $totals = (clone $query)->selectRaw('currency_code,COUNT(*) as deal_count, SUM(deal_amount) as total_amount')
            ->groupBy('currency_code')
            ->get()
            ->keyBy('currency_code');
        return [
            'count' => $count,
            'mxn_deals'    => $totals['MXN']->deal_count ?? 0,
            'usd_deals'    => $totals['USD']->deal_count ?? 0,
            'mxn_total'    => $totals['MXN']->total_amount ?? 0,
            'usd_total'    => $totals['USD']->total_amount ?? 0,
        ];
    }
    public static function totalDeals()
    {
        return self::count();
    }
    public static function totalBuyDeals()
    {
        return self::where('deal_type', BUY)->count();
    }
    public static function totalSellDeals()
    {
        return self::where('deal_type', SELL)->count();
    }

    //deal create validation define
    public function dealCreateValiation($requests)
    {
        $ext = imageExtensions();
        $mimes = implode(",", $ext);
        $rules = [
            'deal_type' => 'required|in:buy,sell',
            'dial_code' => 'required|string',
            'dial_code_iso' => 'required|string|max:5',
            'mobile_number' => 'required',
            'deal_title' => 'required|string',
            'deal_amount' => 'required|numeric|min:1',
            'currency_code' => 'required|max:3|in:USD,MXN',
            'images' => 'nullable|array|min:1|max:3',
            'images.*' => 'image|mimes:' . $mimes . '|max:5120',

        ];

        return Validator($requests, $rules);
    }

    public static function getAdminEearnings($year)
    {
        Carbon::setLocale(App::getLocale());

        // USD earnings per month
        $usdEarnings = self::selectRaw('SUM(admin_earnings) as total, MONTH(created_at) as month')
            ->whereYear('created_at', $year)
            ->where('is_paid', 1)
            ->where('currency_code', 'USD')
            ->groupBy('month')
            ->orderBy('month')
            ->pluck('total', 'month')
            ->toArray();

        // MXN earnings per month
        $mxnEarnings = self::selectRaw('SUM(admin_earnings) as total, MONTH(created_at) as month')
            ->whereYear('created_at', $year)
            ->where('is_paid', 1)
            ->where('currency_code', 'MXN')
            ->groupBy('month')
            ->orderBy('month')
            ->pluck('total', 'month')
            ->toArray();

        $months = [];
        $usdData = [];
        $mxnData = [];

        for ($i = 1; $i <= 12; $i++) {
            $months[] = Carbon::create()->month($i)->translatedFormat('F');
            $usdData[] = $usdEarnings[$i] ?? 0;
            $mxnData[] = $mxnEarnings[$i] ?? 0;
        }

        return [
            'earningTypes' => $months,
            'usdData' => $usdData,
            'mxnData' => $mxnData
        ];
    }
    //get earnings year wise
    public static function getAdminEarningsYearWise()
    {
        $oldestYear = self::min(DB::raw('YEAR(created_at)')) ?? date('Y');

        $usdEarnings = self::selectRaw('SUM(admin_earnings) as total, YEAR(created_at) as year')
            ->where('currency_code', 'USD')
            ->where('is_paid', 1)
            ->groupBy('year')
            ->orderBy('year')
            ->pluck('total', 'year')
            ->toArray();

        $mxnEarnings = self::selectRaw('SUM(admin_earnings) as total, YEAR(created_at) as year')
            ->where('currency_code', 'MXN')
            ->where('is_paid', 1)
            ->groupBy('year')
            ->orderBy('year')
            ->pluck('total', 'year')
            ->toArray();

        $years = [];
        $usdData = [];
        $mxnData = [];

        for ($i = $oldestYear; $i <= date('Y'); $i++) {
            $years[] = $i;
            $usdData[] = $usdEarnings[$i] ?? 0;
            $mxnData[] = $mxnEarnings[$i] ?? 0;
        }

        return [
            'earningTypes' => $years,
            'usdData' => $usdData,
            'mxnData' => $mxnData
        ];
    }
    public function getNextStepIndicator($loggedInCustomer)
    {
        $indicator = null;
        //get buyer or seller
        $customerType = $this->deal_type === BUY ? ($this->from_customer_id == $loggedInCustomer ? BUYER : SELLER)
            : ($this->from_customer_id == $loggedInCustomer ? SELLER : BUYER);
        if ($this->deal_status == PENDING && $loggedInCustomer == $this->to_customer_id) {
            $indicator = 'accept_now';
        } elseif ($this->deal_status == ACCEPTED && $customerType == BUYER && $this->is_paid == 0 && $this->payment_proof == null) {
            $indicator = 'pay_now';
        } elseif ($this->deal_status == ACCEPTED && $customerType == BUYER && $this->is_paid == 0 && $this->payment_proof !== null) {
            $indicator = 'payment_submitted';
        } elseif ($this->deal_status == ACCEPTED && $customerType == SELLER && $this->is_paid == 1) {
            $indicator = 'ship_now';
        } elseif ($this->deal_status == TRANSIT && $customerType == SELLER) {
            $indicator = 'mark_delievered';
        } elseif ($this->deal_status == DELIVERED && $customerType == BUYER) {
            $indicator = 'mark_completed';
        }
        return $indicator;
    }

    //check deleted
    public function isSecondPartyCustomerDeleted($loggedInCustomerID)
    {
        $secondPartyId = $this->from_customer_id == $loggedInCustomerID ? $this->to_customer_id : $this->from_customer_id;
        return CustomerModel::withTrashed()->where('id', $secondPartyId)->exists()
            && CustomerModel::withTrashed()->where('id', $secondPartyId)->onlyTrashed()->exists();
    }
}
