<?php

namespace App\Http\Controllers;

use App\Models\AdminNotificationModel;
use App\Models\CustomerModel;
use App\Models\DealModel;
use App\Models\DealstatusHistoryModel;
use App\Models\disputeDealConversationModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Excel;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Barryvdh\DomPDF\Facade\Pdf;

class DealController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list - ongoing deals')->only('onGoingDeals');
        $this->middleware('permission:list - dispute deals')->only('disputed');
        $this->middleware('permission:list - Cancelled Deals')->only('cancelled');
        $this->middleware('permission:list - closed deals')->only('closed');
        $this->middleware('permission:list - hold amount deals')->only('holdAmount');
        $this->middleware('permission:list - Declined deals')->only('declined');
        $this->middleware('permission:view - deal detail')->only('show');
        $this->middleware('permission:update status - deal detail')->only('updateDealStatus');
        $this->middleware('permission:amount release - deal detail')->only('releaseAmount');
        $this->middleware('permission:conversation - dispute deals')->only(['submitAdminReplyDisputeConversation', 'getMessages']);
        //$this->middleware('permission:Terms and Conditions')->only(['termsAndConditions', 'termsAndConditionsUpdate']);
    }
    public function index(Request $request)
    {
        $deals = (new DealModel())->ongoingDealsPagination($request);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.deal.pagination', compact('deals'))->render()
            ]);
        }

        return view('admin.deal.index', compact('deals'));
    }

    public function onGoingDeals(Request $request)
    {
        $perPage = $request->get('per_page', 10);
        $query = DealModel::ongoingDealsPagination($request);
        $deals = $query->paginate($perPage);
        $deals->getCollection()->transform(function ($deal) {
            if (!$deal->payment_proof && $deal->is_paid == 0) {
                $deal->status_flag = null;
            } elseif ($deal->payment_proof && !$deal->shipping_proof && $deal->is_paid == 0) {
                $deal->status_flag = 'red';
            } elseif ($deal->shipping_proof && $deal->is_release == 0 && $deal->is_paid == 1 && $deal->is_hold == 1 && (in_array($deal->deal_status, [ACCEPTED, TRANSIT]))) {
                $deal->status_flag = 'yellow';
            } elseif ($deal->is_release == 0 && $deal->is_paid == 1 && $deal->is_hold == 1 && $deal->deal_status == COMPLETED) {
                $deal->status_flag = 'green';
            }
            return $deal;
        });
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.deal.ongoing.pagination', compact('deals'))->render()
            ]);
        }

        return view('admin.deal.ongoing.index', compact('deals'));
    }

    public function disputed(Request $request)
    {
        $perPage = $request->get('per_page', 10);
        $deals = DealModel::disputedDealsPagination($request)->paginate($perPage);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.deal.disputed.pagination', compact('deals'))->render()
            ]);
        }

        return view('admin.deal.disputed.index', compact('deals'));
    }
    public function cancelled(Request $request)
    {
        $perPage = $request->get('per_page', 10);
        $deals = DealModel::cancelledDealsPagination($request)->paginate($perPage);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.deal.cancelled.pagination', compact('deals'))->render()
            ]);
        }

        return view('admin.deal.cancelled.index', compact('deals'));
    }
    public function closed(Request $request)
    {
        $perPage = $request->get('per_page', 10);
        $deals = DealModel::closedDealsPagination($request)->paginate($perPage);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.deal.closed.pagination', compact('deals'))->render()
            ]);
        }

        return view('admin.deal.closed.index', compact('deals'));
    }

    public function holdAmount(Request $request)
    {
        $perPage = $request->get('per_page', 10);
        $query = DealModel::holdAmountDealsPagination($request);
        $deals = $query->paginate($perPage);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.deal.hold-amount.pagination', compact('deals'))->render()
            ]);
        }

        return view('admin.deal.hold-amount.index', compact('deals'));
    }
    public function declined(Request $request)
    {
        $perPage = $request->get('per_page', 10);
        $deals = DealModel::declinedDealsPagination($request)->paginate($perPage);
        if ($request->ajax()) {
            return response()->json([
                // 'data' => $deals->items()
                'data' => view('admin.deal.declined.pagination', compact('deals'))->render()
            ]);
        }

        return view('admin.deal.declined.index', compact('deals'));
    }

    public function show($id)
    {
        $realtions = [
            'fromCustomer',
            'toCustomer',
            'amountReleasedTo',
            'disputeDocs',
            'completedByAdmin.roles',
            'cancelByAdmin.roles',
            'disputeByAdmin.roles',
            'completedByAdmin',
            'paymentRejections',
            'shippingRejections',
            'latestPaymentRejection',
            'latestShippingRejection',
            'images'
        ];
        $deal = DealModel::with($realtions)->findOrFail($id);


        if (!$deal) {
            return redirect()->route('admin.deal.show')->with('error', __('message.invalid_details'));
        }

        if ($deal->deal_type == BUY) {

            $fromCustomerPrefix = $deal?->fromCustomer?->full_name . " (" . __('message.buyer') . ")";
            $toCustomerPrefix = $deal?->toCustomer?->full_name . " (" . __('message.seller') . ")";
            $releaseToSeller = $toCustomerPrefix;
            $releaseToSellerId = $deal?->toCustomer?->id;
        } else {
            $fromCustomerPrefix = $deal?->fromCustomer?->full_name . " (" . __('message.seller') . ")";
            $toCustomerPrefix = $deal?->toCustomer?->full_name .  " (" . __('message.buyer') . ")";
            $releaseToSeller = $fromCustomerPrefix;
            $releaseToSellerId = $deal?->fromCustomer?->id;
        }



        $deal->is_read = 1;
        $deal->save();


        //get docs by customer type
        $buyerId = $deal->deal_type == BUY ? $deal->from_customer_id : $deal->to_customer_id;
        $sellerId = $deal->deal_type == BUY ? $deal->to_customer_id : $deal->from_customer_id;

        //dispute conversation data
        $messages = disputeDealConversationModel::with('adminSender')->where('deal_id', $id)
            ->orderBy('created_at', 'asc')->get();

        $buyerMessages = $messages->filter(function ($msg) use ($buyerId) {
            return $msg->sender_id == $buyerId
                || ($msg->sender_type == 'admin' && $msg->receiver_id == $buyerId);
        });

        $sellerMessages = $messages->filter(function ($msg) use ($sellerId) {
            return $msg->sender_id == $sellerId
                || ($msg->sender_type == 'admin' && $msg->receiver_id == $sellerId);
        });
        AdminNotificationModel::where('deal_id', $deal->id)->update(['is_read' => 1]);



        $buyerDisputeDocs = $deal->disputeDocs->where('customer_id', $buyerId);
        $sellerDisputeDocs = $deal->disputeDocs->where('customer_id', $sellerId);

        //deal status history
        $dealStatusHistories = DealStatusHistoryModel::where('deal_id', $id)
            ->orderBy('created_at', 'asc')->get();

        //get release to customer at the time of dispute : accpet or reject
        $releaseToCustomerInfo = null;
        $releaseToCustomerId = null;

        if ($deal->deal_status == DISPUTE_ACCEPTED || $deal->deal_status == DISPUTE_REJECTED) {
            if ($deal->deal_status == DISPUTE_ACCEPTED) {
                // Release to buyer
                $releaseToCustomerId = ($deal->deal_type == BUY) ? $deal->from_customer_id : $deal->to_customer_id;
                $releaseToCustomerName = ($deal->deal_type == BUY) ? $deal?->fromCustomer?->full_name : $deal?->toCustomer?->full_name;
                $roleLabel = __('message.buyer');
            } else {
                // DISPUTE_REJECTED - release to seller
                $releaseToCustomerId = ($deal->deal_type == BUY) ? $deal->to_customer_id : $deal->from_customer_id;
                $releaseToCustomerName = ($deal->deal_type == BUY) ? $deal?->toCustomer?->full_name : $deal?->fromCustomer?->full_name;
                $roleLabel = __('message.seller');
            }

            $releaseToCustomerInfo = $releaseToCustomerName . " (" . $roleLabel . ")";
        }


        return view('admin.deal.show', compact(
            [
                'deal',
                'fromCustomerPrefix',
                'toCustomerPrefix',
                'messages',
                'buyerMessages',
                'sellerMessages',
                'dealStatusHistories',
                'buyerDisputeDocs',
                'sellerDisputeDocs',
                'releaseToCustomerInfo',
                'releaseToCustomerId',
                'releaseToSeller',
                'releaseToSellerId',
                'buyerId',
                'sellerId'
            ]
        ));
    }

    public function updateDealStatus(Request $request, $id)
    {
        try {
            //dd($request->deal_status);
            DB::beginTransaction();
            $deal = (new DealModel)->find($id);
            if (empty($deal)) {
                $message = __('message.invalid_details');
                $extra = ['redirect' =>  route('admin.deal.index')];
                return jsonResponse(false, 500, $message);
            }
            $cancelWithPayment = True;
            $shouldUpdateStatus = false;
            // find buyer and seller

            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return jsonResponse(false, 400, __('message.invalid_deal_data'));
                //return apiResponse(400, "Invalid Deal Data");
            }

            $buyer = $customerDetails[BUYER] ?? null;
            $seller = $customerDetails[SELLER] ?? null;
            $deletedMsg = null;


            //check if one customer deleted then abort the deal
            /*if ($buyer->deleted_at) {
                $deletedMsg = __('message.buyer_deleted');
            } elseif ($seller->deleted_at) {
                $deletedMsg = __('message.seller_deleted');
            }*/
            if ($deletedMsg) {
                return jsonResponse(false, 207, $deletedMsg);
            }
            //identify the sender receiver
            $fromCustomerId = $deal->from_customer_id;
            $toCustomerId = $deal->to_customer_id;

            $nType = '';
            $nTitle = '';
            $nTitleEs = '';
            $nMessage = '';
            $nMessageEs = '';

            //set is_hold true
            // $hold_true_status = [ACCEPTED, TRANSIT, COMPLETED, ON_DISPUTE];
            // if (in_array($request->deal_status, $hold_true_status)) {
            //     $deal->is_hold = 1;
            // }
            if ($request->deal_status == TRANSIT) {
                if ($deal->deal_status == ACCEPTED) {
                    $shouldUpdateStatus = true;
                }
                //notification 
                $nType = 'shipped_item';
                $nTitle = 'Your Order has been shipped';
                $nTitleEs = 'Tu pedido ha sido enviado';
                $nMessage = "By Admin: The item for the deal #{$deal->deal_number} has been " . TRANSIT . ".";
                $nMessageEs = "Por el Administrador: El artículo para el trato #{$deal->deal_number} ha sido enviado.";
            }
            if ($request->deal_status == DELIVERED) {
                if ($deal->deal_status == TRANSIT) {
                    $shouldUpdateStatus = true;
                }
                //notification 
                $nType = 'delivered_item';
                $nTitle = 'Your Order has been Delivered';
                $nTitleEs = 'Tu pedido ha sido entregado';
                $nMessage = "By Admin: The item for the deal #{$deal->deal_number} has been " . DELIVERED . ".";
                $nMessageEs = "Por el Administrador: El artículo para el trato #{$deal->deal_number} ha sido entregado.";
            }

            if ($request->deal_status == CANCEL) {
                if ($request->has('cancel_reason') && !empty($request->cancel_reason)) {
                    $deal->cancel_by = auth()->user()->id;
                    $deal->cancel_reason = $request->cancel_reason;
                    $deal->cancel_type = 'admin';
                    if ($deal->is_hold == 0 && $deal->is_paid == 0 && (!in_array($deal->deal_status, [ACCEPTED, TRANSIT]))) {
                        $deal->end_reason = CANCEL;
                        $deal->deal_status = CLOSED;
                        $cancelWithPayment  = false;
                        $shouldUpdateStatus = false;
                    } else {

                        $cancelWithPayment  = true;
                        $shouldUpdateStatus = true;
                    }
                    //notification 
                    $nType = 'cancelled_deal';
                    $nTitle = 'Deal Cancelled';
                    $nTitleEs = 'Trato cancelado';
                    $nMessage = "By Admin: The deal #{$deal->deal_number} has been cancelled.";
                    $nMessageEs = "Por el Administrador: El trato #{$deal->deal_number} ha sido cancelado.";
                }
            }

            if ($request->deal_status == ON_DISPUTE) {
                if (in_array($deal->deal_status, [TRANSIT, DELIVERED])) {
                    $shouldUpdateStatus = true;
                }
                $deal->dispute_request_by_type = 'admin';
                $deal->dispute_request_by_id = auth()->user()->id;
                $deal->dispute_request_date = now();
                $deal->dispute_reason = $request->dispute_reason;
                $nType = 'deal_on_dispute';
                $nTitle = 'Deal is in Dispute';
                $nTitleEs = 'El trato está en disputa';
                $nMessage = "By Admin: The deal #{$deal->deal_number} has been disputed.";
                $nMessageEs = "Por el Administrador: El trato #{$deal->deal_number} ha sido disputado.";
            }
            if ($request->deal_status == DISPUTE_UNDER_REVIEW) {
                if (in_array($deal->deal_status, [ON_DISPUTE])) {
                    $shouldUpdateStatus = true;
                }
                $nType = 'dispute_under_review';
                $nTitle = 'Dispute Under Review';
                $nTitleEs = 'Disputa en revisión';
                $nMessage = "By Admin: The dispute for deal #{$deal->deal_number} is under review.";
                $nMessageEs = "Por el Administrador: La disputa del trato #{$deal->deal_number} está en revisión.";
            }


            $statusToLog = $request->deal_status;
            $deal->status_datetime = now();


            //notification to second party


            $is_updation_by_admin = 1;
            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            $nType = $nType;
            $nTitle = $nTitle;
            $nTitleEs = $nTitleEs;
            $nMesage = $nMessage;
            $nMesageEs = $nMessageEs;
            $secondPartyId = auth()->user()->id;

            if ($request->deal_status == COMPLETED) {
                if (in_array($deal->deal_status, [DELIVERED])) {
                    $shouldUpdateStatus = true;
                }
                $deal->completed_by_type = 'admin';
                //dd(auth()->user()->id);
                $deal->completed_by_id = auth()->user()->id;
                $nType = 'completed_deal';
                $nTitle = 'Deal Completed Successfully';
                $nTitleEs = 'Trato completado con éxito';

                $formattedAmount = number_format($deal->deal_amount);
                $sellerMessage = "By Admin: The deal #{$deal->deal_number} has been completed. Deal amount \${$formattedAmount} will be released soon.";
                $sellerMessageEs = "Por el Administrador: El trato #{$deal->deal_number} ha sido completado. El monto del trato de \${$formattedAmount} será liberado pronto.";

                $buyerMessage = "By Admin: The deal #{$deal->deal_number} has been completed.";
                $buyerMessageEs = "Por el Administrador: El trato #{$deal->deal_number} ha sido completado.";

                if ($fromCustomerId === $seller->id) {
                    saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                } elseif ($toCustomerId === $seller->id) {

                    saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                }
            } elseif ($request->deal_status == DISPUTE_ACCEPTED) {
                if (in_array($deal->deal_status, [DISPUTE_UNDER_REVIEW])) {
                    $shouldUpdateStatus = true;
                }
                //this type based on buyer side
                $nType = 'dispute_accepted';
                $nBuyerTitle = 'Dispute Accepted';
                $nBuyerTitleEs = 'Disputa aceptada';
                $buyerMessage = "By Admin: The dispute for deal #{$deal->deal_number} has been accepted. The amount transfer in process.";
                $buyerMessageEs = "Por el administrador: La disputa del trato #{$deal->deal_number} ha sido aceptada. La transferencia del monto está en proceso.";
                //sellelr side notification must be vice versa
                $nSellerTitle = 'Dispute resolved against you';
                $nSellerTitleEs = 'Disputa resuelta en su contra';
                $sellerMessage = "By Admin: The dispute for deal #{$deal->deal_number} has been resolved against you. The amount transfer in process.";
                $sellerMessageEs = "Por el administrador: La disputa del trato #{$deal->deal_number} ha sido resuelta en tu contra. La transferencia del monto está en proceso";


                saveNotification($buyer->id, $nType, $nBuyerTitle, $nBuyerTitleEs,  $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                saveNotification($seller->id, $nType,  $nSellerTitle, $nSellerTitleEs,  $sellerMessage,  $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
            } elseif ($request->deal_status == DISPUTE_REJECTED) {
                if (in_array($deal->deal_status, [DISPUTE_UNDER_REVIEW])) {
                    $shouldUpdateStatus = true;
                }
                //this type based on buyer side
                $nType = 'dispute_rejected';


                $nBuyerTitle = 'Dispute Rejected';
                $nBuyerTitleEs = 'Disputa rechazada';
                $buyerMessage = "By Admin: The dispute for deal #{$deal->deal_number} has been rejected. The amount transfer in process.";
                $buyerMessageEs = "Por el administrador: La disputa del trato #{$deal->deal_number} ha sido rechazada. La transferencia del monto está en proceso.";
                //sellelr side notification must be vice versa
                $nSellerTitle = 'Dispute resolved in your favor';
                $nSellerTitleEs = 'Disputa resuelta a su favor';
                $sellerMessage = "By Admin: The dispute for deal #{$deal->deal_number} has been resolved in your favor. The amount transfer in process.";
                $sellerMessageEs = "Por el administrador: La disputa del trato #{$deal->deal_number} ha sido resuelta a tu favor. La transferencia del monto está en proceso.";


                saveNotification($buyer->id, $nType, $nBuyerTitle, $nBuyerTitleEs,  $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                saveNotification($seller->id, $nType,  $nSellerTitle, $nSellerTitleEs,  $sellerMessage,  $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
            } else {
                saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
            }
            if ($shouldUpdateStatus && $cancelWithPayment) {
                $deal->deal_status = $request->deal_status;
            }

            $deal->save();

            //save deal stataus history
            DealstatusHistoryModel::saveHistory($deal->id, $statusToLog);
            DB::commit();
            $message = __('message.deal_status_updated');
            $extra = [
                'redirect' =>  route('admin.deal.show', ['deal' => $id])
            ];
            //dd($extra);
            return jsonResponse(true, 200, $message, $extra);
            // return redirect()->route('admin.deal.show', ['deal' => $id]);
        } catch (\Exception $e) {
            DB::rollback();
            $message = __('message.internal_server_error') . $e->getMessage() . $e->getFile() . $e->getLine();
            return jsonResponse(false, 500, $message);
        }
    }



    public function releaseAmount($dealId, Request $request)
    {

        try {
            DB::beginTransaction();
            $update = [];
            $fecthPaymentStatus = false;
            $deal = (new DealModel)->find($dealId);
            if (!$deal) {
                // abort(404);
                $message = __('message.invalid_details');
                return jsonResponse(false, 500,  $message);
            }

            if (!$deal->is_hold) {
                $message = __('message.invalid_details');
                return jsonResponse(false, 500, $message);
            }
            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return jsonResponse(false, 400, __('message.invalid_deal_data'));
                //return apiResponse(400, "Invalid Deal Data");
            }
            $buyer = $customerDetails[BUYER];
            $seller = $customerDetails[SELLER];
            /* $deletedMsg = null;
            if ($buyer->deleted_at) {
                $deletedMsg = __('message.buyer_deleted');
            } elseif ($seller->deleted_at) {
                $deletedMsg = __('message.seller_deleted');
            }
            if ($deletedMsg) {
                return jsonResponse(false, 207, $deletedMsg);
            }*/
            //identify the sender receiver
            $fromCustomerId = $deal->from_customer_id;
            $toCustomerId = $deal->to_customer_id;

            $nType = '';
            $nTitle = '';
            $nTitleEs = '';
            $nMessage = '';
            $nMessageEs = '';
            $secondPartyId = auth()->user()->id;
            $is_updation_by_admin = 1;
            $nDealnumber = $deal->deal_number;

            $amount_released_to = '';
            if ($deal->deal_status == COMPLETED) {
                $sellerId = ($deal->deal_type == BUY) ? $deal->to_customer_id : $deal->from_customer_id;
                $update = [
                    'deal_status' => CLOSED,
                    'is_release' => 1,
                    'is_hold' => 0,
                    'amount_released_to' =>  $request->completed_relase_to,
                ];
                $amount_released_to = $sellerId;
                //verify to whom release amount, have bank details
                $sellerDetails = CustomerModel::where('id', $sellerId)->first();
                if (in_array(null, [
                    $sellerDetails->account_holder_name,
                    $sellerDetails->account_number,
                    $sellerDetails->bank_name
                ])) {
                    $extra = ['redirect' => route('admin.deal.show', $deal->id)];
                    return jsonResponse(false, 207, __('message.bank_details_null'), $extra);
                }
                /*if ($request->service_charge == 0) {
                    $update += [
                        'transfer_fees_in_percent' => 0,
                        'admin_earnings' => 0,
                        'release_amount' => $deal->deal_amount,
                    ];
                }*/
                //notification send to seller after release paymnet
                $nType = 'amount_release';
                $nTitle = 'Amount Released';
                $nTitleEs = 'Monto Liberado';
                $sellerMessage = "By Admin: The amount has been released for the deal #{$deal->deal_number} ";
                $sellerMessageEs = "Por el Administrador: Se ha liberado el monto para el acuerdo #{$deal->deal_number}";

                if ($fromCustomerId === $sellerId) {
                    $secondPartyId = $toCustomerId;
                    saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                } elseif ($toCustomerId === $sellerId) {
                    $secondPartyId = $fromCustomerId;
                    saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                }
                $update = [
                    'deal_status' => CLOSED,
                    'end_reason' => $deal->deal_status,
                    'is_release' => 1,
                    'is_hold' => 0,
                    'amount_released_to' =>  $sellerId,

                ];
            }
            /*if ($request->service_charge == 0) {
                    $update += [
                        'transfer_fees_in_percent' => 0,
                        'admin_earnings' => 0,
                        'release_amount' => $deal->deal_amount,
                    ];
                }*/
            //check same dank detail verification like in complete case
            $receiverForCancel = CustomerModel::where('id', $request->amount_released_to)->first();
            $receiverForDispute = CustomerModel::where('id', $request->disputed_relase_to)->first();
            if ((in_array(null, [
                $receiverForCancel->account_holder_name ?? null,
                $receiverForCancel->account_number ?? null,
                $receiverForCancel->bank_name ?? null
            ]) && $receiverForCancel) ||  (in_array(null, [
                $receiverForDispute->account_holder_name ?? null,
                $receiverForDispute->account_number ?? null,
                $receiverForDispute->bank_name ?? null
            ])) && $receiverForDispute) {
                $extra = ['redirect' => route('admin.deal.show', $deal->id)];
                return jsonResponse(false, 207, __('message.bank_details_null'), $extra);
            } else if ($deal->deal_status == CANCEL && $deal->is_hold == 1) {

                $update = [
                    'deal_status' => CLOSED,
                    'end_reason' => $deal->deal_status,
                    'amount_released_to' =>  $request->amount_released_to,
                    'is_release' => 1,
                    'is_hold' => 0
                ];
                $amount_released_to = $request->amount_released_to;
                $nType = 'cancelled_deal';
                $nTitle = 'Deal Cancelled';



                $formattedAmount = number_format($deal->deal_amount);
                if ($request->amount_released_to == $seller->id) {

                    $sellerMessage = "By Admin: The deal amount \${$formattedAmount} for deal #{$deal->deal_number} released to you.";
                    $sellerMessageEs = "Por el administrador: El monto del trato de \${$formattedAmount} para el trato n.º {$deal->deal_number} ha sido liberado para usted.";
                    $buyerMessage = "By Admin: The deal amount \${$formattedAmount} for deal #{$deal->deal_number} released to seller.";
                    $buyerMessageEs = "Por el administrador: El monto del trato de \${$formattedAmount} para el trato n.º {$deal->deal_number} ha sido liberado al vendedor.";
                    saveNotification($seller->id, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($buyer->id, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                } elseif ($request->amount_released_to == $buyer->id) {

                    $sellerMessage = "By Admin: The deal amount \${$formattedAmount} for deal #{$deal->deal_number} released to buyer.";
                    $sellerMessageEs = "Por el administrador: El monto del trato de \${$formattedAmount} para el trato n.º {$deal->deal_number} ha sido liberado al comprador.";
                    $buyerMessage = "By Admin: The deal amount \${$formattedAmount} for deal #{$deal->deal_number} released to you.";
                    $buyerMessageEs = "Por el administrador: El monto del trato de \${$formattedAmount} para el trato n.º {$deal->deal_number} ha sido liberado para usted.";
                    saveNotification($seller->id, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($buyer->id, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                }
                /*if ($request->service_charge == 0) {
                    $update += [
                        'transfer_fees_in_percent' => 0,
                        'admin_earnings' => 0,
                        'release_amount' => $deal->deal_amount,
                    ];
                }*/
            } elseif ($deal->deal_status == DISPUTE_ACCEPTED) {
                $fecthPaymentStatus = true;

                $update = [
                    'deal_status' => CLOSED,
                    'end_reason' => $deal->deal_status,
                    'status_datetime' => now(),
                    'amount_released_to' =>  $request->disputed_relase_to,
                    'is_release' => 1,
                    'is_hold' => 0

                ];

                $nType = 'amount_transfer';
                $nTitle = 'Amount Transferred';
                $nTitleEs = 'Monto Transferido';
                $sellerMessage = "By Admin: Disputed deal #{$deal->deal_number}'s amount transferred to Buyer";
                $sellerMessageEs = "Por el administrador: El monto del trato en disputa #{$deal->deal_number} fue transferido al comprador";

                $buyerMessage = "By Admin: Disputed deal #{$deal->deal_number}'s amount transferred to you.";
                $buyerMessageEs = "Por el administrador: El monto del trato en disputa #{$deal->deal_number} fue transferido a usted.";

                if ($fromCustomerId === $buyer->id) {
                    saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                } elseif ($toCustomerId === $buyer->id) {
                    saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                }
                //amount release to sttaus save:
                $paymentStatus = ['key' => MONEY_RETURNED_BUYER, 'value' => __('message.money_returned_buyer')];
            } elseif ($deal->deal_status == DISPUTE_REJECTED) {
                $fecthPaymentStatus = true;

                $update = [
                    'deal_status' => CLOSED,
                    'end_reason' => $deal->deal_status,
                    'status_datetime' => now(),
                    'amount_released_to' =>  $request->disputed_relase_to,
                    'is_release' => 1,
                    'is_hold' => 0

                ];

                $nType = 'amount_transfer';
                $nTitle = 'Amount Transferred';
                $nTitleEs = 'Monto Transferido';
                $sellerMessage = "By Admin: Disputed deal #{$deal->deal_number}'s amount transferred to you.";
                $sellerMessageEs = "Por el administrador: El monto del trato en disputa #{$deal->deal_number} ha sido transferido a ti.";

                $buyerMessage = "By Admin: Disputed deal #{$deal->deal_number}'s amount transferred to seller.";
                $buyerMessageEs = "Por el administrador: El monto del trato en disputa #{$deal->deal_number} ha sido transferido al vendedor..";

                if ($fromCustomerId === $seller->id) {
                    saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                } elseif ($toCustomerId === $seller->id) {
                    saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $sellerMessage, $sellerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                    saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $buyerMessage, $buyerMessageEs, $secondPartyId, $nDealnumber, $dealId, $is_updation_by_admin);
                }
                $paymentStatus = ['key' => MONEY_DEPOSITED_SELLER, 'value' => __('message.money_deposited_seller')];
            }

            $deal->release_amount =  $deal->deal_amount;
            $deal->update($update);
            $deal->refresh();

            if (!$fecthPaymentStatus) {
                $paymentStatus =  paymentStatusCustomText($deal, $amount_released_to);
            }

            //save deal stataus history
            DealstatusHistoryModel::saveHistory($deal->id, $paymentStatus['key']);
            //check good deal bad deals
            processDealsOutcome($deal);


            $message = __('message.updated_success');
            $extra = ['redirect' => route('admin.deal.show', $deal->id)];
            DB::commit();
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }

    //disputed deal conversation
    public function submitAdminReplyDisputeConversation(Request $request)
    {
        try {
            DB::beginTransaction();
            $validation = disputeDealConversationModel::disputeConversationValidation($request->all());
            if ($validation->fails()) {
                return jsonResponse(false, 400, $validation->errors());
            }
            $user = auth()->user();
            $senderId = $user->id;
            $dealId = (int)$request->deal_id;
            $deal = DealModel::select('id', 'deal_number', 'dispute_request_by_id', 'from_customer_id', 'to_customer_id', 'deal_type')->where('id', $dealId)->first();
            $from = $deal->from_customer_id;
            $to = $deal->to_customer_id;
            $dealType = $deal->deal_type;
            $dealIdResult = $deal->id;

            $senderType = (count($user->roles) > 0) ? $user->roles[0]->name : 'Admin';
            disputeDealConversationModel::create([
                'deal_id' => $dealIdResult,
                'sender_id' =>  $senderId,
                'sender_type' => 'admin',
                'receiver_id' => $request->receiver_id,
                'receiver_type' => $request->receiver_type,
                'message' => $request->message,
            ]);

            //$receiverId = disputeCreaedByWhichCustomer($from, $to, $dealType);

            //notification for send conversation to customer
            $nDealnumber = $deal->deal_number;

            $nType = 'reply_by_admin';
            $nTitle = "Dispute Update";
            $nTitleEs = 'Actualización de disputa';
            $nMesage = "An admin has responded to your dispute for Deal #{$nDealnumber}. Please check the conversation for further details";
            $nMesageEs = "Un administrador ha respondido a su disputa por Deal #{$nDealnumber}. Por favor revise la conversación para más detalles.";
            saveNotification($request->receiver_id, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, NULL, $nDealnumber, $dealIdResult, 0);

            DB::commit();


            return jsonResponseForConversation(true, 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }

    //get dispute conversation
    public function getMessages($id)
    {
        try {
            DB::beginTransaction();


            $deal = DealModel::with(['fromCustomer', 'toCustomer'])->where('id', $id)->first();
            if (!$deal) {
                return jsonResponse(false, 400, __('message.please_provide_valid_deal'));
            }

            //get docs by customer type
            $buyerId = $deal->deal_type == BUY ? $deal->from_customer_id : $deal->to_customer_id;
            $sellerId = $deal->deal_type == BUY ? $deal->to_customer_id : $deal->from_customer_id;

            //dispute conversation data
            $messages = disputeDealConversationModel::with('adminSender')->where('deal_id', $id)
                ->orderBy('created_at', 'asc')->get();

            $buyerMessages = $messages->filter(function ($msg) use ($buyerId) {
                return $msg->sender_id == $buyerId
                    || ($msg->sender_type == 'admin' && $msg->receiver_id == $buyerId);
            });

            $sellerMessages = $messages->filter(function ($msg) use ($sellerId) {
                return $msg->sender_id == $sellerId
                    || ($msg->sender_type == 'admin' && $msg->receiver_id == $sellerId);
            });
            DB::commit();
            return response()->json([
                'buyer_html' => view('admin.deal.conversation', [
                    'messages' => $buyerMessages,
                    'deal' => $deal,
                ])->render(),
                'seller_html' => view('admin.deal.conversation', [
                    'messages' => $sellerMessages,
                    'deal' => $deal,
                ])->render(),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }

    // public function getUnreadMessages()
    // {
    //     $notifications = AdminNotificationModel::where('is_read', 0)
    //         ->orderBy('created_at', 'desc')
    //         ->get(['id', 'deal_id', 'message', 'is_read']);

    //     return response()->json([
    //         'notifications' => $notifications
    //     ]);
    // }

    /*public function releaseAmount($id, Request $request)
    {
        try {
            $deal = (new DealModel)->find($id);
            if (!empty($deal)) {
                DB::beginTransaction();

                $update = [
                    'status' => CLOSED,
                ];

                if (checkbox == 0) {
                    $update += [
                        'tras' => 0,
                        'admin_ear' => 0,
                        'rea_amt' => $deal->amount,
                    ];
                }

                $deal->update($update);
                DB::commit();

                $message = 'Deal deleted successfully.';
                $extra = ['redirect' => $redirectRoutes[$dealstatusType] ?? route('admin.deals.ongoing')];
                return jsonResponse(true, 200, $message, $extra);



                // return redirect()->route('admin.customer')->with('success', 'Customer deleted successfully.');
            } else {
                $message = 'Invalid Details';
                $extra = ['redirect' => route('admin.deals.ongoing')];
                return jsonResponse(false, 500, $message, $extra);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            $message = 'Internal Server Error: ' . $e->getMessage();
            $extra = ['redirect' => route('admin.customer.index')];
            return jsonResponse(false, 500, $message, $extra);
        }
    }*/
    public function markPaid($id)
    {

        try {
            $deal = DealModel::find($id);
            if (!$deal) {
                return jsonResponse(false, 400, __('message.please_provide_valid_deal'));
            }
            //check to whom release amount have bank details

            $is_updation_by_admin = 1;
            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return jsonResponse(false, 400, __('message.invalid_deal_data'));
                //return apiResponse(400, "Invalid Deal Data");
            }
            $buyer = $customerDetails[BUYER];
            $seller = $customerDetails[SELLER];
            /*$deletedMsg = null;
            if ($buyer->deleted_at) {
                $deletedMsg = __('message.buyer_deleted');
            } elseif ($seller->deleted_at) {
                $deletedMsg = __('message.seller_deleted');
            }
            if ($deletedMsg) {
                return jsonResponse(false, 207, $deletedMsg);
            }*/
            //identify the sender receiver
            $fromCustomerId = $deal->from_customer_id;
            $toCustomerId = $deal->to_customer_id;
            $deal->is_paid = 1;
            $deal->is_hold = 1;
            $statusToLog = PAYMENT_RECEIVED;
            //notification 
            $nType = 'payment_approved';
            $nTitle = 'Payment Approved';
            $nTitleEs = 'Pago aprobado';
            $nMessage = "Your payment has been approved for the deal #{$deal->deal_number}";
            $nMessageEs = "Su pago ha sido aprobado para el trato n.º {$deal->deal_number}";

            if ($fromCustomerId === $buyer->id) {
                //dd($fromCustomerId, $toCustomerId, "yyyy");
                saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $nMessage, $nMessageEs, $toCustomerId, $nDealnumber, $dealId, $is_updation_by_admin);
            } elseif ($toCustomerId === $buyer->id) {
                //dd($fromCustomerId, $toCustomerId);
                saveNotification($toCustomerId, $nType, $nTitle, $nTitleEs, $nMessage, $nMessageEs, $fromCustomerId, $nDealnumber, $dealId, $is_updation_by_admin);
            }

            //seller side notification for payment
            $nTitleSeller = 'Payment received';
            $nTitleSellerEs = 'Pago recibido';
            $nMessageSeller = "Payment for deal #{$deal->deal_number} has been received from the buyer.";
            $nMessageSellerEs = "Se ha recibido el pago del comprador por el trato n.º {$deal->deal_number}";
            if ($fromCustomerId === $seller->id) {
                saveNotification($fromCustomerId, $nType, $nTitleSeller, $nTitleSellerEs, $nMessageSeller, $nMessageSellerEs, $toCustomerId, $nDealnumber, $dealId, $is_updation_by_admin);
            } elseif ($toCustomerId === $seller->id) {
                saveNotification($toCustomerId, $nType, $nTitleSeller, $nTitleSellerEs, $nMessageSeller, $nMessageSellerEs, $fromCustomerId, $nDealnumber, $dealId, $is_updation_by_admin);
            }
            $deal->save();
            //save deal stataus history
            DealstatusHistoryModel::saveHistory($deal->id, $statusToLog);
            DB::commit();
            $message = __('message.payment_status_updated_successfully');
            $extra = [
                'redirect' =>  route('admin.deal.show', ['deal' => $id])
            ];
            //dd($extra);
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            // $errorMessage = __('message.internal_server_error') . ' ' . $e->getMessage() .
            //     ' in file ' . $e->getFile() . ' on line ' . $e->getLine();
            // return jsonResponse(false, 500, $errorMessage);
            return jsonResponse(false, 500, 'Internal server error.');
        }
    }
}
