<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\AdminModel;
use App\Models\CustomerModel;
use App\Models\DealModel;
use App\Models\disputDdealDocumentModel;
use App\Models\disputeDealConversationModel;
use App\Models\NotificationModel;
use App\Models\settingModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth as CustomerAuth;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;
use App\Models\DealstatusHistoryModel;
use App\Models\AdminNotificationModel;
use App\Models\DealImages;
use App\Services\InviteService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Validation\Rule;

use function PHPSTORM_META\map;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;

class DealApiController extends Controller
{

    public function createDeal(Request $request, InviteService $inviteService)
    {

        try {
            DB::beginTransaction();
            $isInvite = false;
            $deal_image_upload = 0;
            $dialCodeIso = strtoupper($request->dial_code_iso);
            $from_customer_id = getLoggedInCustomerId();
            $from_customer = CustomerModel::select('full_name', 'dial_code', 'mobile_number', 'status')->where('id', $from_customer_id)->first();
            $from_cusmomer_name = $from_customer ?? 'Unknown';
            $to_customer = CustomerModel::where('dial_code', $request->dial_code)
                ->where('mobile_number', $request->mobile_number)
                ->whereNull('deleted_at')->first();

            //check if creator seller then upload item images

            if ($request->deal_type == SELL) {
                if (!$request->hasFile('images') || empty($request->file('images'))) {
                    return apiResponse(400, __('message.seller_must_upload_deal_images'));
                }
                $deal_image_upload = 1;
            }
            $validator = (new DealModel)->dealCreateValiation($request->all());

            if ($validator->fails()) {
                return apiResponse(400, $validator->errors()->first());
            }
            //request reciever mobie validation
            if (!isValidNumber($request->dial_code, $dialCodeIso, $request->mobile_number)) {
                return apiResponse(400, __('message.mobile_number_invalid'));
            }
            if (!$to_customer) {
                $isInvite = (int)true;
                //save info of customer
                $to_customer = CustomerModel::create([
                    'full_name' => 'Invited Customer',
                    'email' => null,
                    'password' => null,
                    'dial_code' => $request->dial_code,
                    'dial_code_iso' => $dialCodeIso,
                    'mobile_number' => $request->mobile_number,
                    'status' => INACTIVE,
                    'is_invited' => $isInvite
                ]);
                $to_customer_moble_number = $request->dial_code . $request->mobile_number;
                $inviteService->sendInvite($to_customer_moble_number);

                $isInviteMessage = $request->deal_type === 'buy' ? __('message.seller_invited') : __('message.buyer_invited');
            }
            //now check the case if customer is guest user
            if ($to_customer) {

                if ($to_customer->is_invited && $to_customer->status == INACTIVE) {
                    $isInvite = true;
                    $to_customer_moble_number = $request->dial_code . $request->mobile_number;
                    //check that if both have existnece deal, then request again will not create another deal
                    $existingDeal = DealModel::where(function ($q) use ($from_customer_id, $to_customer) {
                        $q->where('from_customer_id', $from_customer_id)
                            ->where('to_customer_id', $to_customer->id);
                    })
                        ->where('deal_status', PENDING)
                        ->first();
                    if ($existingDeal) {
                        $data = [
                            'deal_id' => $existingDeal->id,
                            'deal_number' => $existingDeal->deal_number,
                            'is_invite' => $isInvite,
                        ];
                        $inviteService->sendInvite($to_customer_moble_number);
                        return apiResponse(200, __(
                            'message.deal_already_exists_forever',
                            ['mobile_number' => $to_customer_moble_number]
                        ), $data);
                    }

                    $inviteService->sendInvite($to_customer_moble_number);
                    $isInviteMessage = $request->deal_type === 'buy' ? __('message.seller_invited') : __('message.buyer_invited');
                }
            }
            if ($request->dial_code == $from_customer->dial_code && $request->mobile_number == $from_customer->mobile_number) {
                return apiResponse(400, __('message.mobile_number_invalid'));
            }
            //login customer is inactive
            if ($from_customer->status == INACTIVE) {
                $message = __('message.you_are_inaactive_contact');
                $message_description = __('message.inactive_contact_administrator');
                return apiResponse(400, $message, [], '', $message_description);
            }


            //calulate admin earnings and released amount 
            $transferFees = settingModel::where('setting_name', 'transfer_fees')->value('description') ?? 0;
            $admin_earnings = ($request->deal_amount * $transferFees) / 100;
            $totalDealAmount = $request->deal_amount + $admin_earnings;

            //  $request->deal_amount = round($request->deal_amount, 2);
            // $admin_earnings = round(($request->deal_amount * $transferFees) / 100, 2);
            // $totalDealAmount = $request->deal_amount + $admin_earnings;

            $lastDeal = DealModel::latest('id')->first();
            $nextNumber = $lastDeal ? ((int) substr($lastDeal->deal_number, 1)) + 1 : 1;
            $dealNumber = 'D' . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);

            //save deal
            $savedDeal =   DealModel::create([
                'deal_title' => $request->deal_title,
                'from_customer_id' => $from_customer_id,
                'to_customer_id' => $to_customer->id,
                'deal_amount' => $request->deal_amount,
                'transfer_fees_in_percent' => $transferFees,
                'admin_earnings' => $admin_earnings,
                'total_deal_Amount' => $totalDealAmount,
                'deal_type' => $request->deal_type,
                'deal_status' => PENDING,
                'deal_number' => $dealNumber,
                'currency_code' => $request->currency_code

            ]);

            //check if creator seller then upload item images

            if ($deal_image_upload == 1) {
                $uploadedImage =   uploadmultifiles('images', DEAL_IMAGES);
                foreach ($uploadedImage as $imageName) {

                    DealImages::create([
                        'deal_id' => $savedDeal->id,
                        'customer_id' => getLoggedInCustomerId(),
                        'image' => $imageName
                    ]);
                }
            }

            //notification for send request deal api to second party
            $from_cusmomer_name = $from_customer->full_name;
            $nDealnumber = $dealNumber;
            $dealId = $savedDeal->id;
            $nType = 'deal_request_get';
            $nTitle = "Deal Request";
            $nTitleEs = 'Solicitud de trato';
            $nMesage = "You have received the deal request from " . $from_cusmomer_name;
            $nMesageEs = "Has recibido la solicitud de trato de " . $from_cusmomer_name;
            saveNotification($to_customer->id, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $from_customer_id, $nDealnumber, $dealId, 0);

            //save deal status to history
            DealstatusHistoryModel::saveHistory($savedDeal->id, PENDING);


            DB::commit();
            //send reuquest notification to to_customer_id

            $tocustomerType =  ($savedDeal->deal_type == BUY)  ? __('message.seller')  : __('message.buyer');
            $data = [
                'deal_id' => $savedDeal->id,
                'deal_type' => $savedDeal->deal_type,
                'is_invite' => $isInvite,
            ];
            if ($isInvite && $isInviteMessage) {
                $message = $isInviteMessage;
                $message_description = "";
            } else {
                $message = __('message.request_submitted');
                $message_description = __('message.request_submitted_detail', ['tocustomerType' => $tocustomerType]);
            }

            return apiResponse(200, $message, $data, false, $message_description);
        } catch (\Exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }

    public function viewAllDeals(Request $request)
    {


        try {
            DB::beginTransaction();
            $validation = Validator::make($request->all(), [
                'current_page' => 'required|integer',

            ]);

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $currentPage = $request->current_page;

            $perPage = 10;
            $skip = ($currentPage - 1) * $perPage;
            $customerId = getLoggedInCustomerId();



            $deals = DealModel::where(function ($query) use ($customerId) {
                $query->where('from_customer_id', $customerId)
                    ->orwhere('to_customer_id', $customerId);
            }) // ->where('is_paid', 1) Only include paid deals
                ->orderBy('updated_at', 'desc') // Order by updated_at descending
                ->get();


            //invalid or empty
            if ($deals->isEmpty()) {
                DB::commit();
                return apiResponse(200, __('message.no_deals_available'));
            }


            $totalDeals = $deals->count();
            $paginatedDeals = $deals->slice($skip, $perPage)->values();
            $data = $paginatedDeals->map(function ($deal) use ($customerId) {
                $secondPartyColumn =  ($customerId == $deal->from_customer_id) ? 'to_customer_id' : 'from_customer_id';
                $secondParty = CustomerModel::withTrashed()->find($deal->$secondPartyColumn);
                $dealType = dealTypeViceVersa($deal, $customerId) == 'Buy' ? __('message.buy') : __('message.sell');

                return [
                    'deal_id' => $deal->id,
                    'deal_number' => $deal->deal_number,
                    'deal_title' => $deal->deal_title,
                    'deal_amount' => $deal->deal_amount,
                    'deal_amount_in_format' => formatCurrency($deal->deal_amount, $deal->currency_code, true),
                    'deal_type' => $dealType,
                    'deal_status' => checkDisputeDeal($deal->deal_status),
                    'deal_status_lang_wise' => __('readableStatuses.' . checkDisputeDeal($deal->deal_status)),
                    'deal_next_step' => $deal->getNextStepIndicator($customerId),
                    'deal_created_date' => date(DISPLAY_DATE, strtotime($deal->created_at)),
                    'deal_created_time' => date(DISPLAY_TIME, strtotime($deal->created_at)),
                    'second_party' => [
                        'second_party_id' => $secondParty->id ?? null,
                        'second_party_name' => $secondParty->full_name ?? null,
                        'second_party_image' => $secondParty && $secondParty->image
                            ? asset(CUSTOMER_PATH . '/' . $secondParty->image)
                            : null,
                    ],
                ];
            });
            $message = __('message.deals_retrieved_successfully');
            return apiResponse(200, $message, [
                'my_deals' => $data,
                'current_page' => (int) $currentPage,
                'per_page' => $perPage,
                'total_deals' => $totalDeals,
            ]);
        } catch (\exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }


    public function myDeals(Request $request)
    {
        try {
            $today = now()->format('d/m/Y');
            //Log::info('myDeals API Request Input:', $request->all());
            DB::beginTransaction();
            $validation = Validator::make($request->all(), [
                'current_page' => 'required|integer',
                'start_date' => 'nullable|date_format:d/m/Y|before_or_equal:' . $today,
                'end_date' => 'nullable|date_format:d/m/Y|after_or_equal:start_date|before_or_equal:' . $today,

            ]);

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $customerId = getLoggedInCustomerId();
            $currentPage = $request->current_page;
            $perPage = 10;
            $skip = ($currentPage - 1) * $perPage;
            if (!in_array($request->request_type, ['created', 'accepted'])) {
                $message = __('message.invalid_request');
                return apiResponse(400, $message);
            }
            $column = $request->request_type === 'created' ? 'from_customer_id' : 'to_customer_id';
            $secondPartyColumn = $request->request_type === 'created' ? 'to_customer_id' : 'from_customer_id';



            //by deal status
            if ($request->request_type == 'accepted') {
                $query = DealModel::where($column, $customerId)->where('deal_status', '!=', PENDING);
            } else {
                $query = DealModel::where($column, $customerId);
            }



            $deals = $query->get();

            // by date
            if ($request->filled('start_date')) {
                $startDate = app_date_convert_format($request->start_date, 'd/m/Y') . ' 00:00:00';
                $deals = $deals->filter(function ($deal) use ($startDate) {
                    return $deal->created_at >= $startDate;
                });
            }

            if ($request->filled('end_date')) {
                $endDate = app_date_convert_format($request->end_date, 'd/m/Y') . ' 23:59:59';
                $deals = $deals->filter(function ($deal) use ($endDate) {
                    return $deal->created_at <= $endDate;
                });
            }

            // by deal type
            if ($request->filled('deal_type') && $request->deal_type !== 'all') {
                $deals = $deals->filter(function ($deal) use ($customerId, $request) {
                    return strtolower(dealTypeViceVersa($deal, $customerId)) == strtolower($request->deal_type);
                });
            }

            // search
            if ($request->filled('search')) {
                $search = strtolower($request->search);
                $search = trim($search);
                $deals = $deals->filter(function ($deal) use ($search, $secondPartyColumn) {
                    $title = strtolower($deal->deal_title ?? '');
                    $secondParty = $deal->{$secondPartyColumn} ? CustomerModel::withTrashed()->find($deal->{$secondPartyColumn}) : null;
                    $secondPartyName = strtolower(optional($secondParty)->full_name ?? '');

                    return str_starts_with($title, $search) ||
                        str_starts_with($secondPartyName, $search);
                })->values();
            }


            $deals = $deals->sortByDesc('created_at')->values();

            if ($deals->isEmpty()) {
                DB::commit();
                return apiResponse(200, __('message.no_deals_found'));
            }
            $totalDeals = $deals->count();
            $paginatedDeals = $deals->slice($skip, $perPage)->values();
            $data = $paginatedDeals->map(function ($deal) use ($secondPartyColumn, $customerId) {
                $secondParty = CustomerModel::withTrashed()->find($deal->$secondPartyColumn);
                $dealType = dealTypeViceVersa($deal, $customerId) == 'Buy' ? __('message.buy') : __('message.sell');
                //dd($secondParty);
                return [
                    'deal_id' => $deal->id,
                    'deal_title' => $deal->deal_title,
                    'deal_amount' => $deal->deal_amount,
                    'deal_amount_in_format' => formatCurrency($deal->deal_amount, $deal->currency_code),
                    'deal_type' => $dealType,
                    'deal_next_step' => $deal->getNextStepIndicator($customerId),
                    'deal_status' => checkDisputeDeal($deal->deal_status),
                    'deal_status_lang_wise' => __('readableStatuses.' . checkDisputeDeal($deal->deal_status)),
                    'deal_created_date' => date(DISPLAY_DATE, strtotime($deal->created_at)),
                    'deal_created_time' => date(DISPLAY_TIME, strtotime($deal->created_at)),
                    'second_party' => [
                        'second_party_id' => $secondParty->id ?? null,
                        'second_party_name' => $secondParty->full_name ?? null,
                        'second_party_image' => $secondParty?->image ? asset(CUSTOMER_PATH . '/' . $secondParty->image) : null,
                    ],
                ];
            });

            $message = __('message.deals_data_retrieved');
            return apiResponse(200, $message, ['my_deals' => $data, 'current_page' => (int)$currentPage, 'per_page' => $perPage, 'total_deals' => $totalDeals]);
        } catch (\exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }

    public function dealDetail(Request $request)
    {

        try {

            DB::beginTransaction();
            $deal = DealModel::with(['images'])->where('id', $request->deal_id)->first();
            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            $readyForShipping = 0;
            $readyForShippingProof = 0;
            $isPaymentProofRejected = 0;
            $isShippingProofRejected = 0;
            $isPaymentProofAccepted = 0;
            $secondPartyInactive = 0;
            $secondPartyDeleted = 0;

            //shipping proof
            $shippingProof = ($deal->shipping_proof) ? asset(SHIPPING_PROOF_PATH . '/' . $deal->shipping_proof) : '';
            $loggedin_user_id = getLoggedInCustomerId();

            $secondPartyId = ($loggedin_user_id == $deal->from_customer_id) ? $deal->to_customer_id : $deal->from_customer_id;

            $secondParty = CustomerModel::withTrashed()->where('id', $secondPartyId)->first();

            //fetch dispute docs

            //check allowed status
            $allowedStatus = [];
            $displayDisputeLink = 0;
            $displayAmountEditLink = 0;
            $displayCancelBtn = 0;
            $displayDispluteDetail = 0;
            $documentsByBuyer = [];
            $documentsBySeller = [];
            $disputeReasonKey = null;
            $disputeReasonValue = null;
            $disputeStatus = [];
            $canDisplayPayBtn = 0;
            $lastConversation = [];
            $amountTransfer = null;
            $secondPertyDeletedConfirmation = null;
            if ($secondParty->status == INACTIVE) {
                $secondPartyInactive = 1;
            }

            if ($secondParty)

                $dealType = $deal->deal_type;
            $secondPartyAddress = '';
            $customerTypeArray = getCustomerType($deal,  $loggedin_user_id);
            $customerType = $customerTypeArray[1];
            if ($customerType == BUYER) {
                if (!$secondParty->deleted_at == null) {
                    $secondPertyDeletedConfirmation = __('message.seller_deleted');
                }

                $secondPartyAddress = '';
            } else if ($customerType == SELLER) {
                if (!$secondParty->deleted_at == null) {
                    $secondPertyDeletedConfirmation = __('message.buyer_deleted');
                }

                $secondPartyAddress = $secondParty->address;
                $countryName = DB::table('customers')
                    ->leftJoin('countries', 'customers.country_id', '=', 'countries.id')
                    ->where('customers.id', $secondPartyId)
                    ->select('countries.name as country_name')
                    ->value('country_name');

                $stateName = DB::table('customers')
                    ->leftJoin('states', 'customers.state_id', '=', 'states.id')
                    ->where('customers.id', $secondPartyId)
                    ->select('states.name as state_name')
                    ->value('state_name');

                $cityName = DB::table('customers')
                    ->leftJoin('cities', 'customers.city_id', '=', 'cities.id')
                    ->where('customers.id', $secondPartyId)
                    ->select('cities.name as city_name')
                    ->value('city_name');
            }
            $dealType = dealTypeViceVersa($deal, $loggedin_user_id);
            if ($deal->is_paid == 0 && $deal->deal_status == ACCEPTED && $deal->is_hold == 0) {
                if ($customerType == BUYER) {
                    $canDisplayPayBtn = 1;
                }
            }
            //payment and shopping proof status check with when cancel not status cause create problem at app side to show message in cancel case
            // if ($deal->payment_status == 'rejected' && ($deal->deal_status !== CANCEL || $deal->end_reason !== CANCEL)) {
            if ($deal->payment_status == 'rejected') {
                $isPaymentProofRejected = 1;
            }
            //if payment accepted then not display the upload payment proof icon from app side
            if ($deal->payment_status == ACCEPTED) {
                $isPaymentProofAccepted = 1;
            }

            if ($deal->shipping_status == 'rejected') {
                $isShippingProofRejected = 1;
            }
            //not display image to buyer till shipping proof not accepted
            if ($customerType == BUYER && in_array($deal->shipping_status, [PENDING, REJECTED])) {
                $shippingProof = null;
            }

            if ($deal->is_paid == 1 && $deal->is_hold == 1 && $deal->is_release == 0) {
                if ($deal->deal_type == BUY) {
                    $customerIdentity =   $loggedin_user_id == $deal->from_customer_id ? BUYER : SELLER;
                } else {
                    $customerIdentity =   $loggedin_user_id == $deal->from_customer_id ? SELLER : BUYER;
                }
                switch ($deal->deal_status) {
                    case ACCEPTED:
                        if ($customerIdentity == SELLER) {
                            $allowedStatus = [
                                ['key' => TRANSIT, 'value' => 'Shipped'],

                            ];
                        }

                        break;

                    case TRANSIT:
                        if ($customerIdentity == SELLER) {
                            $allowedStatus = [
                                ['key' => DELIVERED, 'value' => 'Delivered'],

                            ];
                        }


                        break;
                    case DELIVERED:
                        if ($customerIdentity == BUYER) {
                            $allowedStatus = [
                                ['key' => COMPLETED, 'value' => 'Completed'],

                            ];
                        }


                        break;
                }
            } else {
                if ($deal->from_customer_id == $loggedin_user_id && in_array($deal->deal_status, [PENDING])) {
                    $displayAmountEditLink = 1;
                }
            }

            //display dispute link
            $cutomerType = getCustomerType($deal,  $loggedin_user_id);
            if ($cutomerType[1] == BUYER && in_array($deal->deal_status, [ACCEPTED, TRANSIT, DELIVERED]) && $deal->is_paid == 1) {

                $displayDisputeLink = 1;
            }
            //display cancel btn
            if (in_array($deal->deal_status, [ACCEPTED])) {
                if ($deal->shipping_proof == null) {
                    $displayCancelBtn = 1;
                } elseif ($deal->shipping_proof && in_array($deal->shipping_status, ['pending', 'accepted'])) {
                    $displayCancelBtn = 0;
                } elseif ($deal->shipping_proof && in_array($deal->shipping_status, ['rejected'])) {
                    $displayCancelBtn = 1;
                } else {
                    $displayCancelBtn = 0;
                }
            }
            //shipping and shipping proof update checks
            if ($cutomerType[1] == SELLER) {
                // shipping proof check
                if (($deal->shipping_proof == null || $deal->shipping_status == 'rejected') && $deal->deal_status == ACCEPTED) {
                    $readyForShippingProof = 1;
                }
                //shipping updation check
                if ($deal->shipping_proof != null && $deal->shipping_status == ACCEPTED) {
                    $readyForShipping = 1;
                }
            }

            $disputed_status =  disputed_status();
            //to display the dispute detail block
            if (in_array($deal->deal_status, $disputed_status) || in_array($deal->end_reason, $disputed_status)) {
                $displayDispluteDetail = 1;
            }
            $canEditDispute = ($deal->deal_status === ON_DISPUTE) ? 1 : 0;

            if ($displayDispluteDetail) {
                //now get disputed docs from buyer and seller side
                if ($deal->deal_type == BUY) {

                    $buyerId = $deal->from_customer_id;
                    $sellerId = $deal->to_customer_id;
                } else { // SELL

                    $buyerId = $deal->to_customer_id;
                    $sellerId = $deal->from_customer_id;
                }
                //dd($buyerId);
                //dd($sellerId);
                $documentsByBuyer = DB::table('dispute_deal_documents')
                    ->where('deal_id', $deal->id)
                    ->where('customer_id', $buyerId)
                    ->get()
                    ->map(function ($doc) {
                        return [
                            'doc_id' => $doc->id,
                            'doc_url' => asset(DISPUTED_DOCS_PATH . '/' . $doc->file_path),
                            'doc_extension' => $doc->file_extension,
                            'doc_type' => $doc->file_type
                        ];
                    });

                $documentsBySeller = DB::table('dispute_deal_documents')
                    ->where('deal_id', $deal->id)
                    ->where('customer_id', $sellerId)
                    ->get()
                    ->map(function ($doc) {
                        return [
                            'doc_id' => $doc->id,
                            'doc_url' => asset(DISPUTED_DOCS_PATH . '/' . $doc->file_path),
                            'doc_extension' => $doc->file_extension,
                            'doc_type' => $doc->file_type
                        ];
                    });


                $disputeReasonKey = null;
                $disputeReasonValue = null;
                $disputeByAdminReason = '';
                $deal->dispute_request_by_type == ADMIN ? 1 : 0;
                if ($deal->dispute_request_by_type == ADMIN) {
                    $disputeByAdmin = 1;
                    $disputeReasonValue = $deal->dispute_reason;
                } else {
                    $disputeByAdmin = 0;
                    $disputeReasonKey = $deal->dispute_reason_id ?? '';
                    if ($disputeReasonKey) {
                        $selectedReason = collect(disputeReasons())->firstWhere('key', $disputeReasonKey);
                        $disputeReasonValue = $selectedReason['value'] ?? $deal->dispute_reason;
                    }

                    //other reason
                    if ($deal->dispute_reason_id == 5) {
                        $disputeReasonValue = $deal->dispute_reason;
                    }
                }
                $disputeStatus = disputeStatusCustomText($deal->deal_status, $dealType) ?? '';
                if ($deal->is_release == 1) {
                    $disputeStatus = disputeStatusCustomText($deal->end_reason, $dealType) ?? '';
                }

                $lastConversation = disputeDealConversationModel::where('deal_id', $deal->id)
                    ->where('receiver_type', CUSTOMER)
                    ->where('sender_type', ADMIN)
                    ->where('receiver_id', $loggedin_user_id)
                    ->orderBy('created_at', 'desc')->first();

                //amount release time statuses :
                if (in_array($deal->end_reason, [DISPUTE_ACCEPTED, DISPUTE_REJECTED]) && $deal->is_release == 1) {
                    $amountTransfer =    disputeAmountTransfer($deal->deal_status, $customerType);
                }
            }
            //cancel status
            $cancelDetail = 0;
            if (in_array($deal->deal_status, [CANCEL]) || in_array($deal->end_reason, [CANCEL])) {
                $cancelDetail = 1;
                $cancelBy = null;
                $releasedToType = null;
                if ($deal->cancel_type == CUSTOMER) {
                    $customerTypeArray = getCustomerType($deal,  $deal->cancel_by);
                    $cancelBy = $customerTypeArray[1];
                } else {
                    $cancelBy = 'Admin';
                }
                if ($deal->is_release == 1) {
                    $customerTypeArray = getCustomerType($deal, $deal->amount_released_to);
                    $releasedToType = $customerTypeArray[1];
                }
            }



            //bank details
            $bankDetails = (new settingModel())->getSetting(['bank', 'card_number']);
            //dd($bank);
            $bank = $bankDetails->bank ?? '';
            $cardNumber = $bankDetails->card_number ?? '';
            //dd($bank, $cardNumber);
            $dealImages = [];
            if ($deal->images) {
                $dealImages = $deal->images?->map(function ($image) {
                    return [
                        'image_url' => asset(DEAL_IMAGES . '/' . $image->image)
                    ];
                }) ?? collect();
            }

            DB::commit();

            $dealTypeLangWise = dealTypeViceVersa($deal, $loggedin_user_id) == 'Buy' ? __('message.buy') : __('message.sell');
            $deal_data = [
                'deal_id' => $deal->id,
                'deal_title' => $deal->deal_title,
                'deal_number' => $deal->deal_number,
                'deal_amount' => $deal->deal_amount,
                'currency_code' => $deal->currency_code,
                'deal_amount_in_format' => formatCurrency($deal->deal_amount, $deal->currency_code),
                'total_deal_amount' => $deal->total_deal_Amount,
                'total_deal_amount_in_format' =>  formatCurrency($deal->total_deal_Amount, $deal->currency_code),
                'commission' => $deal->transfer_fees_in_percent,
                'commission_amount' => $deal->admin_earnings,
                'commission_amount_in_format' => formatCurrency($deal->admin_earnings, $deal->currency_code),
                'deal_images' => $dealImages,
                'payment_proof_image' => ($deal->payment_proof) ? asset(PAYMENT_PROOF_PATH . '/' . $deal->payment_proof) : '',
                'shipping_proof_image' => $shippingProof,
                'deal_type' => $dealTypeLangWise,
                'deal_status' => checkDisputeDeal($deal->deal_status),
                'deal_status_lang_wise' => __('readableStatuses.' . checkDisputeDeal($deal->deal_status)),
                'from_customer_id' => $deal->from_customer_id,
                'is_paid' => $deal->is_paid,
                'can_display_pay_btn' => $canDisplayPayBtn,
                'deal_created_date' => date(DISPLAY_DATE, strtotime($deal->created_at)),
                'deal_created_time' => date(DISPLAY_TIME, strtotime($deal->created_at)),
                //'allowed_statuses' =>  $allowedStatus,
                'display_dispute_link' => $displayDisputeLink,
                'display_amount_edit_link' => $displayAmountEditLink,
                'display_cancel_btn' => $displayCancelBtn,
                'is_shipping_proof_rejected' => $isShippingProofRejected,
                'is_payment_proof_rejected' => $isPaymentProofRejected,
                'is_payment_proof_accepted' => $isPaymentProofAccepted,
                'ready_for_shipping_proof' => $readyForShippingProof,
                'ready_for_shipping_status' => $readyForShipping,
                'second_party' => [
                    'second_party_inactive' => $secondPartyInactive,
                    'second_party_deleted_message' => $secondPertyDeletedConfirmation,

                    'second_party_id' => $secondParty->id ?? null,
                    'second_party_name' => $secondParty->full_name ?? 'Unknown',
                    'good_deals_count' => $secondParty->successful_deals_count ?? 0,
                    'rank' => $secondParty ? $secondParty->getGoodDealsRank($secondParty->id) : null,
                    'second_party_image' => $secondParty && $secondParty->image ? asset(CUSTOMER_PATH . '/' . $secondParty->image) : null,
                    'second_party_mobile_number' => $secondParty->mobile_number ?? null,
                    'second_party_dial_code' => $secondParty->dial_code ?? null,
                    'second_party_dial_code_iso' => $secondParty->dial_code_iso ?? null,
                    'second_party_address' => $secondPartyAddress,
                    'second_party_country' => $countryName ?? null,
                    'second_party_state' => $stateName ?? null,
                    'second_party_city' => $cityName ?? null,


                ],
                'bank_details' => [
                    'bank' => $bank,
                    'card_number' => formatCardNumber($cardNumber),
                ],
            ];
            //dispute info rule
            if ($displayDispluteDetail) {
                $deal_data['dispute_information'] =
                    [
                        'display_displute_detail' => $displayDispluteDetail,
                        'can_edit_dispute' => $canEditDispute,
                        'disputeByAdmin' => $disputeByAdmin,
                        'disputeByAdminReason' => $disputeByAdminReason,
                        'dispute_reason_key' => $disputeReasonKey,
                        'dispute_reason_value' => $disputeReasonValue,
                        'dispute_status' => $disputeStatus,
                        'dispute_docs_by_buyer' => $documentsByBuyer,
                        'dispute_docs_by_seller' => $documentsBySeller,
                        'dispute_last_message' => $lastConversation->message ?? null,
                        'dispute_last_message_sender_type' => $lastConversation->sender_type ?? null,
                        'dispute_last_message_date' => $lastConversation ? date('M d, Y', strtotime($lastConversation->created_at)) : null,
                        'dispute_last_message_time' => $lastConversation ? date('h:i A', strtotime($lastConversation->created_at)) : null,
                        'amount_transfer' => $amountTransfer

                    ];
            }
            if ($cancelDetail) {
                $deal_data['cancel_information'] =
                    [
                        'title' => CANCEL,
                        'cancel_by_admin' => $deal->cancel_type == ADMIN ? 1 : 0,
                        'cancel_reason' => $deal->cancel_reason,
                        'cancel_by' => $cancelBy,
                        'is_deal_paid' => $deal->is_paid == 1 ? 1 : 0,
                        'is_released' => $deal->is_release == 1 ? 1 : 0,
                        'released_to_id' => $deal->amount_released_to ?? null,
                        'released_to' => $releasedToType,
                    ];
            }



            $message = __('message.deal_detail_retrieved');
            return apiResponse(200, $message, ['deal_data' => $deal_data]);
        } catch (\exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }

    //get request deals list (pending one)
    public function requestsDeals(Request $request)

    {
        try {
            $validation = Validator::make($request->all(), [
                'current_page' => 'required|integer',

            ]);

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            $currentPage = $request->current_page;
            $perPage = 10;
            $skip = ($currentPage - 1) * $perPage;
            $customerId = getLoggedInCustomerId();
            $query = DealModel::where('to_customer_id', $customerId)
                ->where('deal_status', PENDING);

            //filter
            if ($request->filled('deal_type') && $request->deal_type !== 'all' && in_array($request->deal_type, [SELL, BUY])) {
                $query->where(function ($q) use ($request, $customerId) {
                    // For SELL filter, logged-in customer is buyer (to_customer_id) in a sell deal
                    if ($request->deal_type === SELL) {
                        $q->where('deal_type', BUY);
                    }
                    // For BUY filter, logged-in customer is seller (to_customer_id) in a buy deal
                    elseif ($request->deal_type === BUY) {
                        $q->where('deal_type', SELL);
                    }
                });
            }

            $deals = $query->orderBy('created_at', 'desc')->get();
            if ($deals->isEmpty()) {
                $message = __('message.no_request_found');
                return apiResponse(200, $message);
            }
            $totalDeals = $deals->count();
            $paginatedDeals = $deals->slice($skip, $perPage)->values();
            $data = $paginatedDeals->map(function ($deal) use ($customerId) {
                $sender = CustomerModel::withTrashed()->find($deal->from_customer_id);
                $dealTypeLangWise = dealTypeViceVersa($deal, $customerId) == 'Buy' ? __('message.buy') : __('message.sell');
                $dealImageUpload = 0;
                if ($deal->deal_type == BUY && $deal->images->count() == 0) {
                    $dealImageUpload = 1;
                }
                return [
                    'deal_id' => $deal->id,
                    'deal_amount' => $deal->deal_amount,
                    'deal_amount_in_format' => formatCurrency($deal->deal_amount, $deal->currency_code),
                    'deal_title' => $deal->deal_title,
                    'deal_number' => $deal->deal_number,
                    'deal_type' => $dealTypeLangWise,
                    'is_image_upload' => $dealImageUpload,
                    'sender' => [
                        'sender_id' => $sender->id ?? null,
                        'sender_full_name' => $sender->full_name ?? 'unknown',
                        'sender_image' => $sender ? asset(CUSTOMER_PATH . '/' . $sender->image) : null,
                        'good_deals_count' => $sender->successful_deals_count ?? 0,
                        'rank' => $sender ? $sender->getGoodDealsRank($sender->id) : null
                    ],
                ];
            });
            $message = __('message.pending_deals_retrieved');
            return apiResponse(200, $message, [
                'requests_deals' => $data,
                'current_page' => (int) $currentPage,
                'per_page' => $perPage,
                'total_deals' => $totalDeals,
            ]);
        } catch (\exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }
    public function requestAction(Request $request)
    {
        try {
            DB::beginTransaction();
            $customerId = getLoggedInCustomerId();
            $customerFullName = CustomerAuth::user()->full_name;
            $customerImage = CustomerAuth::user()->image;
            $customerStatus = CustomerAuth::user()->status;



            //get validation
            $validation = Validator::make($request->all(), [
                'deal_id' => 'required|exists:deals,id',
                'status' => 'required|in:accepted,declined'
            ]);

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            //logged in customer inactive
            if ($customerStatus == INACTIVE) {
                $message = __('message.you_are_inaactive_contact');
                $message_description = __('message.inactive_contact_administrator');
                return apiResponse(400, $message, [], '', $message_description);
            }

            //get deal
            $deal = DealModel::where('id', $request->deal_id)
                ->where('to_customer_id', $customerId)
                ->where('deal_status', PENDING)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            // find buyer and seller
            if ($deal->deal_type == BUY) {
                $fromType = __('message.buyer');
            } elseif ($deal->deal_type == SELL) {
                $fromType = __('message.seller');
            }
            if (!empty($deal->fromCustomer) && $deal->fromCustomer->deleted_at) {
                $message = __('message.deal_unavailable_user_deleted', ['fromType' => $fromType]);
                return apiResponse(400, $message);
            } /*elseif (!empty($deal->fromCustomer) && $deal->fromCustomer->status == INACTIVE) {
                $message = __('message.deal_unavailable_user_inactive', ['fromType' => $fromType]);
                return apiResponse(400, $message);
            }*/
            if ($request->status == 'accepted') {
                $statusEn = 'accepted';
                $statusEs = 'Aceptado';
            } elseif ($request->status == 'declined') {
                $statusEn = 'declined';
                $statusEs = 'Rechazado ';
            }
            //update the status
            $fromCustomerId = $deal->from_customer_id;
            $toCustomerId = $deal->to_customer_id;
            $deal->deal_status = $request->status;
            $deal->save();

            $nPayNowMessage = '';
            $nPayNowMessageEs = '';
            if (getCustomerType($deal, $customerId) == SELLER) {
                $nPayNowMessage = ($request->status === ACCEPTED) ? " You can pay now" : "";
                $nPayNowMessageEs = ($request->status === ACCEPTED) ? " Puedes pagar ahora" : "";
            }

            //save deal stataus history
            DealstatusHistoryModel::saveHistory($deal->id, $deal->deal_status);

            //notification implementation
            $to_cusmomer_name = $customerFullName;
            $nDealId = $deal->id;
            $nType = 'deal_request_action';
            $nTitle = "Deal {$statusEn}";
            $nTitleEs = "Trato {$statusEs}";

            //$nMesage = "Your deal #{$deal->deal_number} request to {$to_cusmomer_name} has been {$request->status}." . $nPayNowMessage;
            $nMesage = "Your deal #{$deal->deal_number} request to {$to_cusmomer_name} has been {$statusEn}." . $nPayNowMessage;
            $nMesageEs = "Tu solicitud de trato #{$deal->deal_number} a {$to_cusmomer_name} ha sido {$statusEs}." . $nPayNowMessageEs;
            saveNotification($fromCustomerId, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $toCustomerId, $deal->deal_number, $nDealId, 0);



            // function saveNotification($customerId, $type, $title, $message,$deal->deal_number,$del->id
            //  $secondPartyId = null, $deal_number = null, $deal_id = null)

            DB::commit();

            $message = __('message.deal_status_updated', ['status' => $request->status]);
            return apiResponse(200, $message);
        } catch (\exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }

    //edit deal amount
    public function editDealAmount(request $request)
    {
        try {

            DB::beginTransaction();
            $customerId = getLoggedInCustomerId();
            //get validation
            $validation = Validator::make($request->all(), [
                'deal_id' => 'required|exists:deals,id',
                'deal_amount' => 'required|numeric|min:1'

            ]);

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $deal = DealModel::with(['fromCustomer', 'toCustomer'])->where('id', $request->deal_id)->first();
            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            //only pending deals should have this right

            if ($deal->deal_status != PENDING || $deal->from_customer_id !=  $customerId) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            $requestedDealAmount = $request->deal_amount;
            $transferFees = $deal->transfer_fees_in_percent;
            $admin_earnings = ($requestedDealAmount * $transferFees) / 100;
            $totalDealAmount = $requestedDealAmount + $admin_earnings;

            $deal->update([
                'deal_amount' =>  $requestedDealAmount,
                'deal_amount_in_format' => formatCurrency($requestedDealAmount, $deal->currency_code),
                'admin_earnings' => $admin_earnings,
                'deal_amount_in_format' => formatCurrency($admin_earnings, $deal->currency_code),
                'total_deal_Amount' => $totalDealAmount,
                'deal_amount_in_format' => formatCurrency($totalDealAmount, $deal->currency_code),

            ]);
            //notification for send request deal api to second party
            $from_cusmomer_name = $deal->fromCustomer->full_name;
            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            $nType = 'deal_amount_edit';
            $nTitle = "Deal amount edited";
            $nTitleEs = 'Monto del trato editado';
            $nMesage = "{$from_cusmomer_name} edited the amount for the deal #{$nDealnumber}";
            $nMesageEs = "{$from_cusmomer_name} editó el monto del trato #{$nDealnumber}";
            saveNotification($deal->toCustomer->id, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $deal->fromCustomer->id, $nDealnumber, $dealId, 0);
            DB::commit();
            //  $transferFees = settingModel::where('setting_name', 'transfer_fees')->value('description') ?? 0;
            // $admin_earnings = ($request->deal_amount * $transferFees) / 100;
            // $totalDealAmount = $request->deal_amount + $admin_earnings;
            $message = __('message.deal_amount_edited');
            return apiResponse(200, $message);
        } catch (\exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }
    //display notifications
    public function getNotifications(Request $request)
    {

        try {
            $language = CustomerAuth::user()->language_code;

            $customerId = getLoggedInCustomerId();
            $currentPage = $request->current_page;
            $perPage = 10;
            //$currentPage = "";
            $skip = ($currentPage - 1) * $perPage;
            $baseQuery = NotificationModel::where('customer_id', $customerId);


            $totalNotifications = (clone $baseQuery)->count();
            $unreadNotifications = (clone $baseQuery)->where('is_read', 0)->count();
            $notifications = (clone $baseQuery)
                ->skip($skip)
                ->take($perPage)
                ->orderBy('created_at', 'desc')
                ->get();


            DB::commit();

            $data =
                $notifications->map(function ($notification) use ($language, $customerId) {

                    $deal = DealModel::where('id', $notification->deal_id)->first();
                    if ($notification->is_updation_by_admin) {
                        $secondParty = CustomerModel::where('id', $notification->second_party_id)->first();
                        //dd($secondParty);
                        $secondParty = [
                            'second_party_id' => $secondParty->id ?? null,
                            'second_party_full_name' => $secondParty->full_name ?? 'Unknown',
                            'second_party_image' => $secondParty ? asset(ADMIN_PATH . '/' . $secondParty->image) : null,
                        ];
                    } else {
                        $secondParty = CustomerModel::where('id', $notification->second_party_id)->first();
                        //dd($secondParty);
                        $secondParty = [
                            'second_party_id' => $secondParty->id ?? null,
                            'second_party_full_name' => $secondParty->full_name ?? 'Unknown',
                            'second_party_image' => $secondParty ? asset(CUSTOMER_PATH . '/' . $secondParty->image) : null,
                        ];
                    }
                    $message = $language === 'es' ? $notification->message_es : $notification->message;
                    $title = $language === 'es' ? $notification->title_es : $notification->title;
                    return [
                        'notification_id' => $notification->id,
                        'title' => $title,
                        'message' => $message,
                        'deal_type' => $deal ? (dealTypeViceVersa($deal, $customerId) == 'Buy' ? __('message.buy') : __('message.sell')) : null,
                        'type' => $notification->type,
                        'deal_number' => $notification->deal_number,
                        'is_read' => $notification->is_read,
                        'second_party' => $secondParty,
                        'deal_info' => $deal ? [
                            'deal_id' => $deal->id,
                            'deal_amount' => $deal->deal_amount,
                            'deal_amount_in_format' => formatCurrency($deal->deal_amount, $deal->currency_code),
                        ] :
                            [
                                'deal_id' => null,
                                'deal_amount' => null,
                                'deal_amount_in_format' => null,
                            ],
                    ];
                });


            $message = __('message.notifications_retrieved');
            return apiResponse(200, $message, ['notifications' => $data, 'per_page' => $perPage, 'current_page' => $currentPage, 'total_notification_count' => $totalNotifications, 'unread_notification_count' => $unreadNotifications]);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }

    public function  markNotificationAsRead(Request $request)
    {

        try {
            DB::beginTransaction();
            $customerId =  getLoggedInCustomerId();

            $validation = Validator::make($request->all(), [
                'notification_id' => 'required|integer|exists:notifications,id',
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            $notification = NotificationModel::where('id', $request->notification_id)
                ->where('customer_id', $customerId)->first();

            if (!$notification) {
                $message = __('message.notification_not_found');
                return apiResponse(400, $message);
            }

            $notification->is_read = 1;
            $notification->save();

            DB::commit();

            $message = __('message.notification_marked_read');
            return apiResponse(200, $message, ['data' => $notification->is_read]);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //delete notification
    public function deleteNotification(Request $request)
    {
        try {
            DB::beginTransaction();

            $customerId = getLoggedInCustomerId();

            $validation = Validator::make($request->all(), [
                'notification_id' => 'required|integer|exists:notifications,id',
            ]);

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            $notification = NotificationModel::where('id', $request->notification_id)
                ->where('customer_id', $customerId)
                ->first();

            if (!$notification) {
                $message = __('message.notification_not_found');
                return apiResponse(400, $message);
            }

            $notification->delete();

            DB::commit();

            $message = __('message.notification_deleted_successfully');
            return apiResponse(200, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //update status 
    public function updateDealStatus(Request $request)
    {
        try {
            DB::beginTransaction();
            $deal = DealModel::where('id', $request->deal_id)->first();
            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            $formattedAmount = number_format($deal->deal_amount);
            $cancelWithPayment = true;
            $customerId = getLoggedInCustomerId();
            //$customerColumn = getCustomerColumn($deal, $customerId);
            $nType = '';
            $nTitle = '';
            $nTitleEs = '';
            $nMesage = '';
            $nMesageEs = '';
            $receiverId = '';
            $secondPartyId = '';

            // if (!$customerColumn) {
            //     $message = "Unauthorized Action";
            //     return apiResponse(400, $message);
            // }

            if ($deal->from_customer_id == $customerId) {
                // You are the one who created the deal
                $receiverId = $deal->to_customer_id;
                $secondPartyId = $deal->to_customer_id;
                $secondParty = CustomerModel::withTrashed()->find($secondPartyId);
            } elseif ($deal->to_customer_id == $customerId) {
                // You are the one who accepted the deal
                $receiverId = $deal->from_customer_id;
                $secondPartyId = $deal->from_customer_id;
                $secondParty = CustomerModel::withTrashed()->find($secondPartyId);
            } else {
                return apiResponse(400, __('message.unauthorized_action'));
            }


            $rules = [
                'deal_id' => 'required|integer|exists:deals,id',
                'deal_status' => 'required|string|in:' . implode(',', dealSatusValuesForValidateUpdation()),
            ];
            if ($request->deal_status === CANCEL) {
                $rules['cancel_reason'] = 'required|string';
            }

            $validation = Validator::make($request->all(), $rules);

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $buyer = $customerDetails[BUYER] ?? null;
            $seller = $customerDetails[SELLER] ?? null;

            //check secong party delted or inactive
            if ($secondParty) {
                if ($buyer && $secondParty->id == $buyer->id) {
                    $secondPartyType = __('message.buyer');
                } elseif ($seller &&  $secondParty->id == $seller->id) {
                    $secondPartyType = __('message.seller');
                }
                if ($secondParty->deleted_at) {
                    $message = __('message.deal_unavailable_user_deleted', ['fromType' => $secondPartyType]);
                    return apiResponse(400, $message);
                } /*elseif ($secondParty->status == INACTIVE) {
                    $message = __('message.deal_unavailable_user_inactive', ['fromType' => $secondPartyType]);
                    return apiResponse(400, $message);
                }*/
            } else {
                $message = __('message.deal_unavailable_user_deleted', ['fromType' => __('message.buyer')]);
                return apiResponse(400, $message);
            }

            //$deal->deal_status = $request->deal_status;
            if ($request->deal_status === COMPLETED) {
                if ($deal->deal_status !== DELIVERED) {
                    $message = __('message.unauthorized_action');
                    return apiResponse(400, $message);
                }
                if ($buyer->id != $customerId) {
                    $message = __('message.unauthorized_action');
                    return apiResponse(400, $message);
                }

                $deal->completed_by_type = CUSTOMER;
                $deal->completed_by_id = getLoggedInCustomerId();
                $nType = 'completed_deal';
                $nTitle = 'Deal Completed Successfully';
                $nTitleEs = 'Trato completado con éxito';
                $nMesage = "Your deal #{$deal->deal_number} with {$secondParty->full_name} has been successfully completed. Deal amount \${$formattedAmount} will be released soon.";
                $nMesageEs = "Su trato #{$deal->deal_number} con {$secondParty->full_name} se ha completado con éxito. El monto del trato de \${$formattedAmount} será liberado pronto.";
            }
            //cancel request
            if ($request->deal_status === CANCEL) {

                if (!in_array($deal->deal_status, [ACCEPTED])) {
                    $message = __('message.unauthorized_action');
                    return apiResponse(400, $message);
                }
                if ($deal->deal_status === ACCEPTED && in_array($deal->shipping_status, [PENDING, ACCEPTED]) && $deal->shipping_proof != null) {
                    $message = __('message.unauthorized_action');
                    return apiResponse(400, $message);
                }
                $deal->cancel_by = $customerId;
                $deal->cancel_reason = $request->cancel_reason;
                $deal->cancel_type = CUSTOMER;
                $deal->is_read = 0;

                if ($deal->is_hold == 0 && $deal->is_paid == 0) {
                    $deal->end_reason = CANCEL;
                    $deal->deal_status = CLOSED;
                    $cancelWithPayment  = false;
                }

                $nType = 'cancelled_deal';
                $nTitle = 'Deal Cancelled';
                $nTitleEs = 'Trato cancelado';
                $nMesage = "The deal #{$deal->deal_number} with {$secondParty->full_name} has been cancelled.";
                $nMesageEs = "El trato #{$deal->deal_number} con {$secondParty->full_name} ha sido cancelado.";
            }

            //transit/shipped request
            if ($request->deal_status === TRANSIT) {
                if ($deal->deal_status !== ACCEPTED || !$deal->shipping_proof || $seller->id != $customerId) {
                    return apiResponse(400, __('message.unauthorized_action'));
                }
                $nType = 'shipped_item';
                $nTitle = 'Your Order has been shipped';
                $nTitleEs = 'Su pedido ha sido enviado';
                $nMesage = "The Deal #{$deal->deal_number} with {$secondParty->full_name} has been shipped.";
                $nMesageEs = "El trato #{$deal->deal_number} con {$secondParty->full_name} ha sido enviado.";
            }

            //delivered
            if ($request->deal_status === DELIVERED) {
                if ($deal->deal_status !== TRANSIT) {
                    $message = __('message.unauthorized_action');
                    return apiResponse(400, $message);
                }
                if ($seller->id != $customerId) {
                    $message = __('message.unauthorized_action');;
                    return apiResponse(400, $message);
                }
                $nType = 'delivered_item';
                $nTitle = 'Your Order has been Delivered';
                $nTitleEs = 'Su pedido ha sido entregado';
                $nMesage = "The item for your deal #{$deal->deal_number} have been successfully delivered.";
                $nMesageEs = "El artículo de tu trato #{$deal->deal_number} ha sido entregado con éxito.";
            }
            //check while is_paid 0 only cancel can perfom
            if ($deal->is_paid == 0 && $request->deal_status !== CANCEL) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            if ($cancelWithPayment) {
                $deal->deal_status = $request->deal_status;
            }

            $deal->save();
            $deal->refresh();

            //notification to second party

            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            $nType = $nType;
            $nTitle = $nTitle;
            $nTitleEs = $nTitleEs;
            $nMesage = $nMesage;
            $nMesageEs = $nMesageEs;

            if ($secondParty && !$secondParty->deleted_at && $secondParty->status == ACTIVE) {
                saveNotification($receiverId, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $secondPartyId, $nDealnumber, $dealId, 0);
            }


            //save deal stataus history
            DealstatusHistoryModel::saveHistory($deal->id, $request->deal_status);

            //check good deal bad deals
            processDealsOutcome($deal);

            DB::commit();
            $message = __('message.deal_status_updated_string');
            return apiResponse(200, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    public function getDisputeReasons(Request $request)
    {
        $customerId = getLoggedInCustomerId();
        $deal = DealModel::where('id', $request->deal_id)->first();
        //get column for search
        if (!$deal) {
            $message = __('message.deal_not_found');
            return apiResponse(400, $message);
        }

        $customerColumn = getCustomerColumn($deal, $customerId);

        if (!$customerColumn) {
            $message = __('message.unauthorized_action');
            return apiResponse(400, $message);
        }


        $validation = Validator::make($request->all(), [
            'deal_id' => 'required|integer|exists:deals,id',

        ]);
        if ($validation->fails()) {
            return apiResponse(400, $validation->errors()->first());
        }
        $reasons = disputeReasons();
        return apiResponse(200, __('message.success'), $reasons);
    }
    public function submitDispute(Request $request)
    {
        try {
            DB::beginTransaction();
            $customerId = getLoggedInCustomerId();
            $deal = DealModel::where('id', $request->deal_id)->first();


            //get column for search
            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            if ($deal->isSecondPartyCustomerDeleted($customerId)) {
                $message = __('message.seller_deleted');
                return apiResponse(400, $message);
            }
            //submit dispute check
            if (!in_array($deal->deal_status, [TRANSIT, DELIVERED])) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }

            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $buyer = $customerDetails[BUYER];
            $seller = $customerDetails[SELLER];



            if ($buyer->id != $customerId) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }


            if ($deal->is_paid == 0) {
                $message =  __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            $allowedExtensions = implode(',', array_merge(imageExtensions(), docExtensions()));
            $rules = [
                'deal_id' => 'required|exists:deals,id',
                'reason_id' => 'required|numeric|min:1',
                'documents' => 'required|array|max:5',
                'documents.*' => "file|mimes:$allowedExtensions|max:5120",
                // 'documents.*' => "file|mimes:$allowedExtensions|max:5120",
            ];
            $reasonValue = '';
            if ($request->reason_id == 5) {
                $rules['other_reason'] = 'required|string';
                $rules['reason_id'] = 'nullable';
                $reasonValue = $request->other_reason;
            }
            $validation = Validator::make($request->all(), $rules, __('validation.custom'));


            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            //other reason selected validation

            $deal->deal_status = ON_DISPUTE;
            $deal->dispute_request_by_type = CUSTOMER;
            $deal->dispute_request_by_id = $customerId;
            $deal->is_read = 0;
            $deal->dispute_reason = $reasonValue;
            $deal->dispute_reason_id = $request->reason_id;
            $deal->dispute_request_date = now();
            $deal->save();


            //upload docs

            $filesName = uploadmultifiles('documents', DISPUTED_DOCS_PATH . '/');
            foreach ($filesName as $file) {
                $fileExtension = pathinfo($file, PATHINFO_EXTENSION);
                $fileType = multimediaType($fileExtension);


                // if (in_array($fileExtension, videosExtensions())) {
                //     $videoPoster = generateVideoThumbnail($file, DISPUTED_DOCS_PATH);
                // }

                disputDdealDocumentModel::create([
                    'deal_id' => $deal->id,
                    'customer_id' =>  $customerId,
                    'file_path' => $file,
                    'file_extension' => $fileExtension,
                    'file_type' => $fileType,

                ]);
            }

            //notification for seller

            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            $nType = 'disputed_deal';
            $nTitle = "Dispute Raised by Buyer";
            $nTitleEs = 'Disputa planteada por el comprador';
            $nMesage = "{$buyer->full_name} has raised a dispute on your deal #{$nDealnumber}. Please check this deal for further action.";
            $nMesageEs = "{$buyer->full_name} ha abierto una disputa en tu trato #{$nDealnumber}. Por favor revisa este trato para tomar las acciones correspondientes.";
            saveNotification($seller->id, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $buyer->id, $nDealnumber, $dealId, 0);
            //save deal stataus history
            DealstatusHistoryModel::saveHistory($deal->id, $deal->deal_status);

            DB::commit();

            $message = __('message.dispute_submitted');
            $message_description = __('message.dispute_submitted_detail', ['deal_number' => $deal->deal_number]);
            return apiResponse(200, $message, [], false, $message_description);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    public function editDispute(Request $request)
    {

        try {
            DB::beginTransaction();
            $customerId = getLoggedInCustomerId();
            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            if ($deal->isSecondPartyCustomerDeleted($customerId)) {
                $message = __('message.seller_deleted');
                return apiResponse(400, $message);
            }
            //deal edit dispute check
            if (!in_array($deal->deal_status, [ON_DISPUTE])) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            /*$customerColumn = getCustomerColumn($deal, $customerId);

            if (!$customerColumn) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }*/

            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $buyer = $customerDetails[BUYER] ?? null;
            $seller = $customerDetails[SELLER] ?? null;
            if ($buyer->id !==  $customerId) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }

            $rules = [
                'reason_id' => 'nullable',
                'documents' => 'nullable|array|max:5',
                'documents.*' => 'nullable|file|mimetypes:image/jpg,image/jpeg,image/png,image/webp,video/mp4,video/mov,video/avi|max:5120',
                // 'documents.*' => 'nullable|file|mimetypes:image/jpg,image/jpeg,image/png,image/webp,video/mp4,video/mov,video/avi|max:5120',
                'deleted_doc_id' => 'nullable|array',
                'deleted_doc_id.*' => [
                    'nullable',
                    'integer',
                    Rule::exists('dispute_deal_documents', 'id')
                        ->where(function ($query) use ($deal, $customerId) {
                            $query->where('deal_id', $deal->id)
                                ->where('customer_id', $customerId);
                        }),
                ],
            ];

            // Only update reason_id if present in request
            if ($request->has('reason_id')) {
                $deal->dispute_reason_id = $request->reason_id;

                // If reason is 'other', also save the text
                if ($request->reason_id == 5) {
                    $deal->dispute_reason = $request->other_reason ?? '';
                } else {
                    $deal->dispute_reason = ''; // or keep existing value if you want
                }
            }
            $validation = Validator::make($request->all(), $rules, __('validation.custom'));

            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            //get seller
            if ($deal->from_customer_id == $customerId) {
                $sellerId = $deal->to_customer_id;
            } else {
                $sellerId = $deal->from_customer_id;
            }
            //save
            $deal->deal_status = ON_DISPUTE;
            $deal->save();
            if ($request->hasFile('documents')) {
                //$filesName = uploadmultifiles('documents', 'uploads/dispute_docs/');
                $filesName = uploadmultifiles('documents', DISPUTED_DOCS_PATH . '/');
                foreach ($filesName as $file) {

                    $fileExtension = pathinfo($file, PATHINFO_EXTENSION);
                    $fileType = multimediaType($fileExtension);
                    $videoPoster = null;

                    if (in_array($fileExtension, videosExtensions())) {
                        $videoPoster = generateVideoThumbnail($file, DISPUTED_DOCS_PATH);
                    }

                    disputDdealDocumentModel::create([
                        'deal_id' => $deal->id,
                        'customer_id' =>  $customerId,
                        'file_path' => $file,
                        'file_extension' => $fileExtension,
                        'file_type' => $fileType,
                        'video_poster' => $videoPoster
                    ]);
                }
            }
            //delete docs
            if ($request->has('deleted_doc_id')) {
                $deletedDocs = disputDdealDocumentModel::where('deal_id', $deal->id)
                    ->whereIn('id', $request->deleted_doc_id)
                    ->where('customer_id', $customerId) // optional, security
                    ->get();
                foreach ($deletedDocs as $doc) {
                    //delete doc
                    $file = public_path(DISPUTED_DOCS_PATH . '/' . $doc->file_path);
                    if (file_exists($file)) {
                        unlink($file);
                    }
                    $doc->delete();
                }
            }
            //notify to seller that buyer has filed a dispute
            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            $nType = 'deal_on_dispute';
            $nTitle = "Buyer edited Disputed deal";
            $nTitleEs = 'Solicitud de trato';
            $nMesage = "Disputed deal has been edited by buyer deal #" . $deal->deal_number;
            $nMesageEs = "El comprador ha editado el trato en disputa n.º " . $deal->deal_number;
            saveNotification($sellerId, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $customerId, $nDealnumber, $dealId, 0);

            //add notification for admin
            saveNotificationAdmin("1", $deal->id, $nTitle, $nMesage, $nMesageEs, 0, $nType);


            DB::commit();
            Log::info('API hit from app dev: ' . json_encode($request->all()));
            $data = [
                'deal_id' => $deal->id,
            ];
            $message = __('message.disputed_deal_edited');
            // $message_description = __("Your dispute for Deal #{$deal->deal_number} has been successfully submitted. Our team will review your evidence and get back to you shortly.");
            return apiResponse(200, $message, ['data' => $data]);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }


    public function deleteDisputeDocs(Request $request)
    {
        try {
            DB::beginTransaction();
            $validation = Validator::make($request->all(), [
                'doc_id' => 'required|exists:dispute_deal_documents,id'
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $customerId = getLoggedInCustomerId();

            $doc = disputDdealDocumentModel::where('id', $request->doc_id)->first();
            if (!$doc) {
                $message = __('message.document_not_found');
                return apiResponse(400, $message);
            }

            $deal = DealModel::where('id', $doc->deal_id)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            if ($deal->isSecondPartyCustomerDeleted($customerId)) {
                $message = __('message.seller_deleted');
                return apiResponse(400, $message);
            }
            //deal edit dispute check
            if (!in_array($deal->deal_status, [ON_DISPUTE])) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            //$customerColumn = getCustomerColumn($deal, $customerId);

            /*if (!$customerColumn) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }*/
            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $seller = $customerDetails[SELLER];
            $buyer = $customerDetails[BUYER];
            if ($seller->id !== $customerId) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }

            //delete doc
            $file = public_path(DISPUTED_DOCS_PATH . '/' . $doc->file_path);
            if (file_exists($file)) {
                unlink($file);
            }
            $doc->delete();
            //notification send to second party
            $to_notification = null;
            $from_notification = null;
            $notification_message_en = null;

            if ($customerId == $seller->id) {
                $to_notification = $buyer->id;
                $from_notification = $seller->id;
                $notification_message_en = "Dispute document deleted by Seller for the deal #{$deal->deal_number}";
                $notification_message_es = "Documento de disputa eliminado por el Vendedor para el trato #{$deal->deal_number}";
                $nDealnumber = $deal->deal_number;
                $dealId = $deal->id;
                $nType = 'disputed_deal';
                $nTitle = "Document Deleted";
                $nTitleEs = 'Solicitud de trato';
                saveNotification($to_notification, $nType, $nTitle, $nTitleEs, $notification_message_en, $notification_message_es, $from_notification, $nDealnumber, $dealId, 0);
                //notification for admin
                saveNotificationAdmin("1", $deal->id, $nTitle, $notification_message_en, $notification_message_es, 0, $nType);
            }
            DB::commit();

            $message = __('message.disputed_document_deleted');
            // $message_description = __("Your dispute for Deal #{$deal->deal_number} has been successfully submitted. Our team will review your evidence and get back to you shortly.");
            return apiResponse(200, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }

    //disputeDocsSeller()
    public function disputeDocsSeller(Request $request)
    {
        try {
            DB::beginTransaction();
            $customerId = getLoggedInCustomerId();
            $deal = DealModel::where('id', $request->deal_id)->first();


            //get column for search
            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            if ($deal->isSecondPartyCustomerDeleted($customerId)) {
                $message = __('message.buyer_deleted');
                return apiResponse(400, $message);
            }
            //deal edit dispute check
            if (!in_array($deal->deal_status, [ON_DISPUTE])) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $seller = $customerDetails[SELLER];
            $buyer = $customerDetails[BUYER];



            if ($seller->id != $customerId) {

                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }


            if ($deal->is_paid == 0) {
                //dd("here2");
                $message =  __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            $allowedExtensions = implode(',', array_merge(imageExtensions(), docExtensions()));
            $rules = [
                'deal_id' => 'required|exists:deals,id',
                'documents' => 'required|array|max:5',
                'documents.*' => "file|mimes:$allowedExtensions|max:5120",
                // 'documents.*' => "file|mimes:$allowedExtensions|max:5120",
            ];


            $validation = Validator::make($request->all(), $rules, __('validation.custom'));


            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $is_update = false;
            //check that seller updated or submitted the dispute docs
            $updated_docs_seller_exists = disputDdealDocumentModel::where('customer_id', $customerId)
                ->where('deal_id', $deal->id)
                ->exists();

            if ($updated_docs_seller_exists) {
                $is_update = true;
            }
            //upload docs

            $filesName = uploadmultifiles('documents', DISPUTED_DOCS_PATH . '/');
            foreach ($filesName as $file) {
                $fileExtension = pathinfo($file, PATHINFO_EXTENSION);
                $fileType = multimediaType($fileExtension);


                // if (in_array($fileExtension, videosExtensions())) {
                //     $videoPoster = generateVideoThumbnail($file, DISPUTED_DOCS_PATH);
                // }

                disputDdealDocumentModel::create([
                    'deal_id' => $deal->id,
                    'customer_id' =>  $customerId,
                    'file_path' => $file,
                    'file_extension' => $fileExtension,
                    'file_type' => $fileType,

                ]);
            }



            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            $nType = 'disputed_deal';
            $nTitle = $is_update ? "Document Update" : "Document Submitted";
            $nTitleEs = $is_update
                ? "Actualización de documentos"
                : "Envío de documentos";
            $nMesage = $is_update ? "Documents for disputed deal has been edited by Seller" : "Documents for disputed deal has been submitted by Seller";
            $nMesageEs = $is_update
                ? "El vendedor ha editado los documentos para el trato en disputa"
                : "El vendedor ha enviado los documentos para el trato en disputa";




            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            //notification for buyer
            saveNotification($buyer->id, $nType, $nTitle, $nTitleEs, $nMesage, $nMesageEs, $seller->id, $nDealnumber, $dealId, 0);
            //notification for admin
            saveNotificationAdmin("1", $deal->id, $nTitle, $nMesage, $nMesageEs, 0, $nType);
            DB::commit();

            $message = __('message.documents_submitted');
            $message_description = __('message.dispute_documents_submitted');
            return apiResponse(200, $message, [], false, $message_description);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //customer side dispute conversation 
    public function DisputeConversationSend(Request $request)
    {

        try {
            DB::beginTransaction();
            $validation = disputeDealConversationModel::disputeConversationValidation($request->all());
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            $senderId = getLoggedInCustomerId();


            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }

            /*$customerColumn = getCustomerColumn($deal,  $senderId);

            if (!$customerColumn) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }*/
            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $buyer = $customerDetails[BUYER];
            $seller = $customerDetails[SELLER];



            if (!in_array($senderId, [$buyer->id, $seller->id])) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            //dispute conversation check
            if (!in_array($deal->deal_status, [ON_DISPUTE, DISPUTE_UNDER_REVIEW])) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            disputeDealConversationModel::create([
                'deal_id' => (int)$request->deal_id,
                'sender_id' =>  $senderId,
                'sender_type' => 'customer',
                'message' => $request->message,
                'receiver_type' => 'admin',
            ]);

            $senderInfo = CustomerModel::select('full_name')->where('id', $senderId)->first();

            //notification for send conversation to admin
            $nDealnumber = $deal->deal_number;

            $nType = 'reply_by_customer';
            $nTitle = "Dispute Update";
            $nMesage = "{$senderInfo->full_name} has send message regarding the dispute for Deal #{$nDealnumber}.";
            $nMesageEs = "{$senderInfo->full_name} ha enviado un mensaje sobre la disputa del trato #{$nDealnumber}.";

            $admins = AdminModel::pluck('id');
            foreach ($admins as $adminId) {
                saveNotificationAdmin($adminId, $deal->id, $nTitle, $nMesage, $nMesageEs, 0, $nType);
            }


            DB::commit();
            $message = __('message.send');
            return apiResponse(200, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //get dispute conversation
    public function getDisputeConversation(Request $request)
    {
        try {
            DB::beginTransaction();

            $validation = Validator::make($request->all(), [
                'deal_id' => 'required|exists:deals,id'
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $customerId = getLoggedInCustomerId();


            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }

            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $buyer = $customerDetails[BUYER];
            $seller = $customerDetails[SELLER];



            if (!in_array($customerId, [$buyer->id, $seller->id])) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            $messages = disputeDealConversationModel::where('deal_id', $request->deal_id)
                ->where(function ($query) use ($customerId) {
                    $query->where(function ($q) use ($customerId) {
                        // Customer sent to admin
                        $q->where('sender_id', $customerId)
                            ->where('sender_type', 'customer')
                            ->where('receiver_type', 'admin');
                    })->orWhere(function ($q) use ($customerId) {
                        // Admin sent to customer
                        $q->where('sender_type', 'admin')
                            ->where('receiver_id', $customerId)
                            ->where('receiver_type', 'customer');
                    });
                })
                ->orderBy('created_at', 'asc')
                ->get()
                ->map(function ($msg) {
                    return [
                        'id'          => $msg->id,
                        'deal_id'     => $msg->deal_id,
                        'sender_id'   => $msg->sender_id,
                        'sender_type' => $msg->sender_type,
                        'receiver_id' => $msg->receiver_id,
                        'receiver_type' => $msg->receiver_type,
                        'message'     => $msg->message,
                        'created_at' => $msg->created_at ? $msg->created_at->format('d M, Y g:i A') : null
                    ];
                });

            DB::commit();
            return apiResponse(200, __('message.messages_retrieved_successfully'), ['messages' => $messages]);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //payment process
    public function paymentProcess(Request $request)
    {
        try {
            DB::beginTransaction();

            $validation = Validator::make($request->all(), [
                'deal_id' => 'required|exists:deals,id',
                'type' => 'required|in:amount_detail,account_info',
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            $type = $request->type;
            $dealId = $deal->id;
            if ($type == 'amount_detail') {
                $dealAmount = $deal->deal_amount;
                $totalDealAmount = $deal->total_deal_Amount;
                $data = [
                    'deal_amount' => $dealAmount,
                    'total_deal_amount' => $totalDealAmount,
                ];
            }
            if ($type == 'account_info') {
                //fetch bank details
                $bankDetails = (new settingModel())->getSetting(['bank', 'card_number']);
                //dd($bank);
                $bank = $bankDetails->bank ?? '';
                $cardNumber = $bankDetails->card_number ?? '';
                //dd($bank, $cardNumber);
                $data = [
                    'bank' => $bank,
                    'card_number' => formatCardNumber($cardNumber),
                ];
            }
            DB::commit();
            return apiResponse(200, __('message.data_retrieved_successfully'), $data);
            //$status, $message, $result = [], $errors = false, $messageDescription = ''

        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    public function submitPaymentProof(Request $request)
    {

        try {
            DB::beginTransaction();
            $customerId = getLoggedInCustomerId();
            //insert payment proof
            $validation = Validator::make($request->all(), [
                'deal_id' => 'required|exists:deals,id',
                'payment_proof' =>  'required|image|mimes:jpg,jpeg,webp,png,gif|max:5120',
                // 'payment_proof' =>  'required|image|mimes:jpg,jpeg,webp,png,gif|max:2048',
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            if ($deal->isSecondPartyCustomerDeleted($customerId)) {
                $message = __('message.seller_deleted');
                return apiResponse(400, $message);
            }
            //checks on payment submit
            if ($deal->is_paid == 1 || $deal->is_hold == 1 || $deal->deal_status !== ACCEPTED) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            //checks on payment accept then no upload
            if ($deal->payment_status == 'accepted' && $deal->payment_proof && file_exists(public_path(PAYMENT_PROOF_PATH . '/' . $deal->payment_proof))) {
                $message = __('message.payment_proof_accepted');
                return apiResponse(400, $message);
            }
            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $buyer = $customerDetails[BUYER];
            // $seller = $customerDetails[SELLER];



            if ($buyer->id != $customerId) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }

            $is_update = !empty($deal->payment_proof);
            $imageName = uploadFile('payment_proof', PAYMENT_PROOF_PATH, $deal->payment_proof);
            //dd($imageName);
            $deal->payment_proof = $imageName;
            $deal->payment_status = PENDING;
            $deal->save();
            //add notification for admin
            $nTitle = "payment proof";
            $nType = "payment_proof";
            $nMesage = $is_update ? "Payment proof has been updated for deal #" . $deal->deal_number :  "You got the payment proof for deal #" . $deal->deal_number;
            $nMesageEs =  $is_update ? "El comprobante de pago ha sido actualizado para el trato #" . $deal->deal_number : "Has recibido el comprobante de pago del trato #" . $deal->deal_number;
            saveNotificationAdmin("1", $deal->id, $nTitle, $nMesage, $nMesageEs, 0, $nType);

            DB::commit();
            $message = $is_update  ? __('message.payment_proof_updated') : __('message.payment_proof_submitted');
            return apiResponse(200, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    public function updatePaymentProof(Request $request)
    {
        try {
            DB::beginTransaction();
            //insert payment proof
            $validation = Validator::make($request->all(), [
                'deal_id' => 'required|exists:deals,id',
                'payment_proof' =>  'required|image|mimes:jpg,jpeg,webp,png,gif|max:5120',
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {
                $message = "Deal not found";
                return apiResponse(400, $message);
            }

            $imageName = uploadFile('payment_proof', PAYMENT_PROOF_PATH);
            //dd($imageName);
            $deal->payment_proof = $imageName;
            $deal->save();
            //add notification for admin
            $nTitle = "payment proof";
            $nType = "payment_proof";
            $nMesage = "You got the payment proof for deal #" . $deal->deal_number;
            $nMesageEs = "Has recibido el comprobante de pago del trato #" . $deal->deal_number;
            saveNotificationAdmin("1", $deal->id, $nTitle, $nMesage, $nMesageEs, 0, $nType);

            DB::commit();
            return apiResponse(200, 'Payment proof submitted successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }

    public function downloadPdf(Request $request)
    {

        try {

            DB::beginTransaction();

            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {

                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }

            $loggedin_user_id = getLoggedInCustomerId();
            $isBuyer = false;
            $isSeller = false;

            if ($deal->deal_type === 'buy') {
                // Buyer created the deal
                $buyer_id = $deal->from_customer_id;
                $seller_id = $deal->to_customer_id;
            } else {
                // Seller created the deal
                $seller_id = $deal->from_customer_id;
                $buyer_id = $deal->to_customer_id;
            }

            $loggedin_user_id = getLoggedInCustomerId();

            if ($loggedin_user_id == $buyer_id) {
                $isBuyer = true;
                $secondPartyId = $seller_id;
                $customerType = BUYER;
            } else {
                $isSeller = true;
                $secondPartyId = $buyer_id;
                $customerType = SELLER;
            }
            $secondParty = CustomerModel::withTrashed()->where('id', $secondPartyId)->first();


            $hideCommission = false;
            $dealType = $deal->deal_type;
            $secondPartyAddress = '';
            $customerTypeArray = getCustomerType($deal,  $loggedin_user_id);
            $customerType = $customerTypeArray[1];
            if ($customerType == BUYER) {
                $secondPartyAddress = '';
            } else if ($customerType == SELLER) {

                $secondPartyAddress = $secondParty->address;
                $countryName = DB::table('customers')
                    ->leftJoin('countries', 'customers.country_id', '=', 'countries.id')
                    ->where('customers.id', $secondPartyId)
                    ->select('countries.name as country_name')
                    ->value('country_name');

                $stateName = DB::table('customers')
                    ->leftJoin('states', 'customers.state_id', '=', 'states.id')
                    ->where('customers.id', $secondPartyId)
                    ->select('states.name as state_name')
                    ->value('state_name');

                $cityName = DB::table('customers')
                    ->leftJoin('cities', 'customers.city_id', '=', 'cities.id')
                    ->where('customers.id', $secondPartyId)
                    ->select('cities.name as city_name')
                    ->value('city_name');
            }
            //$dealType = dealTypeViceVersa($deal, $loggedin_user_id);


            $dealTypeLangWise = dealTypeViceVersa($deal, $loggedin_user_id) == 'Buy' ? __('message.buy') : __('message.sell');

            $customerTypeArray = getCustomerType($deal,  $loggedin_user_id);
            $customerType = $customerTypeArray[1];



            DB::commit();

            $deal_data = (object)[
                'deal_id' => $deal->id,
                'deal_title' => $deal->deal_title,
                'deal_number' => $deal->deal_number,

                'deal_amount' => $deal->deal_amount,
                'total_deal_amount' => $deal->total_deal_Amount,

                'customer_type' => $customerType,
                'commission' => $deal->transfer_fees_in_percent,
                'commission_amount' => $deal->admin_earnings,
                'deal_type' => $dealTypeLangWise,
                'deal_status' => __('statuses.' . $deal->deal_status),
                'from_customer_id' => $deal->from_customer_id,
                'is_paid' => $deal->is_paid,
                'deal_created_date' => date(DISPLAY_DATE, strtotime($deal->created_at)),
                'deal_created_time' => date('h:i A', strtotime($deal->created_at)),
                'second_party' => (object)[
                    'second_party_id' => $secondParty->id ?? null,
                    'second_party_name' => $secondParty->full_name ?? 'Unknown',
                    'second_party_mobile_number' => $secondParty->mobile_number ?? null,
                    'second_party_dial_code' => $secondParty->dial_code ?? null,
                    'second_party_dial_code_iso' => $secondParty->dial_code_iso ?? null,
                    'second_party_address' => $secondPartyAddress,
                    'second_party_country' => $countryName ?? null,
                    'second_party_state' => $stateName ?? null,
                    'cityName' => $cityName ?? null,


                ],

            ];


            $pdf = Pdf::loadView('admin.pdf.deal_detail', compact('deal_data'));
            $publicPath = env('PUBLIC_BASE_PATH', public_path());
            $path = $publicPath . '/' . 'uploads/pdf/';
            $fileName = $deal->deal_number . '_' . $loggedin_user_id . '_' . time() . '.pdf';
            $pdf->save($path . '/' . $fileName);
            return apiResponse(200, __('message.success'),  ['download_url' => url('uploads/pdf/' . $fileName)]);
        } catch (\Exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }


        // return $pdf->stream('deal_' . $deal->deal_number . '.pdf');
    }
    public function updateFcmToken(Request $request)
    {
        $request->validate([
            'fcm_token' => 'required|string',
        ]);

        $customer = auth()->user(); // from Sanctum token

        $customer->update([
            'fcm_token' => $request->fcm_token,
        ]);

        return apiResponse(200, 'FCM token updated');
    }
    public function clear()
    {
        Artisan::call('config:clear');
        Artisan::call('cache:clear');
        Artisan::call('view:clear');
        Artisan::call('route:clear');

        return response()->json(['message' => 'Caches cleared successfully.']);
    }
    public function updateLanguage(Request $request)
    {


        $validation = Validator::make($request->all(), [
            'language_code' => 'required|in:en,es',
        ]);
        if ($validation->fails()) {
            return apiResponse(400, $validation->errors()->first());
        }

        $customer = CustomerAuth::user();
        if (!$customer) {
            return response()->json(['error' => 'Unauthenticated or invalid token.'], 401);
        }

        $customer->language_code = $request->language_code;
        $customer->save();



        return apiResponse(200, __('message.language_updated'));
    }
    public function getCountryInfo(Request $request)
    {
        $countryCode = strtoupper($request->dial_code_iso);
        $countryInfo = db::table('countries')->where('iso2', $countryCode)->first();
        $data = [
            'country_name' =>  $countryInfo->name,
            'country_id' =>  $countryInfo->id,

        ];
        return apiResponse(200, __('message.data_retrieved_successfully'), $data);
    }
    //shipping proof submission
    public function submitShippingProof(Request $request)
    {
        try {
            DB::beginTransaction();
            $customerId = getLoggedInCustomerId();
            //insert payment proof
            $validation = Validator::make($request->all(), [
                'deal_id' => 'required|exists:deals,id',
                'shipping_proof' =>  'required|image|mimes:jpg,jpeg,webp,png,gif|max:5120',
                // 'shipping_proof' =>  'required|image|mimes:jpg,jpeg,webp,png,gif|max:2048',
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $deal = DealModel::where('id', $request->deal_id)->first();

            if (!$deal) {
                $message = __('message.deal_not_found');
                return apiResponse(400, $message);
            }
            if ($deal->isSecondPartyCustomerDeleted($customerId)) {
                $message = __('message.buyer_deleted');
                return apiResponse(400, $message);
            }
            //checks on shipping proof submit
            if ($deal->is_paid == 0 || $deal->deal_status !== ACCEPTED) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }
            //checks on shipping accept then no upload
            if ($deal->shipping_status == 'accepted' && $deal->shipping_proof && file_exists(public_path(SHIPPING_PROOF_PATH . '/' . $deal->shipping_proof))) {
                $message = __('message.shipping_proof_accepted');
                return apiResponse(400, $message);
            }
            // find buyer and seller
            $customerDetails = getBuyerSellerDetails($deal);
            if (!$customerDetails) {
                return apiResponse(400, __('message.invalid_deal_data'));
            }
            $buyer = $customerDetails[BUYER];
            $seller = $customerDetails[SELLER];



            if ($seller->id != $customerId) {
                $message = __('message.unauthorized_action');
                return apiResponse(400, $message);
            }

            $is_update = !empty($deal->shipping_proof);
            $imageName = uploadFile('shipping_proof', SHIPPING_PROOF_PATH, $deal->shipping_proof);
            //dd($imageName);
            $deal->shipping_proof = $imageName;
            $deal->shipping_status = PENDING;
            $deal->save();
            //add notification for admin
            $nTitle = "shipping proof";
            $nType = "shipping_proof";
            $nMesage = $is_update ? "Shipping proof has been updated for deal #" . $deal->deal_number :  "You got the shipping proof for deal #" . $deal->deal_number;
            $nMesageEs =  $is_update ? "La prueba de envío ha sido actualizada para el trato #" . $deal->deal_number : "Has recibido la prueba de envío para el trato #" . $deal->deal_number;
            saveNotificationAdmin("1", $deal->id, $nTitle, $nMesage, $nMesageEs, 0, $nType);

            //notify to buyer too
            $nDealnumber = $deal->deal_number;
            $dealId = $deal->id;
            $nTypeBs = 'shipping_proof';
            $nTitleBs = "shipping proof";
            $nTitleEsBs = 'Solicitud de trato';
            $nMesageBs = $is_update ? "Shipping proof has been updated by Seller for deal #" . $deal->deal_number :  "Shipping proof has been submitted by Seller for deal #" . $deal->deal_number;
            $nMesageEsBs = $is_update
                ? "El vendedor ha actualizado la prueba de envío para el trato n.º " . $deal->deal_number
                : "El vendedor ha enviado la prueba de envío para el trato n.º " . $deal->deal_number;
            //saveNotification($buyer->id, $nTypeBs, $nTitleBs, $nTitleEsBs, $nMesageBs, $nMesageEsBs, $seller->id, $nDealnumber, $dealId, 0);

            DB::commit();
            $message = $is_update  ? __('message.shipping_proof_updated') : __('message.shipping_proof_submitted');
            return apiResponse(200, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //submit deal images seller side at the time he will going to accept request
    public function uploadDealImages(Request $request)
    {
        $ext = imageExtensions();
        $mimes = implode(",", $ext);
        $validation = Validator::make($request->all(), [
            'deal_id'   => 'required|exists:deals,id',
            'images' => 'nullable|array|min:1|max:3',
            'images.*' => 'image|mimes:' . $mimes . '|max:5120',
        ]);

        if ($validation->fails()) {
            return apiResponse(400, $validation->errors()->first());
        }

        $deal = DealModel::where('id', $request->deal_id)->first();
        $customerId = auth()->user()->id;

        //check if to customer is seller
        $tocustomerType =  ($deal->deal_type == BUY)  ? SELLER  : BUYER;
        if ($tocustomerType != SELLER) {
            return apiResponse(400, __('message.seller_can_upload_deal_images'));
        }
        $uploadedImage =   uploadmultifiles('images', DEAL_IMAGES);
        foreach ($uploadedImage as $imageName) {

            DealImages::create([
                'deal_id' => $deal->id,
                'customer_id' => $customerId,
                'image' => $imageName
            ]);
        }
        return apiResponse(200, __('message.deal_images_upload_success'));
    }
}
